package com.ngrob.android.bluemoon.features.calendar.screens

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.ngrob.android.bluemoon.MenstruationCycleTracker
import com.ngrob.android.bluemoon.core.data.repository.BleedingRepository
import com.ngrob.android.bluemoon.core.data.repository.PillRepository
import com.ngrob.android.bluemoon.core.data.repository.SexRepository
import com.ngrob.android.bluemoon.features.calendar.components.CardEntryItem
import com.ngrob.android.bluemoon.features.calendar.components.TrackingTypes
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.time.Clock
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.YearMonth
import java.time.format.DateTimeFormatter
import javax.inject.Inject
import kotlin.time.Duration.Companion.seconds

data class TrackedItem(
    val id: Int,
    val cardEntryItem: CardEntryItem,
    val values: Map<String, List<String>>,
    val loggedAt: LocalDateTime
)

data class CalendarUiState(
    val selectedDay: LocalDate? = null,
    val bleedingDays: List<LocalDate> = emptyList(),
    val dayEntries: List<TrackedItem> = emptyList(),
    val bleedingDaysPredicted: List<LocalDate> = emptyList(),
    val fabVisible: Boolean = false
)

@HiltViewModel
class CalendarViewModel @Inject constructor(
    private val savedStateHandle: SavedStateHandle,
    private val sexRepository: SexRepository,
    private val bleedingRepository: BleedingRepository,
    private val pillRepository: PillRepository,
    private val clock: Clock
) : ViewModel() {

    private var menstruationCycleTracker: MenstruationCycleTracker = MenstruationCycleTracker()

    private var selectedDay = savedStateHandle.getStateFlow(
        key = CALENDAR_SELECTED_DAY,
        initialValue = LocalDate.now(clock)
    )
    private var currentMonth = savedStateHandle.getStateFlow(
        key = CALENDAR_CURRENT_MONTH,
        initialValue = YearMonth.now(clock)
    )

    @OptIn(ExperimentalCoroutinesApi::class)
    val uiState: StateFlow<CalendarUiState> =
        combine(selectedDay, currentMonth) { selectedDay, currentMonth ->
            combineCalenderUiState(selectedDay, currentMonth)
        }.flatMapLatest { it }.stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5.seconds.inWholeMilliseconds),
            initialValue = CalendarUiState(),
        )

    init {
        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                bleedingRepository.getAllBleedingDates().collect { bleedingDates ->
                    menstruationCycleTracker = MenstruationCycleTracker(bleedingDates)
                }
            }

        }
    }


    fun selectDay(newDay: LocalDate) {
        savedStateHandle[CALENDAR_SELECTED_DAY] = newDay
    }

    private fun combineCalenderUiState(selectedDay: LocalDate, currentMonth: YearMonth) = combine(
        bleedingRepository.getBleedingEntryOfDay(selectedDay),
        bleedingRepository.getAllBleedingsOfMonth(currentMonth),
        sexRepository.getSexWithContraceptivesOfDate(selectedDay),
        pillRepository.getPillOfDay(selectedDay),
    ) { bleedingOfDay, bleedingsOfMonth, sex, pill ->

        val allEntries: MutableList<TrackedItem> = mutableListOf()

        val sexEntries = sex.map { entry ->
            TrackedItem(
                id = entry.sex.id!!,
                loggedAt = LocalDateTime.parse(
                    entry.sex.loggedAt, DateTimeFormatter.ISO_DATE_TIME
                ) ?: LocalDateTime.now(clock),
                values = mapOf("type" to listOf(entry.sex.type.toString()),
                    "contraceptives" to entry.contraceptives.map {
                        it.contraceptive.toString()
                    }),
                cardEntryItem = CardEntryItem.Sex
            )
        }
        allEntries.addAll(sexEntries)

        if (pill != null) {
            val pillEntry = TrackedItem(
                id = pill.id!!,
                loggedAt = LocalDateTime.parse(
                    pill.loggedAt, DateTimeFormatter.ISO_DATE_TIME
                ) ?: LocalDateTime.now(clock),
                values = mapOf("type" to listOf("Birth control")),
                cardEntryItem = CardEntryItem.Medication
            )
            allEntries.add(pillEntry)
        }

        if (bleedingOfDay != null) {
            val bleedingStrength =
                if (bleedingOfDay.strength != null) bleedingOfDay.strength.toString() else "Not logged"
            val bleedingEntry = TrackedItem(
                id = bleedingOfDay.id!!,
                loggedAt = LocalDateTime.parse(
                    bleedingOfDay.loggedAt, DateTimeFormatter.ISO_DATE_TIME
                ) ?: LocalDateTime.now(clock), values = mapOf(
                    "strength" to listOf(bleedingStrength)
                ), cardEntryItem = CardEntryItem.Bleeding
            )
            allEntries.add(bleedingEntry)
        }
        CalendarUiState(
            selectedDay = selectedDay,
            bleedingDaysPredicted = menstruationCycleTracker.predictPeriodDaysOfMonth(currentMonth),
            bleedingDays = bleedingsOfMonth,
            dayEntries = allEntries.sortedBy { it.loggedAt }.toList(),
            fabVisible = selectedDay <= LocalDate.now(clock)
        )


    }

    fun onDismissed(value: TrackedItem) {
        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                when (value.cardEntryItem.type) {
                    TrackingTypes.BLEEDING -> bleedingRepository.deleteBleedingById(value.id)
                    TrackingTypes.SEX -> sexRepository.deleteSexById(value.id)
                    TrackingTypes.MOOD -> TODO()
                    TrackingTypes.MEDICATION -> pillRepository.deletePillById(value.id)
                }
            }
        }

    }

    fun onMonthChanged(newMonth: YearMonth) {
        savedStateHandle[CALENDAR_CURRENT_MONTH] = newMonth
    }
}


private const val CALENDAR_SELECTED_DAY = "calendarSelectedDay"
private const val CALENDAR_CURRENT_MONTH = "calendarCurrentMonth"