package com.ngrob.android.bluemoon.features.calendar.screens

import android.annotation.SuppressLint
import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.slideOut
import androidx.compose.animation.togetherWith
import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.Orientation
import androidx.compose.foundation.gestures.rememberScrollableState
import androidx.compose.foundation.gestures.scrollable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Card
import androidx.compose.material3.Icon
import androidx.compose.material3.Scaffold
import androidx.compose.material.Text
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material3.FabPosition
import androidx.compose.material3.FloatingActionButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableFloatStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.runtime.snapshotFlow
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.navigation.NavHostController
import androidx.navigation.compose.rememberNavController
import com.ngrob.android.bluemoon.core.ui.theme.Black
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import com.ngrob.android.bluemoon.core.ui.theme.White
import com.ngrob.android.bluemoon.features.calendar.components.Calendar
import com.ngrob.android.bluemoon.features.calendar.components.EntriesList
import com.ngrob.android.bluemoon.features.calendar.components.WeekCalendar
import io.github.boguszpawlowski.composecalendar.CalendarState
import io.github.boguszpawlowski.composecalendar.WeekCalendarState
import io.github.boguszpawlowski.composecalendar.header.MonthState
import io.github.boguszpawlowski.composecalendar.header.WeekState
import io.github.boguszpawlowski.composecalendar.rememberSelectableCalendarState
import io.github.boguszpawlowski.composecalendar.rememberSelectableWeekCalendarState
import io.github.boguszpawlowski.composecalendar.selection.DynamicSelectionState
import io.github.boguszpawlowski.composecalendar.selection.SelectionMode
import io.github.boguszpawlowski.composecalendar.week.Week
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import java.time.LocalDate
import java.time.YearMonth
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import java.util.stream.Collectors
import java.util.stream.LongStream


@SuppressLint("UnusedMaterial3ScaffoldPaddingParameter")
@Composable
fun CalendarScreen(
    modifier: Modifier = Modifier,
    nav: NavHostController,
    viewModel: CalendarViewModel = hiltViewModel(),
    onSelectedDay: (LocalDate) -> Unit
) {
    val uiState by viewModel.uiState.collectAsState()
    val weekState = rememberSaveable(saver = WeekState.Saver()) {
        WeekState(initialWeek = Week.now())
    }
    val monthState = rememberSaveable(saver = MonthState.Saver()) {
        MonthState(initialMonth = weekState.currentWeek.yearMonth)
    }

    LaunchedEffect(monthState) {
        snapshotFlow { monthState.currentMonth }.onEach { viewModel.onMonthChanged(it) }
            .launchIn(this)
    }

    val calendarState: CalendarState<DynamicSelectionState> = rememberSelectableCalendarState(
        confirmSelectionChange = {
            if (it.isEmpty()) {
                viewModel.selectDay(LocalDate.now())
                onSelectedDay(LocalDate.now())
            } else {
                viewModel.selectDay(it[0])
                onSelectedDay(it[0])
            }
            true
        },
        initialSelectionMode = SelectionMode.Single,
        initialSelection = if (uiState.selectedDay != null) listOf(uiState.selectedDay!!) else emptyList(),
        monthState = monthState
    )

    val weekCalendarState: WeekCalendarState<DynamicSelectionState> =
        rememberSelectableWeekCalendarState(
            confirmSelectionChange = {
                if (it.isEmpty()) {
                    viewModel.selectDay(LocalDate.now())
                    onSelectedDay(LocalDate.now())
                } else {
                    viewModel.selectDay(it[0])
                    onSelectedDay(it[0])
                }
                true
            },
            initialSelectionMode = SelectionMode.Single,
            initialSelection = if (uiState.selectedDay != null) listOf(uiState.selectedDay!!) else emptyList(),
            weekState = weekState,
            selectionState = calendarState.selectionState,

            )
    LaunchedEffect(calendarState.selectionState.selection) {
        val selection = calendarState.selectionState.selection
        if (selection.isEmpty()) {
            return@LaunchedEffect
        }
        monthState.currentMonth = YearMonth.from(selection[0])
        val weekStart = selection[0].minusDays(selection[0].dayOfWeek.value.toLong() - 1)

        weekState.currentWeek = Week(
            LongStream.iterate(0) { i -> i + 1 }.limit(7).mapToObj { i ->
            weekStart.plusDays(
                i
            )
        }.collect(Collectors.toList())

        )
    }

    var scrollOffset by remember { mutableFloatStateOf(0f) }

    var monthCalendarVisible by remember { mutableStateOf(true) }

    LaunchedEffect(scrollOffset) {
        monthCalendarVisible = scrollOffset >= 0
    }

    LaunchedEffect(monthCalendarVisible) {
        if (monthCalendarVisible) {
            snapshotFlow { monthState.currentMonth }.onEach { viewModel.onMonthChanged(it) }
                .launchIn(this)
            return@LaunchedEffect
        }
        snapshotFlow { weekState.currentWeek.yearMonth }.onEach { viewModel.onMonthChanged(it) }
            .launchIn(this)
    }
    val bleedingDays = uiState.bleedingDays
    val dayEntries = uiState.dayEntries
    val dateFormatted = uiState.selectedDay?.format(
        DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
    )
    Scaffold(
        modifier.fillMaxSize(),
        floatingActionButtonPosition = FabPosition.End,
        floatingActionButton = {
            if (uiState.fabVisible) {
                FloatingActionButton(
                    modifier = Modifier.size(65.dp),
                    containerColor = Black,
                    contentColor = Color.White,
                    onClick = { nav.navigate("add_symptom") },
                    shape = CircleShape,
                ) {
                    Icon(
                        imageVector = Icons.Filled.Add,
                        contentDescription = "Track a symptom",
                        Modifier.size(30.dp)
                    )
                }
            }
        }) {
        Column(
            Modifier
                .background(color = Gray900)
                .scrollable(
                    orientation = Orientation.Vertical,
                    state = rememberScrollableState { delta ->
                        scrollOffset += delta
                        delta
                    })
        ) {
            AnimatedContent(monthCalendarVisible, label = "Calendar", transitionSpec = {
                fadeIn(
                    animationSpec = tween(10)
                ) togetherWith slideOut(
                    animationSpec = tween(10),
                    targetOffset = { IntOffset(0, it.height) })
            }) { monthCalendarVisible ->

                if (monthCalendarVisible) {
                    Calendar(
                        Modifier.testTag("MonthCalendar"),
                        calendarState = calendarState,
                        bleedingDays = bleedingDays,
                        predictedBleedingDays = uiState.bleedingDaysPredicted
                    )
                } else {
                    WeekCalendar(
                        calendarState = weekCalendarState,
                        bleedingDays = bleedingDays,
                        predictedBleedingDays = uiState.bleedingDaysPredicted
                    )
                }


            }
            Card(
                modifier = Modifier.fillMaxSize(),
                backgroundColor = White,
                elevation = 0.dp,
                shape = RoundedCornerShape(30.dp, 30.dp, 0.dp, 0.dp)
            ) {
                Column {
                    Text(
                        modifier = Modifier.padding(vertical = 8.dp, horizontal = 20.dp),
                        text = dateFormatted ?: "No day selected",
                        fontWeight = FontWeight.W500,
                        fontSize = 18.sp,
                        color = Gray900
                    )
                    EntriesList(items = dayEntries, onDismissed = {
                        viewModel.onDismissed(it)
                    })
                }

            }
        }
    }
}

@Preview(showBackground = true)
@Composable
fun CalendarViewPreview() {
    BluemoonTheme {
        CalendarScreen(modifier = Modifier, rememberNavController(), onSelectedDay = {})
    }
}