package com.ngrob.android.bluemoon.features.calendar.components

import android.annotation.SuppressLint
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.ngrob.android.bluemoon.core.ui.theme.BleedingRed
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.Gray600
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import io.github.boguszpawlowski.composecalendar.SelectableWeekCalendar
import io.github.boguszpawlowski.composecalendar.WeekCalendarState
import io.github.boguszpawlowski.composecalendar.day.DayState
import io.github.boguszpawlowski.composecalendar.header.WeekState
import io.github.boguszpawlowski.composecalendar.rememberSelectableWeekCalendarState
import io.github.boguszpawlowski.composecalendar.selection.DynamicSelectionState
import java.time.LocalDate
import java.time.format.TextStyle
import java.util.Locale


@Composable
fun WeekHeader(weekState: WeekState) {
    Row {
        Text(
            weekState.currentWeek.yearMonth.month.getDisplayName(TextStyle.FULL, Locale.ENGLISH),
            style = MaterialTheme.typography.h2,
            modifier = Modifier.padding(15.dp, 15.dp, 15.dp, 0.dp)
        )
        Text(
            weekState.currentWeek.yearMonth.year.toString(),
            style = MaterialTheme.typography.h2,
            modifier = Modifier.padding(0.dp, 15.dp, 15.dp, 15.dp)
        )

    }
}
@SuppressLint("SuspiciousIndentation")
@Composable
fun CustomWeekDay(
    state: DayState<DynamicSelectionState>,
    isBleedingDay: Boolean,
    isPredictedBleedingDate: Boolean,
    ) {
    val selectionState = state.selectionState
    val date = state.date
    Box(
        modifier = Modifier
            .fillMaxWidth()
            .height(100.dp)
            .size(60.dp)
            .padding(2.dp)
            .clip(RoundedCornerShape(10.dp))
            .background(if (selectionState.isDateSelected(date)) Gray600 else Color.Transparent)
            .clickable { selectionState.onDateSelected(date) },
        contentAlignment = Alignment.Center,

        ) {
        Column(
            modifier = Modifier
                .fillMaxHeight()
                .fillMaxWidth()
                .padding(2.dp, 4.dp),
            verticalArrangement = Arrangement.SpaceBetween,
            horizontalAlignment = Alignment.CenterHorizontally
        ) {
            Text(
                text = date.dayOfWeek.getDisplayName(TextStyle.SHORT, Locale.ENGLISH),
                color = Color.White,
            )
            Column(modifier = Modifier.padding(0.dp, 2.dp)) {
                Text(
                    text = date.dayOfMonth.toString(),
                    color = Color.White,
                    fontWeight = if (state.isCurrentDay) FontWeight.Bold else FontWeight.Normal,
                    modifier = Modifier.padding(0.dp, 8.dp)
                )

                Box(
                    modifier = Modifier
                        .align(Alignment.CenterHorizontally)
                        .size(8.dp)
                        .clip(CircleShape).alpha(if (isPredictedBleedingDate) 0.5f else 1f)
                        .background(if (isBleedingDay || isPredictedBleedingDate) BleedingRed else Color.Transparent)
                )
            }
        }
    }
}

@Composable
fun WeekCalendar(
    modifier: Modifier = Modifier,
    calendarState: WeekCalendarState<DynamicSelectionState>,
    bleedingDays: List<LocalDate>,
    predictedBleedingDays: List<LocalDate>

) {

    SelectableWeekCalendar(
        modifier
            .background(color = Gray900)
            .padding(0.dp, 0.dp, 0.dp, 10.dp),
        dayContent = { dayState ->
            val isBleedingDay = dayState.date in bleedingDays
            val isPredictedBleedingDay = dayState.date in predictedBleedingDays
            CustomWeekDay(state = dayState, isBleedingDay, isPredictedBleedingDay)
        },
        horizontalSwipeEnabled = true,
        weekHeader = { WeekHeader(weekState = it) },
        daysOfWeekHeader = { },
        calendarState = calendarState,
    )
}

@Preview(showBackground = true)
@Composable
fun WeekCalendarPreview() {
    BluemoonTheme {
        WeekCalendar(
            calendarState = rememberSelectableWeekCalendarState(),
            bleedingDays = listOf(LocalDate.now()),
            predictedBleedingDays = listOf(LocalDate.now().plusDays(1L))
        )
    }
}