package com.ngrob.android.bluemoon.features.calendar.components

import androidx.annotation.StringRes
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.Card
import androidx.compose.material.DismissDirection
import androidx.compose.material.DismissValue
import androidx.compose.material.ExperimentalMaterialApi
import androidx.compose.material.Icon
import androidx.compose.material.SwipeToDismiss
import androidx.compose.material.Text
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.rememberDismissState
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.ngrob.android.bluemoon.R
import com.ngrob.android.bluemoon.core.ui.theme.BleedingRed
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.MedicationBlue
import com.ngrob.android.bluemoon.core.ui.theme.MoodGreen
import com.ngrob.android.bluemoon.core.ui.theme.SexYellow
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle

enum class TrackingTypes(val displayString: String) {
    BLEEDING("Bleeding"),
    SEX("Sex"),
    MOOD("Mood"),
    MEDICATION("Medication"),
}

sealed class CardEntryItem(
    val type: TrackingTypes,
    @StringRes val displayEmoji: Int,
    val color: Color
) {
    object Bleeding : CardEntryItem(TrackingTypes.BLEEDING, R.string.blood_emoji, BleedingRed)
    object Sex : CardEntryItem(TrackingTypes.SEX, R.string.sex_emoji, SexYellow)
    object Mood : CardEntryItem(TrackingTypes.MOOD, R.string.smiley_emoji, MoodGreen)
    object Medication : CardEntryItem(TrackingTypes.MEDICATION, R.string.pill_emoji, MedicationBlue)
}

@OptIn(ExperimentalMaterialApi::class)
@Composable
fun EntryCard(
    modifier: Modifier = Modifier,
    entryItem: CardEntryItem,
    values: Map<String, List<String>>,
    loggedAt: LocalDateTime,
    onDismissed: () -> Unit,
) {

    val loggedAtFormatted = loggedAt.format(DateTimeFormatter.ofLocalizedDateTime(FormatStyle.SHORT))
    val dismissState = rememberDismissState(confirmStateChange = {
        if (it == DismissValue.DismissedToStart || it == DismissValue.DismissedToEnd){
            onDismissed()
        }
        true
    })
    SwipeToDismiss(state = dismissState , background = {
        val color = when(dismissState.dismissDirection){
            DismissDirection.EndToStart -> Color.Red
            DismissDirection.StartToEnd -> Color.Red
            null -> Color.Transparent
        }
        Box(modifier= Modifier.clip(RoundedCornerShape(10.dp)).fillMaxSize()
            .background(color)){
            Icon(imageVector = Icons.Default.Delete, contentDescription = "Delete", tint= Color.White, modifier = Modifier.align(
                Alignment.CenterEnd).padding(horizontal = 15.dp))
            Icon(imageVector = Icons.Default.Delete, contentDescription = "Delete", tint= Color.White, modifier = Modifier.align(
                Alignment.CenterStart).padding(horizontal = 15.dp))
        }
    }, modifier= Modifier.padding(10.dp) , dismissContent =  {
        Card(modifier = modifier
            , shape = RoundedCornerShape(10.dp)) {
            Row(
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.SpaceBetween,
                modifier = Modifier
                    .fillMaxWidth()
                    .background(entryItem.color)
                    .padding(10.dp)

            ) {
                Column {
                    Text(text = entryItem.type.displayString, fontWeight = FontWeight.Bold)
                    values.map {
                        Row {
                            Text(text = "${it.key}: ", fontWeight = FontWeight.W500)
                            it.value.map {
                                Text(text = "${it}, ")
                            }

                        }
                    }
                    Text(text = "logged @ ${loggedAtFormatted}")
                }
                Text(stringResource(entryItem.displayEmoji), fontSize = 50.sp)
            }
        }

    })
}

@Preview(showBackground = false)
@Composable
fun EntryCardPreview() {
    BluemoonTheme {
        EntryCard(
            entryItem = CardEntryItem.Sex,
            values = mapOf("type" to listOf("partner"), "contraceptives" to listOf("pill", "condom")),
            loggedAt = LocalDateTime.now(),
            onDismissed = {}
        )
    }
}