package com.ngrob.android.bluemoon.features.calendar.components

import android.annotation.SuppressLint
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.ngrob.android.bluemoon.core.ui.theme.BleedingRed
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.Gray600
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import io.github.boguszpawlowski.composecalendar.CalendarState
import io.github.boguszpawlowski.composecalendar.SelectableCalendar
import io.github.boguszpawlowski.composecalendar.day.DayState
import io.github.boguszpawlowski.composecalendar.header.MonthState
import io.github.boguszpawlowski.composecalendar.rememberSelectableCalendarState
import io.github.boguszpawlowski.composecalendar.selection.DynamicSelectionState
import java.time.DayOfWeek
import java.time.LocalDate
import java.time.format.TextStyle
import java.util.Locale

@Composable
fun MonthHeader(monthState: MonthState) {
    Row {
        Text(
            monthState.currentMonth.month.getDisplayName(TextStyle.FULL, Locale.ENGLISH),
            style = MaterialTheme.typography.h2,
            modifier = Modifier.padding(15.dp, 15.dp, 15.dp, 0.dp)
        )
        Text(
            monthState.currentMonth.year.toString(),
            style = MaterialTheme.typography.h2,
            modifier = Modifier.padding(0.dp, 15.dp, 15.dp, 15.dp)
        )

    }
}

@Composable
fun CustomWeekHeader(daysOfWeek: List<DayOfWeek>) {
    Row {
        daysOfWeek.forEach { dayOfWeek ->
            Text(
                textAlign = TextAlign.Center,
                text = dayOfWeek.getDisplayName(TextStyle.SHORT, Locale.ROOT),
                style = MaterialTheme.typography.h3,
                modifier = Modifier
                    .weight(1f)
                    .wrapContentHeight()

            )
        }
    }

}

@SuppressLint("SuspiciousIndentation")
@Composable
fun CustomDay(
    modifier: Modifier = Modifier,
    state: DayState<DynamicSelectionState>,
    isBleedingDay: Boolean,
    isPredictedBleedingDate: Boolean,
) {
    val selectionState = state.selectionState
    val date = state.date
    Box(
        modifier = modifier
            .size(45.dp)
            .padding(5.dp)
            .clip(CircleShape)
            .background(if (selectionState.isDateSelected(date)) Gray600 else Color.Transparent)
            .clickable { selectionState.onDateSelected(date) }, contentAlignment = Alignment.Center

    ) {
        Column() {
            Text(
                text = date.dayOfMonth.toString(),
                color = Color.White,
                fontWeight = if (state.isCurrentDay) FontWeight.Bold else FontWeight.Normal
            )

            Box(
                modifier = Modifier
                    .align(Alignment.CenterHorizontally)
                    .clip(CircleShape)
                    .size(8.dp)
                    .alpha(if (isPredictedBleedingDate) 0.5f else 1f)
                    .background(if (isBleedingDay || isPredictedBleedingDate) BleedingRed else Color.Transparent)
            )
        }

    }

}

@Composable
fun Calendar(
    modifier: Modifier = Modifier,
    calendarState: CalendarState<DynamicSelectionState>,
    bleedingDays: List<LocalDate>,
    predictedBleedingDays: List<LocalDate>
) {

    SelectableCalendar(
        modifier.background(color = Gray900),
        dayContent = { dayState ->
            val isBleedingDay = dayState.date in bleedingDays
            val isPredictedBleedingDay = dayState.date in predictedBleedingDays
            CustomDay(
                modifier = Modifier
                    .align(Alignment.Center)
                    .testTag("calendarDayPredictedBleeding$isPredictedBleedingDay"),
                state = dayState,
                isBleedingDay,
                isPredictedBleedingDay
            )
        },
        daysOfWeekHeader = { CustomWeekHeader(daysOfWeek = it) },
        showAdjacentMonths = false,
        monthHeader = { MonthHeader(monthState = it) },
        calendarState = calendarState,
    )
}

@Preview(showBackground = true)
@Composable
fun CalendarPreview() {
    BluemoonTheme {
        Calendar(
            calendarState = rememberSelectableCalendarState(),
            bleedingDays = listOf(LocalDate.now()),
            predictedBleedingDays = listOf(LocalDate.now().plusDays(2))
        )
    }
}