package com.ngrob.android.bluemoon.core.reminder

import android.content.Context
import android.content.Intent
import com.ngrob.android.bluemoon.core.data.repository.UserDataRepository
import com.ngrob.android.bluemoon.core.notifications.NotificationType
import com.ngrob.android.bluemoon.core.util.getDateToSchedulePillReminder
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.time.Clock
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.LocalTime
import javax.inject.Inject


@AndroidEntryPoint
internal class ReminderRebootReceiver : HiltBroadcastReceiver() {

    @Inject
    lateinit var userDataRepository: UserDataRepository

    @Inject
    lateinit var reminderScheduler: ReminderScheduler

    @Inject
    lateinit var clock: Clock

    override fun onReceive(context: Context?, intent: Intent?) {
        super.onReceive(context, intent)
        if (intent?.action == "android.intent.action.BOOT_COMPLETED" || intent?.action == "android.app.action.SCHEDULE_EXACT_ALARM_PERMISSION_STATE_CHANGED") {
            context.let {
                CoroutineScope(Dispatchers.Default).launch {
                    rescheduleAllAlarms()
                }
            }
        }
    }

    private suspend fun rescheduleAllAlarms() {
        userDataRepository.userData.collect { userData ->
            if (userData.reminderSettings.remindForTrackingSymptomsDaily) {
                val dateTime = LocalDateTime.of(LocalDate.now(clock), LocalTime.of(18, 0))
                reminderScheduler.schedule(ReminderItem(dateTime, NotificationType.DAILY_REMINDER))
            }

            if (userData.reminderSettings.remindForTakingPill) {
                if (!userData.trackBirthControl) {
                    return@collect
                }
                val pillRecipe = userData.pillRecipe
                val scheduledTime = userData.reminderSettings.pillReminderTime
                val dateTime =
                    getDateToSchedulePillReminder(
                        pillRecipe,
                        currentDate = LocalDate.now(clock),
                        scheduledTime = scheduledTime
                    )
                reminderScheduler.schedule(ReminderItem(dateTime, NotificationType.PILL_REMINDER))
            }
        }
    }
}