package com.ngrob.android.bluemoon.core.reminder

import android.app.AlarmManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import com.ngrob.android.bluemoon.core.notifications.NotificationType
import dagger.hilt.android.qualifiers.ApplicationContext
import java.time.ZoneId
import javax.inject.Inject
import javax.inject.Singleton


@Singleton
class ReminderNotificationScheduler @Inject constructor(
    @ApplicationContext private val context: Context,
) : ReminderScheduler {
    private val alarmManager = context.getSystemService(AlarmManager::class.java)
    override fun schedule(reminderItem: ReminderItem) {
        val intent = Intent(context, ReminderNotificationReceiver::class.java).apply {
            putExtra("EXTRA_NOTIFICATION", reminderItem.notificationType)
        }
        val alarmTime =
            reminderItem.alarmTime.atZone(ZoneId.systemDefault()).toEpochSecond() * 1000L

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S && !alarmManager.canScheduleExactAlarms()) {
            alarmManager.set(
                AlarmManager.RTC_WAKEUP, alarmTime, PendingIntent.getBroadcast(
                    context,
                    reminderItem.notificationType.hashCode(),
                    intent,
                    PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
                )
            )
            return
        }

        alarmManager.setExactAndAllowWhileIdle(
            AlarmManager.RTC_WAKEUP, alarmTime, PendingIntent.getBroadcast(
                context,
                reminderItem.notificationType.hashCode(),
                intent,
                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
            )
        )
    }

    override fun cancel(notificationType: NotificationType) {
        val indent = Intent(context, ReminderNotificationReceiver::class.java)
        val pendingIntent = PendingIntent.getBroadcast(
            context,
            notificationType.hashCode(),
            indent,
            PendingIntent.FLAG_NO_CREATE or PendingIntent.FLAG_IMMUTABLE
        ) ?: return
        alarmManager.cancel(pendingIntent)
    }

    override fun cancelAll() {
        cancel(NotificationType.PILL_REMINDER)
        cancel(NotificationType.DAILY_REMINDER)
        cancel(NotificationType.UPCOMING_PERIOD_REMINDER)
    }
}