package com.ngrob.android.bluemoon.core.reminder

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.os.Build
import androidx.annotation.CallSuper
import com.ngrob.android.bluemoon.core.data.repository.UserDataRepository
import com.ngrob.android.bluemoon.core.notifications.NotificationType
import com.ngrob.android.bluemoon.core.notifications.SystemTrayNotifier
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import java.io.Serializable
import javax.inject.Inject


abstract class HiltBroadcastReceiver : BroadcastReceiver() {
    @CallSuper
    override fun onReceive(context: Context?, intent: Intent?) {
    }
}

@AndroidEntryPoint
internal class ReminderNotificationReceiver : HiltBroadcastReceiver() {
    @Inject
    lateinit var systemTrayNotifier: SystemTrayNotifier

    @Inject
    lateinit var userDataRepository: UserDataRepository

    @Inject
    lateinit var bluemoonReminderNotificationScheduler: BluemoonReminderNotificationScheduler
    override fun onReceive(context: Context?, intent: Intent?) {
        super.onReceive(context, intent)
        val notification =
            intent?.getSerializable("EXTRA_NOTIFICATION", NotificationType::class.java) ?: return
        systemTrayNotifier.postNotification(notification)
        context?.let {
            CoroutineScope(Dispatchers.Default).launch {
                userDataRepository.userData.collectLatest {
                    bluemoonReminderNotificationScheduler.rescheduleRepeatingReminder(
                        notification,
                        it
                    )

                }
            }
        }
    }


}

// https://stackoverflow.com/questions/72571804/getserializableextra-and-getparcelableextra-are-deprecated-what-is-the-alternat
fun <T : Serializable?> Intent.getSerializable(key: String, mClass: Class<T>): T {
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU)
        this.getSerializableExtra(key, mClass)!!
    else
        this.getSerializableExtra(key) as T
}


