package com.ngrob.android.bluemoon.core.reminder

import com.ngrob.android.bluemoon.MenstruationCycleTracker
import com.ngrob.android.bluemoon.core.model.UserData
import com.ngrob.android.bluemoon.core.notifications.NotificationType
import com.ngrob.android.bluemoon.core.util.getDateToSchedulePillInFuture
import com.ngrob.android.bluemoon.core.util.getDateToSchedulePillReminder
import java.time.Clock
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.LocalTime
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
class BluemoonReminderNotificationScheduler @Inject constructor(
    private val reminderScheduler: ReminderScheduler,
    private val clock: Clock
) : BluemoonReminderScheduler {

    override fun scheduleUpcomingPeriodReminder(userData: UserData, bleedings: List<LocalDate>) {
        reminderScheduler.cancel(NotificationType.UPCOMING_PERIOD_REMINDER)

        if (!userData.reminderSettings.remindForUpcomingPeriod) return
        val nextPeriod = MenstruationCycleTracker(bleedings).predictStartDateOfUpcomingPeriod()
            ?: return
        val reminderTime = LocalDateTime.of(nextPeriod.minusDays(2L), LocalTime.of(9, 0))
        reminderScheduler.schedule(
            reminderItem = ReminderItem(
                reminderTime,
                NotificationType.UPCOMING_PERIOD_REMINDER
            )
        )
    }


    private fun schedulePillReminder(userData: UserData) {
        reminderScheduler.cancel(NotificationType.PILL_REMINDER)
        if (!userData.trackBirthControl || !userData.reminderSettings.remindForTakingPill) return

        val pillRecipe = userData.pillRecipe
        val scheduledTime = userData.reminderSettings.pillReminderTime
        val dateTime =
            getDateToSchedulePillReminder(
                pillRecipe,
                currentDate = LocalDate.now(clock),
                scheduledTime = scheduledTime
            )
        reminderScheduler.schedule(
            ReminderItem(
                dateTime,
                NotificationType.PILL_REMINDER
            )
        )
    }

    private fun scheduleDailyReminder(userData: UserData) {
        reminderScheduler.cancel(NotificationType.DAILY_REMINDER)
        if (!userData.reminderSettings.remindForTrackingSymptomsDaily) return
        val dateTime = LocalDateTime.of(LocalDate.now(clock), LocalTime.of(18, 0))
        reminderScheduler.schedule(ReminderItem(dateTime, NotificationType.DAILY_REMINDER))

    }


    override fun scheduleAll(userData: UserData, bleedings: List<LocalDate>) {
        scheduleUpcomingPeriodReminder(userData, bleedings)
        scheduleDailyReminder(userData)
        schedulePillReminder(userData)
    }

    fun rescheduleRepeatingReminder(notificationType: NotificationType, userData: UserData) {
        when (notificationType) {
            NotificationType.PILL_REMINDER -> reschedulePillReminder(userData)
            NotificationType.DAILY_REMINDER -> rescheduleDailyReminder(userData)
            NotificationType.UPCOMING_PERIOD_REMINDER -> Unit
        }
    }

    private fun reschedulePillReminder(userData: UserData) {
        if (!userData.trackBirthControl) return
        val pillRecipe = userData.pillRecipe
        val scheduledTime = userData.reminderSettings.pillReminderTime
        val dateTime = getDateToSchedulePillInFuture(
            pillRecipe,
            currentDate = LocalDate.now(clock),
            scheduledTime = scheduledTime
        )
        reminderScheduler.schedule(ReminderItem(dateTime, NotificationType.PILL_REMINDER))
    }

    private fun rescheduleDailyReminder(userData: UserData) {
        if (!userData.reminderSettings.remindForTrackingSymptomsDaily) return
        reminderScheduler.schedule(
            ReminderItem(
                LocalDateTime.of(
                    LocalDate.now(clock).plusDays(1),
                    LocalTime.of(18, 0)
                ),
                NotificationType.DAILY_REMINDER
            )
        )
    }
}