package com.ngrob.android.bluemoon.core.onboarding

import android.content.Context
import android.net.Uri
import android.util.Log
import com.google.gson.Gson
import com.google.gson.JsonSyntaxException
import com.ngrob.android.bluemoon.core.database.model.Bleeding
import com.ngrob.android.bluemoon.core.database.model.BleedingStrength
import com.ngrob.android.bluemoon.core.model.FloData
import kotlinx.serialization.json.Json
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import java.time.temporal.ChronoUnit


/**
 * Extract Flo data from a JSON file.
 *
 * @param uri The URI of the JSON file.
 * @param context
 * @return A FloData object containing the data from the JSON.
 */
fun extractFloData(uri: Uri, context: Context): FloData? {
    val gson = Gson()
    val contentResolver = context.contentResolver
    var jsonString: String?

    contentResolver.openInputStream(uri)?.bufferedReader().use { reader ->
        jsonString = reader?.readText()
        val gsonObject: FloData? = try {
            gson.fromJson(jsonString, FloData::class.java)
        } catch (e: JsonSyntaxException) {
            Log.w("SettingsViewModel", "JSON starts with an array or a literal, instead of an object.")
            null
        }
        if (gsonObject?.operationalData != null) {
            Log.d("Gson object", "Gson read ${gsonObject.operationalData?.cycles?.size} cycles from the JSON file")
            return gsonObject
        }
        else {
            Log.w("SettingsViewModel", "Failed to read file content or file is empty.")
            return null
        }
    }
}


/**
 * Convert a FloData Gson object to a List of Bleedings.
 *
 * @param floData The FloData object containing the cycle data.
 * @return A list of Bleedings that represent the cycles.
 */
fun extractBleedings(floData: FloData): List<Bleeding> {
    val allBleedings = mutableListOf<Bleeding>()
    val dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")

    if (floData.operationalData == null) {
        return emptyList()
    }

    floData.operationalData?.cycles?.forEach { cycle ->
        val startDate = LocalDate.parse(cycle.periodStartDate.substringBefore("."), dateFormatter)
        val endDate = LocalDate.parse(cycle.periodEndDate.substringBefore("."), dateFormatter)
        val loggedAtString = LocalDateTime.parse(cycle.createdAt.substringBefore("."), dateFormatter).format(DateTimeFormatter.ISO_LOCAL_DATE_TIME)
        val intensities: Map<Long, BleedingStrength?> = deserializePeriodIntensities(cycle.periodIntensities)

        // between() includes the start date, but not the end date
        val periodLength = if (startDate.equals(endDate)) 1 else ChronoUnit.DAYS.between(startDate, endDate) + 1
        for (dateIdx in 0 until periodLength) {
            val bleedingEntity = Bleeding(
                id = null,
                strength = intensities[dateIdx],
                date = startDate.plusDays(dateIdx).format(DateTimeFormatter.ISO_LOCAL_DATE),
                loggedAt = loggedAtString
            )
            allBleedings.add(bleedingEntity)
        }
    }

    return allBleedings
}

fun deserializePeriodIntensities(periodIntensities: String): Map<Long, BleedingStrength?> {
    val intensitiesMapStringInt: Map<String, Int> = Json.decodeFromString(periodIntensities)
    val intensitiesMap: Map<Long, BleedingStrength?> = intensitiesMapStringInt.mapKeys { it.key.toLong() }
        .mapValues { (_, value) -> intToBleedingStrength(value) }

    return intensitiesMap
}

fun intToBleedingStrength(value: Int): BleedingStrength? {
    return when (value) {
        1 -> BleedingStrength.LIGHT
        2 -> BleedingStrength.MEDIUM
        3 -> BleedingStrength.HEAVY
        else -> null
    }
}
