package com.ngrob.android.bluemoon.core.notifications

import android.Manifest.permission
import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import dagger.hilt.android.qualifiers.ApplicationContext
import javax.inject.Inject
import javax.inject.Singleton
import android.content.pm.PackageManager.PERMISSION_GRANTED
import android.os.Build.VERSION
import android.os.Build.VERSION_CODES
import androidx.core.app.ActivityCompat.checkSelfPermission
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import com.ngrob.android.bluemoon.R

private const val REMINDERS_NOTIFICATION_CHANNEL_ID = ""
private const val REMINDERS_NOTIFICATION_GROUP = "REMINDERS_PERIOD"
private const val TARGET_ACTIVITY_NAME = "com.ngrob.android.bluemoon.MainActivity"

@Singleton
internal class SystemTrayNotifier @Inject constructor(
    @ApplicationContext private val context: Context,
) : Notifier {
    override fun postNotification(notificationType: NotificationType) = with(context) {
        if (checkSelfPermission(this, permission.POST_NOTIFICATIONS) != PERMISSION_GRANTED) {
            return@with
        }
        val notification = createNewsNotification {
            setSmallIcon(
                R.drawable.ic_notification,
            )
                .setContentTitle(notificationType.contentTitle)
                .setContentText(notificationType.contentDescription)
                .setContentIntent(reminderPendingIntent())
                .setGroup(REMINDERS_NOTIFICATION_GROUP)
                .setAutoCancel(true)
        }
        val notificationManager = NotificationManagerCompat.from(this)
        notificationManager.notify(1, notification)

    }

    private fun Context.createNewsNotification(
        block: NotificationCompat.Builder.() -> Unit,
    ): Notification {
        ensureNotificationChannelExists()
        return NotificationCompat.Builder(
            this,
            REMINDERS_NOTIFICATION_CHANNEL_ID,
        )
            .setPriority(NotificationCompat.PRIORITY_DEFAULT)
            .apply(block)
            .build()
    }

    private fun Context.ensureNotificationChannelExists() {
        if (VERSION.SDK_INT < VERSION_CODES.O) return

        val channel = NotificationChannel(
            REMINDERS_NOTIFICATION_CHANNEL_ID,
            getString(R.string.core_notifications_reminder_notification_channel_name),
            NotificationManager.IMPORTANCE_DEFAULT,
        ).apply {
            description =
                getString(R.string.core_notifications_reminder_notification_channel_description)
        }
        // Register the channel with the system
        NotificationManagerCompat.from(this).createNotificationChannel(channel)
    }

    private fun Context.reminderPendingIntent(
    ): PendingIntent? = PendingIntent.getActivity(
        this,
        0,
        Intent().apply {
            action = Intent.ACTION_VIEW
            component = ComponentName(
                packageName,
                TARGET_ACTIVITY_NAME
            )
        },
        PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE,
    )

}