package com.ngrob.android.bluemoon.core.datastore

import android.util.Log
import androidx.datastore.core.DataStore
import com.ngrob.android.bluemoon.core.model.PillRecipe
import com.ngrob.android.bluemoon.core.model.ReminderSettings
import com.ngrob.android.bluemoon.core.model.UserData
import kotlinx.coroutines.flow.map
import java.io.IOException
import java.time.Clock
import java.time.LocalDate
import java.time.LocalTime
import java.time.format.DateTimeFormatter
import javax.inject.Inject

class BluemoonPreferencesDataSource @Inject constructor(
    private val userPreferences: DataStore<UserPreferences>,
    private val clock: Clock
) {

    val userData = userPreferences.data.map {
        UserData(
            trackBirthControl = it.trackBirthControl,
            pillRecipe = if (it.hasPillRecipe()) PillRecipe(
                numberOfPills = it.pillRecipe.numberOfPills, firstIntake = LocalDate.parse(
                    it.pillRecipe.pillFirstIntake, DateTimeFormatter.ISO_LOCAL_DATE
                )
            ) else PillRecipe(
                numberOfPills = 24, firstIntake = LocalDate.now(clock)
            ),
            reminderSettings = ReminderSettings(
                it.reminderSettings.remindForUpcomingPeriod,
                it.reminderSettings.remindForTrackingSymptomsDaily,
                it.reminderSettings.remindForTakingPill,
                if (it.reminderSettings.hasPillReminderTime()) LocalTime.parse(it.reminderSettings.pillReminderTime) else LocalTime.now(
                    clock
                )
            )
        )

    }

    suspend fun setPillRecipe(pillRecipe: PillRecipe) {
        try {
            userPreferences.updateData {
                it.copy {

                    this.pillRecipe = PillRecipeProto.getDefaultInstance().copy {
                        this.pillFirstIntake =
                            pillRecipe.firstIntake.format(DateTimeFormatter.ISO_LOCAL_DATE)
                        this.numberOfPills = pillRecipe.numberOfPills
                    }

                }
            }
        } catch (
            ioException: IOException
        ) {
            Log.e("BluemoonPreferences", "Failed to update user preferences", ioException)
        }
    }

    suspend fun setTrackBirthControl(trackBirthControl: Boolean) {
        try {
            userPreferences.updateData {
                it.copy {
                    this.trackBirthControl = trackBirthControl
                }
            }
        } catch (
            ioException: IOException
        ) {
            Log.e("BluemoonPreferences", "Failed to update user preferences", ioException)
        }
    }

    suspend fun setReminderSettings(reminderSettings: ReminderSettings) {
        try {
            userPreferences.updateData {
                it.copy {
                    this.reminderSettings = ReminderSettingsProto.getDefaultInstance().copy {
                        this.remindForTakingPill = reminderSettings.remindForTakingPill
                        this.remindForTrackingSymptomsDaily =
                            reminderSettings.remindForTrackingSymptomsDaily
                        this.remindForUpcomingPeriod = reminderSettings.remindForUpcomingPeriod
                        this.pillReminderTime = reminderSettings.pillReminderTime.toString()
                    }
                }
            }
        } catch (
            ioException: IOException
        ) {
            Log.e("BluemoonPreferences", "Failed to update user preferences", ioException)
        }
    }
}

