package com.ngrob.android.bluemoon.core.data.repository

import com.ngrob.android.bluemoon.core.database.dao.BleedingDao
import com.ngrob.android.bluemoon.core.database.model.Bleeding
import com.ngrob.android.bluemoon.core.database.model.BleedingStrength
import com.ngrob.android.bluemoon.core.reminder.BluemoonReminderScheduler
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.map
import java.time.Clock
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.YearMonth
import java.time.format.DateTimeFormatter
import javax.inject.Inject


class OfflineBleedingRepository @Inject constructor(
    private val bleedingDao: BleedingDao,
    private val userDataRepository: UserDataRepository,
    private val bluemoonReminderScheduler: BluemoonReminderScheduler,
    private val clock: Clock,
) : BleedingRepository {

    override fun getAllBleedingsOfMonth(month: YearMonth): Flow<List<LocalDate>> {
        val startDate = month.atDay(1)
        val endDate = month.atEndOfMonth()
        val bleedingFlow = bleedingDao.getAllBleedingDaysOfMonth(
            startDate.format(DateTimeFormatter.ISO_LOCAL_DATE),
            endDate.format(DateTimeFormatter.ISO_LOCAL_DATE)
        )
        return bleedingFlow.map { dateStrings ->
            val bleedingDays: MutableList<LocalDate> = mutableListOf()
            dateStrings.forEach { value ->
                bleedingDays.add(LocalDate.parse(value, DateTimeFormatter.ISO_LOCAL_DATE))
            }
            bleedingDays.toList()
        }
    }

    override fun getAllBleedingDates(): Flow<List<LocalDate>> {
        val bleedingFlow = bleedingDao.getAllBleedingDates().map { dateStrings ->
            val bleedingDays: MutableList<LocalDate> = mutableListOf()
            dateStrings.forEach { value ->
                bleedingDays.add(LocalDate.parse(value, DateTimeFormatter.ISO_LOCAL_DATE))
            }
            bleedingDays.toList()
        }
        val userDataFlow = userDataRepository.userData

        return combine(bleedingFlow, userDataFlow) { bleedings, userData ->
            bluemoonReminderScheduler.scheduleUpcomingPeriodReminder(userData, bleedings)
            bleedings
        }.flowOn(Dispatchers.IO)


    }

    override fun getBleedingEntryOfDay(day: LocalDate): Flow<Bleeding?> {
        return bleedingDao.getBleedingOfDay(day.format(DateTimeFormatter.ISO_LOCAL_DATE))
    }

    override suspend fun addBleeding(strength: BleedingStrength, date: LocalDate) {
        val bleedingEntity = Bleeding(
            id = null,
            strength = strength,
            date = date.format(DateTimeFormatter.ISO_LOCAL_DATE),
            loggedAt = LocalDateTime.now(clock).format(DateTimeFormatter.ISO_LOCAL_DATE_TIME)
        )
        bleedingDao.insertBleedings(bleedingEntity)
    }

    override suspend fun updateBleeding(strength: BleedingStrength, date: LocalDate) {
        val bleedingEntity =
            bleedingDao.getBleedingOfDay(date.format(DateTimeFormatter.ISO_LOCAL_DATE))
                .firstOrNull()

        if (bleedingEntity == null) {
            bleedingDao.upsertBleedings(
                Bleeding(
                    id = null,
                    strength = strength,
                    date = date.format(DateTimeFormatter.ISO_LOCAL_DATE),
                    loggedAt = LocalDateTime.now(clock)
                        .format(DateTimeFormatter.ISO_LOCAL_DATE_TIME)
                )
            )
            return
        }
        bleedingEntity.strength = strength
        bleedingEntity.loggedAt =
            LocalDateTime.now(clock).format(DateTimeFormatter.ISO_LOCAL_DATE_TIME)
        bleedingDao.upsertBleedings(bleedingEntity)
    }

    override suspend fun addBleedings(dates: List<LocalDate>) {
        val entities = dates.map { date ->
            val bleedingEntity = Bleeding(
                id = null,
                strength = null,
                date = date.format(DateTimeFormatter.ISO_LOCAL_DATE),
                loggedAt = LocalDateTime.now(clock).format(DateTimeFormatter.ISO_LOCAL_DATE_TIME)
            )
            bleedingEntity
        }
        bleedingDao.insertBleedings(*entities.toTypedArray())
    }

    override fun importBleedings(bleedings: List<Bleeding>) {
        bleedingDao.insertBleedings(*bleedings.toTypedArray())
    }

    override suspend fun deleteBleedings(dates: List<LocalDate>) {
        val datesToDeleteIso = dates.map { date ->
            date.format(DateTimeFormatter.ISO_LOCAL_DATE)
        }
        bleedingDao.deleteBleedingDates(datesToDeleteIso)
    }

    override suspend fun deleteBleedingById(id: Int) {
        bleedingDao.deleteBleedingById(id)
    }

    override suspend fun deleteAllBleeding() {
        bleedingDao.deleteAllBleeding()
    }

}