package com.ngrob.android.bluemoon

import java.time.LocalDate
import java.time.Period
import java.time.YearMonth
import java.time.temporal.ChronoUnit
import java.util.stream.Collectors
import java.util.stream.LongStream
import kotlin.math.roundToInt

data class MenstruationCycle(val startDate: LocalDate, val endDate: LocalDate)


class MenstruationCycleTracker() {

    private val cycles = mutableListOf<MenstruationCycle>()
    private var meanPeriodLength: Int = 0
    private var meanCycleLength: Int = 0

    constructor(bleedingDates: List<LocalDate>) : this() {
        bleedingDates.sorted().forEach { date ->
            if (cycles.isEmpty() || date > cycles.last().endDate.plusDays(4)) {
                // Start a new cycle
                cycles.add(MenstruationCycle(startDate = date, endDate = date))
            } else {
                // Extend the current cycle
                cycles[cycles.size - 1] = cycles.last().copy(endDate = date)
            }
        }

        // Calculate mean period and cycle length
        if (cycles.isNotEmpty()) {
            var periodLengthSum = ChronoUnit.DAYS.between(
                cycles[0].startDate,
                cycles[0].endDate
            ).toInt() + 1
            var cycleLengthSum = 0
            var prevCycleStartDate = cycles[0].startDate
            for (i in 1..<cycles.size) {
                periodLengthSum += ChronoUnit.DAYS.between(
                    cycles[i].startDate,
                    cycles[i].endDate
                ).toInt() + 1
                cycleLengthSum += ChronoUnit.DAYS.between(prevCycleStartDate, cycles[i].startDate)
                    .toInt()
                prevCycleStartDate = cycles[i].startDate
            }

            meanCycleLength =
                if (cycles.size > 1) (cycleLengthSum / (cycles.size - 1).toFloat()).roundToInt() else 0
            meanPeriodLength = (periodLengthSum / cycles.size.toFloat()).roundToInt()
        }

    }

    fun getCycles(): List<MenstruationCycle> {
        return cycles.toList()
    }

    fun getRecentCycle(): MenstruationCycle? {
        return if (cycles.isEmpty()) null else cycles.last()
    }

    fun getCycleDayOfBleedingDate(bleeding: LocalDate): Long {
        val cycleOfBleeding = this.findCycleOfBleeding(bleeding)
        if (cycleOfBleeding == null) {
            println(this.cycles)
            return 0L
        }

        return ChronoUnit.DAYS.between(cycleOfBleeding.startDate, bleeding) + 1
    }

    private fun findCycleOfBleeding(date: LocalDate): MenstruationCycle? {
        return this.cycles.find {
            val periodFromStart = Period.between(it.startDate.minusDays(1), date)
            val periodToEnd = Period.between(date, it.endDate.plusDays(1))
            !periodFromStart.isNegative && !periodToEnd.isNegative && !periodFromStart.isZero && !periodToEnd.isZero

        }
    }

    fun getPeriodLengthMean(): Int {
        val defaultMean = 5
        return if (meanPeriodLength != 0) meanPeriodLength else defaultMean
    }

    fun predictStartDateOfUpcomingPeriod(): LocalDate? {
        if (cycles.size < 2) {
            return null
        }
        val currentCycleStartDate = getRecentCycle()?.startDate ?: return null
        return currentCycleStartDate.plusDays(meanCycleLength.toLong())
    }
    private fun predictPeriod(): List<LocalDate> {
        if (cycles.size < 2) {
            return listOf()
        }
        val bleedingPredicted = mutableListOf<LocalDate>()
        var currentCycleStartDate = getRecentCycle()?.startDate ?: return listOf()
        // Predict for 6 cycles
        for (i in 1..6) {
            val startOfCycle = currentCycleStartDate.plusDays(meanCycleLength.toLong())
            val period = LongStream.iterate(0) { j -> j + 1 }
                .limit(meanPeriodLength.toLong())
                .mapToObj { k ->
                    startOfCycle.plusDays(
                        k
                    )
                }
                .collect(Collectors.toList())
            currentCycleStartDate = startOfCycle
            bleedingPredicted.addAll(period)
        }
        return bleedingPredicted
    }

    fun predictPeriodDaysOfMonth(yearMonth: YearMonth): List<LocalDate> {
        val currentCycleStartDate = getRecentCycle()?.startDate ?: return listOf()

        // If requested prediction is more than 6 month in the future of the most current cycle don't to prediction
        if (YearMonth.from(currentCycleStartDate).plusMonths(6L) < yearMonth) {
            return listOf()
        }

        val allPredicted = predictPeriod()

        return allPredicted.filter {
            YearMonth.from(it) == yearMonth
        }
    }
}

