package com.ngrob.android.bluemoon

import android.content.pm.ActivityInfo
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.animation.EnterTransition
import androidx.compose.animation.ExitTransition
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.BottomAppBar
import androidx.compose.material.BottomNavigation
import androidx.compose.material.BottomNavigationItem
import androidx.compose.material.Icon
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.navigation.NavController
import androidx.navigation.NavDestination.Companion.hierarchy
import androidx.navigation.NavGraph.Companion.findStartDestination
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import com.ngrob.android.bluemoon.features.calendar.screens.CalendarScreen
import com.ngrob.android.bluemoon.core.ui.theme.BluemoonTheme
import com.ngrob.android.bluemoon.core.ui.theme.Gray600
import com.ngrob.android.bluemoon.core.ui.theme.Gray900
import com.ngrob.android.bluemoon.features.dashboard.screens.DashboardScreen
import com.ngrob.android.bluemoon.features.edit_period.screens.EditPeriodScreen
import com.ngrob.android.bluemoon.features.settings.screens.SettingsScreen
import com.ngrob.android.bluemoon.features.symptom.screens.AddSymptomScreen
import dagger.hilt.android.AndroidEntryPoint
import de.raphaelebner.roomdatabasebackup.core.RoomBackup
import android.Manifest
import android.graphics.Color
import android.os.Build
import androidx.activity.SystemBarStyle
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.background
import androidx.compose.material3.Scaffold
import com.google.accompanist.permissions.ExperimentalPermissionsApi
import com.google.accompanist.permissions.isGranted
import com.google.accompanist.permissions.rememberPermissionState
import com.google.accompanist.permissions.shouldShowRationale

@AndroidEntryPoint
class MainActivity : ComponentActivity() {

    private lateinit var roomBackup: RoomBackup

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge(transparentBarStyle, transparentBarStyle)

        roomBackup = RoomBackup(this@MainActivity)
        requestedOrientation = (ActivityInfo.SCREEN_ORIENTATION_PORTRAIT)
        setContent {
            BluemoonTheme {
                BluemoonApp(roomBackup)
            }
        }
    }
}

private val transparentBarStyle = SystemBarStyle.auto(
    lightScrim = Color.TRANSPARENT,
    darkScrim = Color.TRANSPARENT,
    detectDarkMode = { true }
)

@OptIn(ExperimentalPermissionsApi::class)
@Composable
fun BluemoonApp(
    roomBackup: RoomBackup,
    viewModel: BluemoonAppViewModel = hiltViewModel()
) {

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
        val notificationPermissionState =
            rememberPermissionState(Manifest.permission.POST_NOTIFICATIONS)

        LaunchedEffect(notificationPermissionState) {
            val permissionResult = notificationPermissionState.status

            if (!permissionResult.isGranted) {
                if (permissionResult.shouldShowRationale) {
                    // Show a rationale if needed (optional)
                } else {
                    // Request the permission
                    notificationPermissionState.launchPermissionRequest()
                }
            }
        }
    }

    val navController = rememberNavController()
    val uiState by viewModel.uiState.collectAsState()

    val items = listOf(
        BottomNavItem.Dashboard,
        BottomNavItem.Calendar,
        //BottomNavItem.Statistics,
        BottomNavItem.Settings,
    )
    var currentRoute by remember { mutableStateOf(navController.currentDestination?.route) }

    // Observe changes in the current route
    DisposableEffect(navController) {
        val listener = NavController.OnDestinationChangedListener { _, _, _ ->
            currentRoute = navController.currentDestination?.route
        }
        navController.addOnDestinationChangedListener(listener)
        // Remove the listener when the composable is disposed
        onDispose {
            navController.removeOnDestinationChangedListener(listener)
        }
    }

    Scaffold(
        modifier = Modifier
            .fillMaxSize(),
        bottomBar = {
            if (currentRoute != Route.AddSymptom.route && currentRoute != Route.EditPeriod.route) {
                BottomAppBar(
                    modifier = Modifier
                        .height(65.dp)
                        .clip(RoundedCornerShape(0.dp, 0.dp, 0.dp, 0.dp)),
                    cutoutShape = CircleShape,
                    contentPadding = PaddingValues(0.dp),
                    elevation = 50.dp
                ) {
                    BottomNavigation(Modifier.padding(0.dp), elevation = 0.dp) {
                        val navBackStackEntry by navController.currentBackStackEntryAsState()
                        val currentDestination = navBackStackEntry?.destination
                        items.forEach { screen ->
                            if (screen === null) {
                                BottomNavigationItem(
                                    selected = false,
                                    onClick = { },
                                    icon = {})
                            } else {
                                BottomNavigationItem(
                                    icon = {
                                        Icon(
                                            screen.icon,
                                            contentDescription = screen.label
                                        )
                                    },
                                    selectedContentColor = Gray900,
                                    unselectedContentColor = Gray600,
                                    label = { Text(text = screen.label) },
                                    selected = currentDestination?.hierarchy?.any { it.route == screen.route } == true,
                                    onClick = {
                                        navController.navigate(screen.route) {
                                            // Pop up to the start destination of the graph to
                                            // avoid building up a large stack of destinations
                                            // on the back stack as users select items
                                            popUpTo(navController.graph.findStartDestination().id) {
                                                saveState = true
                                            }
                                            // Avoid multiple copies of the same destination when
                                            // re selecting the same item
                                            launchSingleTop = true
                                            // Restore state when re selecting a previously selected item
                                            restoreState = true
                                        }
                                    }
                                )
                            }

                        }
                    }
                }
            }
        }
    ) { innerPadding ->
        NavHost(
            navController,
            startDestination = BottomNavItem.Dashboard.route,
            enterTransition = { EnterTransition.None },
            exitTransition = { ExitTransition.None },
            modifier = Modifier
                .fillMaxSize()
                .background(Gray900)
        ) {
            composable(BottomNavItem.Dashboard.route) {
                DashboardScreen(
                    Modifier.padding(innerPadding),
                    navController,
                )
            }
            composable(BottomNavItem.Calendar.route) {
                CalendarScreen(
                    Modifier.padding(innerPadding),
                    navController,
                    onSelectedDay = { newDate -> viewModel.selectDay(newDate) })
            }
            composable(BottomNavItem.Statistics.route) {
            }
            composable(BottomNavItem.Settings.route) {
                SettingsScreen(
                    Modifier.padding(innerPadding),
                    roomBackup = roomBackup
                )
            }
            composable(Route.AddSymptom.route) {
                AddSymptomScreen(
                    Modifier.padding(innerPadding),
                    navController,
                    currentDate = uiState.selectedDay
                )
            }
            composable(Route.EditPeriod.route) {
                EditPeriodScreen(Modifier.padding(innerPadding), navController)
            }
        }
    }
}


@Preview(showBackground = true)
@Composable
fun DefaultPreview() {
    BluemoonTheme {
        BluemoonApp(
            roomBackup = RoomBackup(
                LocalContext.current
            )
        )
    }
}
