package com.ngrob.android.bluemoon

import android.Manifest
import androidx.compose.ui.semantics.SemanticsActions
import androidx.compose.ui.semantics.getOrNull
import androidx.compose.ui.test.ExperimentalTestApi
import androidx.compose.ui.test.SemanticsMatcher
import androidx.compose.ui.test.assertIsDisplayed
import androidx.compose.ui.test.hasText
import androidx.compose.ui.test.junit4.createAndroidComposeRule
import androidx.compose.ui.test.onNodeWithText
import androidx.compose.ui.test.performClick
import androidx.datastore.core.DataStore
import androidx.datastore.core.DataStoreFactory
import androidx.test.rule.GrantPermissionRule
import com.ngrob.android.bluemoon.core.clock.ClockModule
import com.ngrob.android.bluemoon.core.data.repository.BleedingRepository
import com.ngrob.android.bluemoon.core.datastore.UserPreferences
import com.ngrob.android.bluemoon.core.datastore.UserPreferencesSerializer
import com.ngrob.android.bluemoon.core.datastore.di.DataStoreModule
import dagger.Module
import dagger.Provides
import dagger.hilt.android.testing.HiltAndroidRule
import dagger.hilt.android.testing.HiltAndroidTest
import dagger.hilt.components.SingletonComponent
import dagger.hilt.testing.TestInstallIn
import kotlinx.coroutines.runBlocking
import org.junit.After
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import java.io.File
import java.time.Clock
import java.time.Instant
import java.time.LocalDate
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import javax.inject.Inject
import javax.inject.Singleton


@Module
@TestInstallIn(components = [SingletonComponent::class], replaces = [ClockModule::class])
object TestClockModule {
    @Provides
    fun provideClock(): Clock =
//        1719838458 = 1st of July 2024
        Clock.fixed(Instant.ofEpochSecond(1719838458), ZoneId.systemDefault())
}

@Module
@TestInstallIn(components = [SingletonComponent::class], replaces = [DataStoreModule::class])
object TestDataStoreModule {

    @Singleton
    @Provides
    fun provideInMemoryDataStore(
        userPreferencesSerializer: UserPreferencesSerializer
    ): DataStore<UserPreferences> = DataStoreFactory.create(
        serializer = userPreferencesSerializer,
        produceFile = { File.createTempFile("user_preferences", "pb") }
    )
}

@HiltAndroidTest
class EditPeriodTest {


    @get:Rule(order = 0)
    var hiltRule = HiltAndroidRule(this)


    @get:Rule(order = 1)
    val composeTestRule = createAndroidComposeRule<MainActivity>()

    @JvmField
    @Rule(order = 2)
    var runtimePermissionRule: GrantPermissionRule =
        GrantPermissionRule.grant(Manifest.permission.POST_NOTIFICATIONS)

    @Inject
    lateinit var bleedingRepository: BleedingRepository

    @Before
    fun setUp() {
        hiltRule.inject()
        runBlocking { bleedingRepository.deleteAllBleeding() }
    }

    @After
    fun tearDown() {
        runBlocking {
            bleedingRepository.deleteAllBleeding()
        }
    }

    @OptIn(ExperimentalTestApi::class)
    @Test
    fun onPeriodEndedCanDeleteBleedingDay() {
        val dateToRemove = LocalDate.of(2024, 7, 5).format(
            DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
        )

        val dateCanBeRemovedMatcher = SemanticsMatcher("", matcher = {
            it.config.getOrNull(SemanticsActions.OnClick)?.label == "Remove bleeding day on $dateToRemove"
        })

        val dateCanBeAddedMatcher = SemanticsMatcher("", matcher = {
            it.config.getOrNull(SemanticsActions.OnClick)?.label == "Add bleeding day on $dateToRemove"
        })

        composeTestRule.onNodeWithText("Period started", ignoreCase = true)
            .performClick()
        composeTestRule.waitForIdle()
        composeTestRule.onNodeWithText("Period ended", ignoreCase = true).assertIsDisplayed()
        composeTestRule.onNodeWithText("Period ended", ignoreCase = true).performClick()
        composeTestRule.waitUntilAtLeastOneExists(hasText("Save Data"))

        composeTestRule.onNode(dateCanBeRemovedMatcher)
            .assertIsDisplayed()
            .performClick()

        composeTestRule.onNodeWithText("Save Data").performClick()
        composeTestRule.waitUntilAtLeastOneExists(
            hasText(
                "Overview",
                substring = true,
                ignoreCase = true
            ), timeoutMillis = 10000
        )
        composeTestRule.onNodeWithText("Period", substring = true, ignoreCase = true)
            .assertIsDisplayed()
            .performClick()
        composeTestRule.waitUntilAtLeastOneExists(hasText("Save Data"))
        composeTestRule.onNode(dateCanBeAddedMatcher).assertIsDisplayed()
    }

    @OptIn(ExperimentalTestApi::class)
    @Test
    fun onPeriodEndedCanAddBleedingDay() {
        val dateToAdd = LocalDate.of(2024, 7, 6).format(
            DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
        )

        val dateCanBeRemovedMatcher = SemanticsMatcher("", matcher = {
            it.config.getOrNull(SemanticsActions.OnClick)?.label == "Remove bleeding day on $dateToAdd"
        })

        val dateCanBeAddedMatcher = SemanticsMatcher("", matcher = {
            it.config.getOrNull(SemanticsActions.OnClick)?.label == "Add bleeding day on $dateToAdd"
        })

        composeTestRule.onNodeWithText("Period started", ignoreCase = true).performClick()
        composeTestRule.onNodeWithText("Period ended", ignoreCase = true).assertIsDisplayed()
        composeTestRule.onNodeWithText("Period ended", ignoreCase = true).performClick()
        composeTestRule.waitUntilAtLeastOneExists(hasText("Save Data"))

        composeTestRule.onNode(
            dateCanBeAddedMatcher
        )
            .assertIsDisplayed()
            .performClick()

        composeTestRule.onNodeWithText("Save Data").performClick()
        composeTestRule.waitUntilAtLeastOneExists(
            hasText(
                "Overview",
                substring = true,
                ignoreCase = true
            ), timeoutMillis = 10000
        )
        composeTestRule.onNodeWithText("Period", substring = true, ignoreCase = true)
            .assertIsDisplayed()
            .performClick()
        composeTestRule.waitUntilAtLeastOneExists(hasText("Save Data"))
        composeTestRule.onNode(dateCanBeRemovedMatcher).assertIsDisplayed()
    }
}