package com.ngrob.android.bluemoon

import android.Manifest
import androidx.compose.ui.semantics.SemanticsActions
import androidx.compose.ui.semantics.getOrNull
import androidx.compose.ui.test.ExperimentalTestApi
import androidx.compose.ui.test.SemanticsMatcher
import androidx.compose.ui.test.filter
import androidx.compose.ui.test.hasClickAction
import androidx.compose.ui.test.hasTestTag
import androidx.compose.ui.test.hasText
import androidx.compose.ui.test.junit4.createAndroidComposeRule
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.onNodeWithText
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performSemanticsAction
import androidx.compose.ui.test.performTouchInput
import androidx.compose.ui.test.swipeDown
import androidx.compose.ui.test.swipeLeft
import androidx.test.rule.GrantPermissionRule
import com.ngrob.android.bluemoon.core.data.repository.BleedingRepository
import dagger.hilt.android.testing.HiltAndroidRule
import dagger.hilt.android.testing.HiltAndroidTest
import kotlinx.coroutines.runBlocking
import org.junit.After
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import java.time.LocalDate
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import javax.inject.Inject


@HiltAndroidTest
class CalendarPredictBleedingTest {


    @get:Rule(order = 0)
    var hiltRule = HiltAndroidRule(this)


    @get:Rule(order = 1)
    val composeTestRule = createAndroidComposeRule<MainActivity>()

    @JvmField
    @Rule(order = 2)
    var runtimePermissionRule: GrantPermissionRule =
        GrantPermissionRule.grant(Manifest.permission.POST_NOTIFICATIONS)

    @Inject
    lateinit var bleedingRepository: BleedingRepository

    @Before
    fun setUp() {
        hiltRule.inject()
        runBlocking { bleedingRepository.deleteAllBleeding() }
    }

    @After
    fun tearDown() {
        runBlocking {
            bleedingRepository.deleteAllBleeding()
        }
    }

    @OptIn(ExperimentalTestApi::class)
    @Test
    fun onTwoPeriodsEnteredPredictBleedingDays() {
        val firstPeriodDays = listOf(
            LocalDate.of(2024, 6, 1).format(
                DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
            ),
            LocalDate.of(2024, 6, 2).format(
                DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
            ),
            LocalDate.of(2024, 6, 3).format(
                DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
            ),
            LocalDate.of(2024, 6, 4).format(
                DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
            ),
            LocalDate.of(2024, 6, 5).format(
                DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM)
            ),
        )
        val datesToAdd = SemanticsMatcher("", matcher = {
            it.config.getOrNull(SemanticsActions.OnClick)?.label in firstPeriodDays.map { date -> "Add bleeding day on $date" }
        })


        composeTestRule.onNodeWithText("Period started", ignoreCase = true)
            .performClick()
        composeTestRule.waitForIdle()
        composeTestRule.onNodeWithText("Period ended", ignoreCase = true).performClick()
        composeTestRule.waitUntilAtLeastOneExists(hasText("Save Data"))

        composeTestRule.onNodeWithTag("editScreenLazyColumn").performTouchInput { swipeDown() }
        composeTestRule.waitForIdle()

        // Add second period in edit view
        while (true) {
            val nodes = composeTestRule.onAllNodes(datesToAdd).filter(hasClickAction())
                .fetchSemanticsNodes()

            if (nodes.isEmpty()) {
                break
            }

            nodes[0].let { node ->
                composeTestRule.onNode(
                    matcher = SemanticsMatcher(
                        "",
                        matcher = {
                            node.config.get(SemanticsActions.OnClick).label == it.config.getOrNull(
                                SemanticsActions.OnClick
                            )?.label
                        })
                ).performSemanticsAction(SemanticsActions.OnClick)
            }

            composeTestRule.waitForIdle()
        }


        composeTestRule.onNodeWithText("Save Data").performClick()
        composeTestRule.waitUntilAtLeastOneExists(
            hasText(
                "Overview",
                substring = true,
                ignoreCase = true
            ), timeoutMillis = 10000
        )

        composeTestRule.onNodeWithText("Calendar").performClick()
        composeTestRule.waitForIdle()


        composeTestRule.onNodeWithTag("MonthCalendar").performTouchInput { swipeLeft() }
        composeTestRule.waitUntilAtLeastOneExists(
            hasTestTag("calendarDayPredictedBleedingtrue")
        )
    }


}