package bou.amine.apps.readerforselfossv2.android.fragments

import android.content.Context
import android.content.Intent
import android.content.res.TypedArray
import android.graphics.Bitmap
import android.graphics.Typeface
import android.os.Bundle
import android.util.TypedValue.DATA_NULL_UNDEFINED
import android.view.GestureDetector
import android.view.InflateException
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.webkit.WebResourceResponse
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.Toast
import androidx.appcompat.app.AlertDialog
import androidx.fragment.app.Fragment
import bou.amine.apps.readerforselfossv2.android.ImageActivity
import bou.amine.apps.readerforselfossv2.android.R
import bou.amine.apps.readerforselfossv2.android.databinding.FragmentArticleBinding
import bou.amine.apps.readerforselfossv2.android.model.ParecelableItem
import bou.amine.apps.readerforselfossv2.android.model.toModel
import bou.amine.apps.readerforselfossv2.android.model.toParcelable
import bou.amine.apps.readerforselfossv2.android.utils.acra.sendSilentlyWithAcraWithName
import bou.amine.apps.readerforselfossv2.android.utils.bottombar.addHomeMadeActionItem
import bou.amine.apps.readerforselfossv2.android.utils.getColorFromAttr
import bou.amine.apps.readerforselfossv2.android.utils.glide.bitmapFitCenter
import bou.amine.apps.readerforselfossv2.android.utils.glide.getBitmapInputStream
import bou.amine.apps.readerforselfossv2.android.utils.glide.getGlideImageForResource
import bou.amine.apps.readerforselfossv2.android.utils.isUrlValid
import bou.amine.apps.readerforselfossv2.android.utils.maybeIfContext
import bou.amine.apps.readerforselfossv2.android.utils.openItemUrlInBrowserAsNewTask
import bou.amine.apps.readerforselfossv2.android.utils.openUrlInBrowserAsNewTask
import bou.amine.apps.readerforselfossv2.android.utils.shareLink
import bou.amine.apps.readerforselfossv2.model.MercuryModel
import bou.amine.apps.readerforselfossv2.model.SelfossModel
import bou.amine.apps.readerforselfossv2.repository.Repository
import bou.amine.apps.readerforselfossv2.rest.MercuryApi
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import bou.amine.apps.readerforselfossv2.service.ConnectivityService
import bou.amine.apps.readerforselfossv2.utils.getHtmlDecoded
import bou.amine.apps.readerforselfossv2.utils.getImages
import bou.amine.apps.readerforselfossv2.utils.getThumbnail
import bou.amine.apps.readerforselfossv2.utils.isEmptyOrNullOrNullString
import com.leinardi.android.speeddial.SpeedDialView
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.acra.ktx.sendSilentlyWithAcra
import org.kodein.di.DI
import org.kodein.di.DIAware
import org.kodein.di.android.x.closestDI
import org.kodein.di.instance
import java.net.MalformedURLException
import java.net.URL
import java.util.Locale
import java.util.concurrent.ExecutionException

private const val IMAGE_JPG = "image/jpg"
private const val IMAGE_PNG = "image/png"
private const val IMAGE_WEBP = "image/webp"

private const val WHITE_COLOR_HEX = 0xFFFFFF

private const val DEFAULT_FONT_SIZE = 16

class ArticleFragment :
    Fragment(),
    DIAware {
    private var colorOnSurface: Int = 0
    private var colorSurface: Int = 0
    private var fontSize: Int = DEFAULT_FONT_SIZE
    private lateinit var item: SelfossModel.Item
    private var url: String? = null
    private lateinit var contentText: String
    private lateinit var contentSource: String
    private lateinit var contentImage: String
    private lateinit var contentTitle: String
    private lateinit var allImages: ArrayList<String>
    private lateinit var fab: SpeedDialView
    private lateinit var textAlignment: String
    private lateinit var binding: FragmentArticleBinding

    override val di: DI by closestDI()
    private val repository: Repository by instance()
    private val appSettingsService: AppSettingsService by instance()
    private val connectivityService: ConnectivityService by instance()

    private var typeface: Typeface? = null
    private var resId: Int = 0
    private var font = ""

    private val mercuryApi: MercuryApi by instance()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val pi: ParecelableItem = requireArguments().getParcelable(ARG_ITEMS)!!

        item = pi.toModel()
    }

    @Suppress("detekt:LongMethod")
    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View {
        try {
            binding = FragmentArticleBinding.inflate(inflater, container, false)

            try {
                url = item.getLinkDecoded()
            } catch (e: Exception) {
                e.sendSilentlyWithAcra()
            }

            colorOnSurface = getColorFromAttr(com.google.android.material.R.attr.colorOnSurface)
            colorSurface = getColorFromAttr(com.google.android.material.R.attr.colorSurface)

            contentText = item.content
            contentTitle = item.title.getHtmlDecoded()
            contentImage = item.getThumbnail(repository.baseUrl)
            contentSource =
                try {
                    item.sourceAuthorAndDate()
                } catch (e: Exception) {
                    e.sendSilentlyWithAcraWithName("Article Fragment parse date")
                    item.sourceAuthorOnly()
                }
            allImages = item.getImages()

            fontSize = appSettingsService.getFontSize()
            font = appSettingsService.getFont()

            refreshAlignment()

            handleFloatingToolbar()

            binding.source.text = contentSource
            if (typeface != null) {
                binding.source.typeface = typeface
            }

            handleContent()
        } catch (e: InflateException) {
            e.sendSilentlyWithAcraWithName("webview not available")
            maybeIfContext {
                AlertDialog
                    .Builder(it)
                    .setMessage(it.getString(R.string.webview_dialog_issue_message))
                    .setTitle(it.getString(R.string.webview_dialog_issue_title))
                    .setPositiveButton(
                        android.R.string.ok,
                    ) { _, _ ->
                        appSettingsService.disableArticleViewer()
                        requireActivity().finish()
                    }.create()
                    .show()
            }
        }

        return binding.root
    }

    private fun handleContent() {
        if (contentText.isEmptyOrNullOrNullString()) {
            if (connectivityService.isNetworkAvailable() && url.isUrlValid()) {
                getContentFromMercury(url!!)
            }
        } else {
            binding.titleView.text = contentTitle
            if (typeface != null) {
                binding.titleView.typeface = typeface
            }

            htmlToWebview()

            if (!contentImage.isEmptyOrNullOrNullString() && context != null) {
                binding.imageView.visibility = View.VISIBLE
                maybeIfContext { it.bitmapFitCenter(contentImage, binding.imageView, appSettingsService) }
            } else {
                binding.imageView.visibility = View.GONE
            }
        }
    }

    private fun handleFloatingToolbar() {
        fab = binding.speedDial
        fab.mainFabClosedIconColor = colorOnSurface
        fab.mainFabOpenedIconColor = colorOnSurface

        maybeIfContext { handleFloatingToolbarActionItems(it) }

        fab.setOnActionSelectedListener { actionItem ->
            when (actionItem.id) {
                R.id.share_action -> requireActivity().shareLink(url, contentTitle)
                R.id.open_action -> requireActivity().openItemUrlInBrowserAsNewTask(this@ArticleFragment.item)
                R.id.unread_action ->
                    if (this@ArticleFragment.item.unread) {
                        CoroutineScope(Dispatchers.IO).launch {
                            repository.markAsRead(this@ArticleFragment.item)
                        }
                        this@ArticleFragment.item.unread = false
                        maybeIfContext {
                            Toast
                                .makeText(
                                    it,
                                    R.string.marked_as_read,
                                    Toast.LENGTH_LONG,
                                ).show()
                        }
                    } else {
                        CoroutineScope(Dispatchers.IO).launch {
                            repository.unmarkAsRead(this@ArticleFragment.item)
                        }
                        this@ArticleFragment.item.unread = true
                        maybeIfContext {
                            Toast
                                .makeText(
                                    it,
                                    R.string.marked_as_unread,
                                    Toast.LENGTH_LONG,
                                ).show()
                        }
                    }

                else -> Unit
            }
            false
        }
    }

    private fun handleFloatingToolbarActionItems(c: Context) {
        fab.addHomeMadeActionItem(
            R.id.share_action,
            resources.getDrawable(R.drawable.ic_share_white_24dp),
            R.string.reader_action_share,
            colorOnSurface,
            colorSurface,
            c,
        )
        fab.addHomeMadeActionItem(
            R.id.open_action,
            resources.getDrawable(R.drawable.ic_open_in_browser_white_24dp),
            R.string.reader_action_open,
            colorOnSurface,
            colorSurface,
            c,
        )
        fab.addHomeMadeActionItem(
            R.id.unread_action,
            resources.getDrawable(R.drawable.ic_baseline_white_eye_24dp),
            R.string.unmark,
            colorOnSurface,
            colorSurface,
            c,
        )
    }

    fun refreshAlignment() {
        textAlignment =
            when (appSettingsService.getActiveAllignment()) {
                1 -> "justify"
                2 -> "left"
                else -> "justify"
            }

        htmlToWebview()
    }

    @Suppress("detekt:SwallowedException")
    private fun getContentFromMercury(url: String) {
        binding.progressBar.visibility = View.VISIBLE

        CoroutineScope(Dispatchers.Main).launch {
            try {
                val response = mercuryApi.query(url)
                if (response.success && response.data != null) {
                    handleMercuryData(response.data!!)
                } else {
                    openInBrowserAfterFailing()
                }
            } catch (e: Exception) {
                openInBrowserAfterFailing()
            }
        }
    }

    private fun handleMercuryData(data: MercuryModel.ParsedContent) {
        if (data.error == true || data.failed == true) {
            openInBrowserAfterFailing()
        } else {
            binding.titleView.text = data.title.orEmpty()
            if (typeface != null) {
                binding.titleView.typeface = typeface
            }
            URL(data.url)
            url = data.url!!

            contentText = data.content.orEmpty()
            htmlToWebview()

            handleLeadImage(data.lead_image_url)

            binding.nestedScrollView.scrollTo(0, 0)
            binding.progressBar.visibility = View.GONE
        }
    }

    private fun handleLeadImage(leadImageUrl: String?) {
        if (!leadImageUrl.isNullOrEmpty()) {
            maybeIfContext {
                binding.imageView.visibility = View.VISIBLE
                it.bitmapFitCenter(leadImageUrl, binding.imageView, appSettingsService)
            }
        } else {
            binding.imageView.visibility = View.GONE
        }
    }

    private fun handleImageLoading() {
        binding.webcontent.webViewClient =
            object : WebViewClient() {
                @Deprecated("Deprecated in Java")
                override fun shouldOverrideUrlLoading(
                    view: WebView?,
                    url: String,
                ): Boolean =
                    if (url.isUrlValid() &&
                        binding.webcontent.hitTestResult.type != WebView.HitTestResult.SRC_IMAGE_ANCHOR_TYPE
                    ) {
                        maybeIfContext { it.openUrlInBrowserAsNewTask(url) }
                        true
                    } else {
                        false
                    }

                @Suppress("detekt:SwallowedException", "detekt:ReturnCount")
                @Deprecated("Deprecated in Java")
                override fun shouldInterceptRequest(
                    view: WebView,
                    url: String,
                ): WebResourceResponse? {
                    val (mime: String?, compression: Bitmap.CompressFormat) =
                        if (url
                                .lowercase(Locale.US)
                                .contains(".jpg") ||
                            url.lowercase(Locale.US).contains(".jpeg")
                        ) {
                            Pair(IMAGE_JPG, Bitmap.CompressFormat.JPEG)
                        } else if (url.lowercase(Locale.US).contains(".png")) {
                            Pair(IMAGE_PNG, Bitmap.CompressFormat.PNG)
                        } else if (url.lowercase(Locale.US).contains(".webp")) {
                            Pair(IMAGE_WEBP, Bitmap.CompressFormat.WEBP)
                        } else {
                            return super.shouldInterceptRequest(view, url)
                        }

                    try {
                        val image = view.getGlideImageForResource(url, appSettingsService)
                        return WebResourceResponse(
                            mime,
                            "UTF-8",
                            getBitmapInputStream(image, compression),
                        )
                    } catch (e: ExecutionException) {
                        return super.shouldInterceptRequest(view, url)
                    }
                }
            }
    }

    @Suppress("detekt:LongMethod", "detekt:ImplicitDefaultLocale")
    private fun htmlToWebview() {
        maybeIfContext {
            val attrs: IntArray = intArrayOf(android.R.attr.fontFamily)
            val a: TypedArray = it.obtainStyledAttributes(resId, attrs)

            binding.webcontent.settings.standardFontFamily = a.getString(0)
            ""
        }
        binding.webcontent.visibility = View.VISIBLE

        val colorSurfaceString =
            String.format(
                "#%06X",
                WHITE_COLOR_HEX and (if (colorSurface != DATA_NULL_UNDEFINED) colorSurface else WHITE_COLOR_HEX),
            )

        val colorOnSurfaceString =
            String.format(
                "#%06X",
                WHITE_COLOR_HEX and (if (colorOnSurface != DATA_NULL_UNDEFINED) colorOnSurface else 0),
            )

        binding.webcontent.settings.useWideViewPort = true
        binding.webcontent.settings.loadWithOverviewMode = true
        binding.webcontent.settings.javaScriptEnabled = false

        handleImageLoading()
        try {
            val gestureDetector =
                GestureDetector(
                    activity,
                    object : GestureDetector.SimpleOnGestureListener() {
                        override fun onSingleTapUp(e: MotionEvent): Boolean = performClick()
                    },
                )

            binding.webcontent.setOnTouchListener { _, event ->
                gestureDetector.onTouchEvent(
                    event,
                )
            }
        } catch (e: IllegalStateException) {
            e.sendSilentlyWithAcraWithName("Gesture detector issue ?")
            return
        }

        binding.webcontent.settings.layoutAlgorithm =
            WebSettings.LayoutAlgorithm.TEXT_AUTOSIZING

        var baseUrl: String? = null
        try {
            val itemUrl = URL(url.orEmpty())
            baseUrl = itemUrl.protocol + "://" + itemUrl.host
        } catch (e: MalformedURLException) {
            e.sendSilentlyWithAcraWithName("htmlToWebview > ${url.orEmpty()}")
        }

        val fontName: String =
            maybeIfContext {
                when (font) {
                    it.getString(R.string.open_sans_font_id) -> "Open Sans"
                    it.getString(R.string.roboto_font_id) -> "Roboto"
                    it.getString(R.string.source_code_pro_font_id) -> "Source Code Pro"
                    else -> ""
                }
            }?.toString().orEmpty()

        val fontLinkAndStyle =
            if (fontName.isNotEmpty()) {
                """<link href="https://fonts.googleapis.com/css?family=${
                    fontName.replace(
                        " ",
                        "+",
                    )
                }" rel="stylesheet">
                |<style>
                |   * {
                |       font-family: '$fontName';
                |   }
                |</style>
                """.trimMargin()
            } else {
                ""
            }
        try {
            binding.webcontent.loadDataWithBaseURL(
                baseUrl,
                """<html>
                |<head>
                |   <meta name="viewport" content="width=device-width, initial-scale=1">
                |   <style>
                |      img {
                |        display: inline-block;
                |        height: auto;
                |        width: 100%;
                |        max-width: 100%;
                |      }
                |      a {
                |        color: ${
                    String.format(
                        "#%06X",
                        WHITE_COLOR_HEX and (maybeIfContext { it.resources.getColor(R.color.colorAccent) } as Int),
                    )
                } !important;
                |      }
                |      *:not(a) {
                |        color: $colorOnSurfaceString;
                |      }
                |      * {
                |        font-size: ${fontSize}px;
                |        text-align: $textAlignment;
                |        word-break: break-word;
                |        overflow:hidden;
                |        line-height: 1.5em;
                |        background-color: $colorSurfaceString;
                |      }
                |      body, html {
                |        background-color: $colorSurfaceString !important;
                |        border-color: $colorSurfaceString  !important;
                |        padding: 0 !important;
                |        margin: 0 !important;
                |      }
                |      a, pre, code {
                |        text-align: $textAlignment;
                |      }
                |      pre, code {
                |        white-space: pre-wrap;
                |        width:100%;
                |        background-color: $colorSurfaceString;
                |      }
                |   </style>
                |   $fontLinkAndStyle
                |</head>
                |<body>
                |   $contentText
                |</body>
                """.trimMargin(),
                "text/html",
                "utf-8",
                null,
            )
        } catch (e: IllegalStateException) {
            e.sendSilentlyWithAcraWithName("Context required is still null ?")
        }
    }

    fun volumeButtonScrollDown() {
        val height = binding.nestedScrollView.measuredHeight
        binding.nestedScrollView.smoothScrollBy(0, height / 2)
    }

    fun volumeButtonScrollUp() {
        val height = binding.nestedScrollView.measuredHeight
        binding.nestedScrollView.smoothScrollBy(0, -height / 2)
    }

    private fun openInBrowserAfterFailing() {
        binding.progressBar.visibility = View.GONE
        maybeIfContext {
            it.openItemUrlInBrowserAsNewTask(this@ArticleFragment.item)
        }
    }

    companion object {
        private const val ARG_ITEMS = "items"

        fun newInstance(item: SelfossModel.Item): ArticleFragment {
            val fragment = ArticleFragment()
            val args = Bundle()
            args.putParcelable(ARG_ITEMS, item.toParcelable())
            fragment.arguments = args
            return fragment
        }
    }

    fun performClick(): Boolean {
        if (allImages != null &&
            (
                binding.webcontent.hitTestResult.type == WebView.HitTestResult.IMAGE_TYPE ||
                    binding.webcontent.hitTestResult.type == WebView.HitTestResult.SRC_IMAGE_ANCHOR_TYPE
            )
        ) {
            val position: Int = allImages.indexOf(binding.webcontent.hitTestResult.extra)

            val intent = Intent(activity, ImageActivity::class.java)
            intent.putExtra("allImages", allImages)
            intent.putExtra("position", position)
            startActivity(intent)
            return false
        }
        return false
    }
}
