package bou.amine.apps.readerforselfossv2.android

import android.app.NotificationChannel
import android.app.NotificationManager
import android.content.Context
import android.os.Build
import android.widget.Toast
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ProcessLifecycleOwner
import androidx.multidex.MultiDexApplication
import bou.amine.apps.readerforselfossv2.android.testing.TestingHelper
import bou.amine.apps.readerforselfossv2.dao.DriverFactory
import bou.amine.apps.readerforselfossv2.dao.ReaderForSelfossDB
import bou.amine.apps.readerforselfossv2.di.networkModule
import bou.amine.apps.readerforselfossv2.repository.Repository
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import bou.amine.apps.readerforselfossv2.service.ConnectivityService
import io.github.aakira.napier.DebugAntilog
import io.github.aakira.napier.Napier
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.acra.ACRA
import org.acra.ReportField
import org.acra.config.httpSender
import org.acra.config.toast
import org.acra.data.StringFormat
import org.acra.ktx.initAcra
import org.acra.sender.HttpSender
import org.kodein.di.DI
import org.kodein.di.DIAware
import org.kodein.di.bind
import org.kodein.di.instance
import org.kodein.di.singleton

class MyApp :
    MultiDexApplication(),
    DIAware {
    override val di by DI.lazy {
        bind<AppSettingsService>() with singleton { AppSettingsService(ACRA.isACRASenderServiceProcess() || TestingHelper().isUnitTest()) }
        import(networkModule)
        bind<DriverFactory>() with singleton { DriverFactory(applicationContext) }
        bind<ReaderForSelfossDB>() with singleton { ReaderForSelfossDB(driverFactory.createDriver()) }
        bind<ConnectivityService>() with singleton { ConnectivityService() }
        bind<Repository>() with
            singleton {
                Repository(
                    instance(),
                    instance(),
                    instance(),
                    instance(),
                )
            }
    }

    private val repository: Repository by instance()
    private val driverFactory: DriverFactory by instance()
    private val connectivityService: ConnectivityService by instance()

    override fun onCreate() {
        super.onCreate()
        Napier.base(DebugAntilog())

        if (!ACRA.isACRASenderServiceProcess()) {
            tryToHandleBug()

            handleNotificationChannels()

            ProcessLifecycleOwner.get().lifecycle.addObserver(
                AppLifeCycleObserver(
                    connectivityService,
                ),
            )

            CoroutineScope(Dispatchers.Default).launch {
                connectivityService.networkAvailableProvider.collect { networkAvailable ->
                    val toastMessage =
                        if (networkAvailable) {
                            repository.handleDBActions()
                            R.string.network_connectivity_retrieved
                        } else {
                            R.string.network_connectivity_lost
                        }
                    launch(Dispatchers.Main) {
                        Toast
                            .makeText(
                                applicationContext,
                                toastMessage,
                                Toast.LENGTH_SHORT,
                            ).show()
                    }
                }
            }
        }

        repository.migrate(driverFactory)
    }

    override fun attachBaseContext(base: Context?) {
        super.attachBaseContext(base)

        initAcra {
            sendReportsInDevMode = false
            reportFormat = StringFormat.JSON
            reportContent =
                listOf(
                    ReportField.REPORT_ID,
                    ReportField.INSTALLATION_ID,
                    ReportField.APP_VERSION_CODE,
                    ReportField.APP_VERSION_NAME,
                    ReportField.BUILD,
                    ReportField.ANDROID_VERSION,
                    ReportField.BRAND,
                    ReportField.PHONE_MODEL,
                    ReportField.AVAILABLE_MEM_SIZE,
                    ReportField.TOTAL_MEM_SIZE,
                    ReportField.STACK_TRACE,
                    ReportField.APPLICATION_LOG,
                    ReportField.LOGCAT,
                    ReportField.INITIAL_CONFIGURATION,
                    ReportField.CRASH_CONFIGURATION,
                    ReportField.IS_SILENT,
                    ReportField.USER_APP_START_DATE,
                    ReportField.USER_COMMENT,
                    ReportField.USER_CRASH_DATE,
                    ReportField.USER_EMAIL,
                    ReportField.CUSTOM_DATA,
                )
            toast {
                // required
                text = getString(R.string.crash_toast_text)
                length = Toast.LENGTH_SHORT
            }
            httpSender {
                uri =
                    "https://bugs.amine-bouabdallaoui.fr/report" // best guess, you may need to adjust this
                basicAuthLogin = "qMEscjj89Gwt6cPR"
                basicAuthPassword = "Yo58QFlGzFaWlBzP"
                httpMethod = HttpSender.Method.POST
            }
        }
    }

    private fun handleNotificationChannels() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val notificationManager = getSystemService(NOTIFICATION_SERVICE) as NotificationManager

            val name = getString(R.string.notification_channel_sync)
            val importance = NotificationManager.IMPORTANCE_LOW
            val mChannel = NotificationChannel(AppSettingsService.SYNC_CHANNEL_ID, name, importance)

            val newItemsChannelname = getString(R.string.new_items_channel_sync)
            val newItemsChannelimportance = NotificationManager.IMPORTANCE_DEFAULT
            val newItemsChannelmChannel =
                NotificationChannel(
                    AppSettingsService.NEW_ITEMS_CHANNEL,
                    newItemsChannelname,
                    newItemsChannelimportance,
                )

            notificationManager.createNotificationChannel(mChannel)
            notificationManager.createNotificationChannel(newItemsChannelmChannel)
        }
    }

    private fun tryToHandleBug() {
        val oldHandler = Thread.getDefaultUncaughtExceptionHandler()

        Thread.setDefaultUncaughtExceptionHandler { thread, e ->
            if (e is NoClassDefFoundError &&
                e.stackTrace.asList().any {
                    it.toString().contains("android.view.ViewDebug")
                }
            ) {
                // Nothing
            } else {
                oldHandler.uncaughtException(thread, e)
            }
        }
    }

    class AppLifeCycleObserver(
        val connectivityService: ConnectivityService,
    ) : DefaultLifecycleObserver {
        override fun onResume(owner: LifecycleOwner) {
            super.onResume(owner)
            connectivityService.start()
        }

        override fun onPause(owner: LifecycleOwner) {
            connectivityService.stop()
            super.onPause(owner)
        }
    }
}
