package bou.amine.apps.readerforselfossv2.android.adapters

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Toast
import androidx.appcompat.app.AlertDialog
import androidx.recyclerview.widget.RecyclerView
import bou.amine.apps.readerforselfossv2.android.R
import bou.amine.apps.readerforselfossv2.android.UpsertSourceActivity
import bou.amine.apps.readerforselfossv2.android.databinding.SourceListItemBinding
import bou.amine.apps.readerforselfossv2.android.testing.CountingIdlingResourceSingleton
import bou.amine.apps.readerforselfossv2.android.utils.glide.circularDrawable
import bou.amine.apps.readerforselfossv2.model.SelfossModel
import bou.amine.apps.readerforselfossv2.repository.Repository
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import bou.amine.apps.readerforselfossv2.utils.getHtmlDecoded
import bou.amine.apps.readerforselfossv2.utils.getIcon
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.kodein.di.DI
import org.kodein.di.DIAware
import org.kodein.di.android.closestDI
import org.kodein.di.instance

class SourcesListAdapter(
    private val app: Activity,
    private val items: ArrayList<SelfossModel.SourceDetail>,
) : RecyclerView.Adapter<SourcesListAdapter.ViewHolder>(),
    DIAware {
    override val di: DI by closestDI(app)

    override fun onCreateViewHolder(
        parent: ViewGroup,
        viewType: Int,
    ): ViewHolder {
        val binding = SourceListItemBinding.inflate(LayoutInflater.from(parent.context), parent, false)
        return ViewHolder(binding)
    }

    override fun onBindViewHolder(
        holder: ViewHolder,
        position: Int,
    ) {
        holder.bind(items[position], position)
    }

    override fun getItemId(position: Int) = position.toLong()

    override fun getItemViewType(position: Int) = position

    override fun getItemCount(): Int = items.size

    inner class ViewHolder(
        val binding: SourceListItemBinding,
    ) : RecyclerView.ViewHolder(binding.root) {
        private val context: Context = app.applicationContext
        private val repository: Repository by instance()
        private val appSettingsService: AppSettingsService by instance()

        fun bind(
            source: SelfossModel.SourceDetail,
            position: Int,
        ) {
            binding.apply {
                sourceTitle.text = source.title.getHtmlDecoded()
                if (source.getIcon(repository.baseUrl).isEmpty()) {
                    itemImage.setBackgroundAndText(source.title.getHtmlDecoded())
                } else {
                    context.circularDrawable(source.getIcon(repository.baseUrl), itemImage, appSettingsService)
                }

                errorText.apply {
                    visibility = if (!source.error.isNullOrBlank()) View.VISIBLE else View.GONE
                    text = source.error
                }

                deleteBtn.setOnClickListener { showDeleteConfirmationDialog(source, position) }

                root.setOnClickListener {
                    repository.setSelectedSource(source)
                    app.startActivity(Intent(app, UpsertSourceActivity::class.java))
                }
            }
        }

        private fun showDeleteConfirmationDialog(
            source: SelfossModel.SourceDetail,
            position: Int,
        ) {
            AlertDialog
                .Builder(app)
                .setTitle(app.getString(R.string.confirm_delete_title))
                .setMessage(app.getString(R.string.confirm_delete_message, source.title))
                .setPositiveButton(android.R.string.ok) { _, _ -> deleteSource(source, position) }
                .setNegativeButton(android.R.string.cancel, null)
                .show()
        }

        private fun deleteSource(
            source: SelfossModel.SourceDetail,
            position: Int,
        ) {
            CountingIdlingResourceSingleton.increment()
            CoroutineScope(Dispatchers.IO).launch {
                val successfullyDeletedSource = repository.deleteSource(source.id, source.title)
                CountingIdlingResourceSingleton.increment()
                launch(Dispatchers.Main) {
                    if (successfullyDeletedSource) {
                        items.removeAt(position)
                        notifyItemRemoved(position)
                        notifyItemRangeChanged(position, itemCount)
                    } else {
                        Toast
                            .makeText(
                                app,
                                R.string.can_delete_source,
                                Toast.LENGTH_SHORT,
                            ).show()
                    }
                    CountingIdlingResourceSingleton.decrement()
                }
                CountingIdlingResourceSingleton.decrement()
            }
        }
    }
}
