package bou.amine.apps.readerforselfossv2.android

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.text.TextUtils
import android.view.Menu
import android.view.MenuItem
import android.view.View
import android.view.inputmethod.EditorInfo
import android.widget.TextView
import android.widget.Toast
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate
import bou.amine.apps.readerforselfossv2.android.databinding.ActivityLoginBinding
import bou.amine.apps.readerforselfossv2.android.testing.CountingIdlingResourceSingleton
import bou.amine.apps.readerforselfossv2.android.utils.isBaseUrlInvalid
import bou.amine.apps.readerforselfossv2.repository.Repository
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import com.mikepenz.aboutlibraries.LibsBuilder
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.acra.ACRA
import org.kodein.di.DIAware
import org.kodein.di.android.closestDI
import org.kodein.di.instance

private const val MAX_INVALID_LOGIN_BEFORE_ALERT_DISPLAYED = 3

class LoginActivity :
    AppCompatActivity(),
    DIAware {
    private var inValidCount: Int = 0
    private var isWithLogin = false

    private lateinit var binding: ActivityLoginBinding

    override val di by closestDI()
    private val repository: Repository by instance()
    private val appSettingsService: AppSettingsService by instance()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        handleTheme()

        binding = ActivityLoginBinding.inflate(layoutInflater)
        val view = binding.root

        setContentView(view)

        setSupportActionBar(binding.toolbar)

        handleBaseUrlFail()

        if (appSettingsService.getBaseUrl().isNotEmpty()) {
            showProgress(true)
            goToMain()
        }

        handleActions()
    }

    @SuppressLint("WrongConstant") // Constant is fetched from the settings
    private fun handleTheme() {
        AppCompatDelegate.setDefaultNightMode(appSettingsService.getCurrentTheme())
    }

    private fun handleActions() {
        binding.passwordView.setOnEditorActionListener(
            TextView.OnEditorActionListener { _, id, _ ->
                if (id == R.id.loginView || id == EditorInfo.IME_NULL) {
                    attemptLogin()
                    return@OnEditorActionListener true
                }
                false
            },
        )

        binding.signInButton.setOnClickListener { attemptLogin() }

        binding.withLogin.setOnCheckedChangeListener { _, b ->
            isWithLogin = !isWithLogin
            val visi: Int = if (b) View.VISIBLE else View.GONE

            binding.loginView.visibility = visi
            binding.passwordView.visibility = visi
        }
    }

    private fun handleBaseUrlFail() {
        if (intent.getBooleanExtra("baseUrlFail", false)) {
            val alertDialog = AlertDialog.Builder(this).create()
            alertDialog.setTitle(getString(R.string.warning_wrong_url))
            alertDialog.setMessage(getString(R.string.base_url_error))
            alertDialog.setButton(
                AlertDialog.BUTTON_NEUTRAL,
                "OK",
            ) { dialog, _ -> dialog.dismiss() }
            alertDialog.show()
        }
    }

    private fun goToMain() {
        CountingIdlingResourceSingleton.increment()
        CoroutineScope(Dispatchers.IO).launch {
            repository.updateApiInformation()
            ACRA.errorReporter.putCustomData(
                "SELFOSS_API_VERSION",
                appSettingsService.getApiVersion().toString(),
            )
            CountingIdlingResourceSingleton.decrement()
        }
        val intent = Intent(this, HomeActivity::class.java)
        startActivity(intent)
        finish()
    }

    private fun preferenceError() {
        appSettingsService.resetLoginInformation()

        binding.urlView.error = getString(R.string.wrong_infos)
        binding.loginView.error = getString(R.string.wrong_infos)
        binding.passwordView.error = getString(R.string.wrong_infos)
        binding.urlView.requestFocus()

        showProgress(false)
    }

    @Suppress("detekt:LongMethod")
    private fun attemptLogin() {
        // Reset errors.
        binding.urlView.error = null
        binding.loginView.error = null
        binding.passwordView.error = null

        // Store values at the time of the login attempt.
        val url =
            binding.urlView.text
                .toString()
                .trim()
        val login =
            binding.loginView.text
                .toString()
                .trim()
        val password =
            binding.passwordView.text
                .toString()
                .trim()

        val cancelUrl = failInvalidUrl(url)
        if (cancelUrl) return
        val cancelDetails = failLoginDetails(password, login)
        if (cancelDetails) return
        showProgress(true)

        appSettingsService.updateSelfSigned(binding.selfSigned.isChecked)

        repository.refreshLoginInformation(url, login, password)

        CountingIdlingResourceSingleton.increment()
        CoroutineScope(Dispatchers.IO).launch {
            try {
                repository.updateApiInformation()
                val result = repository.login()
                CountingIdlingResourceSingleton.increment()
                launch(Dispatchers.Main) {
                    if (result) {
                        val errorFetching = repository.checkIfFetchFails()
                        if (!errorFetching) {
                            goToMain()
                        } else {
                            preferenceError()
                        }
                    } else {
                        preferenceError()
                    }
                    CountingIdlingResourceSingleton.decrement()
                }
            } catch (e: Exception) {
                CountingIdlingResourceSingleton.increment()
                launch(Dispatchers.Main) {
                    if (e.message?.startsWith("No transformation found") == true) {
                        Toast
                            .makeText(
                                applicationContext,
                                R.string.application_selfoss_only,
                                Toast.LENGTH_LONG,
                            ).show()
                        preferenceError()
                    }
                    CountingIdlingResourceSingleton.decrement()
                }
            } finally {
                CountingIdlingResourceSingleton.decrement()
            }
        }
    }

    private fun failLoginDetails(
        password: String,
        login: String,
    ): Boolean {
        var lastFocusedView: View? = null
        var cancel = false
        if (isWithLogin) {
            if (TextUtils.isEmpty(password)) {
                binding.passwordView.error = getString(R.string.error_invalid_password)
                lastFocusedView = binding.passwordView
                cancel = true
            }

            if (TextUtils.isEmpty(login)) {
                binding.loginView.error = getString(R.string.error_field_required)
                lastFocusedView = binding.loginView
                cancel = true
            }
        }
        maybeCancelAndFocusView(cancel, lastFocusedView)
        return cancel
    }

    private fun failInvalidUrl(url: String): Boolean {
        val focusView = binding.urlView
        var cancel = false
        if (url.isBaseUrlInvalid()) {
            cancel = true
            binding.urlView.error = getString(R.string.login_url_problem)
            inValidCount++
            if (inValidCount == MAX_INVALID_LOGIN_BEFORE_ALERT_DISPLAYED) {
                val alertDialog = AlertDialog.Builder(this).create()
                alertDialog.setTitle(getString(R.string.warning_wrong_url))
                alertDialog.setMessage(getString(R.string.text_wrong_url))
                alertDialog.setButton(
                    AlertDialog.BUTTON_NEUTRAL,
                    "OK",
                ) { dialog, _ -> dialog.dismiss() }
                alertDialog.show()
                inValidCount = 0
            }
        }
        maybeCancelAndFocusView(cancel, focusView)
        return cancel
    }

    private fun maybeCancelAndFocusView(
        cancel: Boolean,
        focusView: View?,
    ) {
        if (cancel) {
            focusView?.requestFocus()
        }
    }

    private fun showProgress(show: Boolean) {
        val shortAnimTime = resources.getInteger(android.R.integer.config_shortAnimTime)

        binding.loginForm.visibility = if (show) View.GONE else View.VISIBLE
        binding.loginForm
            .animate()
            .setDuration(shortAnimTime.toLong())
            .alpha(
                if (show) 0F else 1F,
            ).setListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        binding.loginForm.visibility = if (show) View.GONE else View.VISIBLE
                    }
                },
            )

        binding.loginProgress.visibility = if (show) View.VISIBLE else View.GONE
        binding.loginProgress
            .animate()
            .setDuration(shortAnimTime.toLong())
            .alpha(
                if (show) 1F else 0F,
            ).setListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        binding.loginProgress.visibility = if (show) View.VISIBLE else View.GONE
                    }
                },
            )
    }

    override fun onCreateOptionsMenu(menu: Menu): Boolean {
        menuInflater.inflate(R.menu.login_menu, menu)
        return true
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return when (item.itemId) {
            R.id.issue_tracker -> {
                val browserIntent =
                    Intent(Intent.ACTION_VIEW, Uri.parse(AppSettingsService.BUG_URL))
                startActivity(browserIntent)
                return true
            }

            R.id.about -> {
                LibsBuilder()
                    .withAboutIconShown(true)
                    .withAboutVersionShown(true)
                    .withAboutSpecial2("Bug reports")
                    .withAboutSpecial2Description(AppSettingsService.BUG_URL)
                    .withAboutSpecial1("Project Page")
                    .withAboutSpecial1Description(AppSettingsService.SOURCE_URL)
                    .withShowLoadingProgress(false)
                    .start(this)
                true
            }

            else -> super.onOptionsItemSelected(item)
        }
    }
}
