package bou.amine.apps.readerforselfossv2.android

import android.content.Context
import android.view.View
import android.widget.EditText
import android.widget.ImageView
import android.widget.RelativeLayout
import androidx.annotation.DrawableRes
import androidx.annotation.StringRes
import androidx.core.graphics.drawable.toBitmap
import androidx.core.view.isVisible
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout
import androidx.test.core.app.ApplicationProvider
import androidx.test.espresso.Espresso.openActionBarOverflowOrOptionsMenu
import androidx.test.espresso.PerformException
import androidx.test.espresso.Root
import androidx.test.espresso.UiController
import androidx.test.espresso.ViewAction
import androidx.test.espresso.matcher.RootMatchers.isPlatformPopup
import androidx.test.espresso.matcher.ViewMatchers.hasSibling
import androidx.test.espresso.matcher.ViewMatchers.isRoot
import androidx.test.espresso.matcher.ViewMatchers.withChild
import androidx.test.espresso.matcher.ViewMatchers.withClassName
import androidx.test.espresso.matcher.ViewMatchers.withId
import androidx.test.espresso.matcher.ViewMatchers.withParent
import androidx.test.espresso.matcher.ViewMatchers.withResourceName
import androidx.test.espresso.matcher.ViewMatchers.withText
import androidx.test.espresso.util.HumanReadables
import androidx.test.espresso.util.TreeIterables
import org.hamcrest.CoreMatchers.allOf
import org.hamcrest.CoreMatchers.any
import org.hamcrest.Description
import org.hamcrest.Matcher
import org.hamcrest.Matchers
import org.hamcrest.TypeSafeMatcher
import java.util.concurrent.TimeoutException

fun withError(
    @StringRes id: Int,
): TypeSafeMatcher<View?> {
    return object : TypeSafeMatcher<View?>() {
        override fun matchesSafely(view: View?): Boolean {
            if (view != null && (view !is EditText || view.error == null)) {
                return false
            }
            val context = view!!.context

            return (view as EditText).error.toString() == context.getString(id)
        }

        override fun describeTo(description: Description?) {
            // Nothing
        }
    }
}

fun waitUntilShown(
    viewText: String,
    millis: Long,
): ViewAction {
    return object : ViewAction {
        override fun getConstraints(): Matcher<View> = isRoot()

        override fun getDescription(): String = "wait for $millis millis, for a specific view with text <$viewText> to be visible."

        override fun perform(
            uiController: UiController,
            view: View,
        ) {
            uiController.loopMainThreadUntilIdle()
            val startTime = System.currentTimeMillis()
            val endTime = startTime + millis
            val viewMatcher = withText(viewText)

            do {
                for (child in TreeIterables.breadthFirstViewTraversal(view)) {
                    if (viewMatcher.matches(child) && child.isShown) {
                        return
                    }
                }

                uiController.loopMainThreadForAtLeast(100)
            } while (System.currentTimeMillis() < endTime)

            // timeout happens
            throw PerformException
                .Builder()
                .withActionDescription(this.description)
                .withViewDescription(HumanReadables.describe(view))
                .withCause(TimeoutException())
                .build()
        }
    }
}

fun waitForRecyclerViewToStopLoading(millis: Long): ViewAction {
    return object : ViewAction {
        override fun getConstraints(): Matcher<View> = any(View::class.java)

        override fun getDescription(): String = "wait for  $millis millis for the recyclerview to stop loading."

        override fun perform(
            uiController: UiController,
            view: View?,
        ) {
            uiController.loopMainThreadUntilIdle()
            val startTime = System.currentTimeMillis()
            val endTime = startTime + millis

            do {
                // either the empty view is displayed
                for (child in TreeIterables.breadthFirstViewTraversal(view)) {
                    // found view with required ID
                    if (withId(R.id.emptyText).matches(child) && child.isVisible) {
                        return
                    }
                }

                // or the refresh layout is refreshing
                if (view is SwipeRefreshLayout && !view.isRefreshing) {
                    return
                }
                uiController.loopMainThreadForAtLeast(100)
            } while (System.currentTimeMillis() < endTime)

            // timeout happens
            throw PerformException
                .Builder()
                .withActionDescription(this.description)
                .withViewDescription(HumanReadables.describe(view))
                .withCause(TimeoutException())
                .build()
        }
    }
}

fun isPopupWindow(): Matcher<Root> = isPlatformPopup()

fun withDrawable(
    @DrawableRes id: Int,
) = object : TypeSafeMatcher<View>() {
    override fun describeTo(description: Description) {
        description.appendText("ImageView with drawable same as drawable with id $id")
    }

    @Suppress("detekt:SwallowedException")
    override fun matchesSafely(view: View): Boolean {
        val context = view.context
        val expectedBitmap = context.getDrawable(id)!!.toBitmap()
        try {
            return view is ImageView && view.drawable.toBitmap().sameAs(expectedBitmap)
        } catch (e: Exception) {
            return false
        }
    }
}

fun hasBottombarItemText(
    @StringRes id: Int,
): Matcher<View>? =
    allOf(
        withResourceName("fixed_bottom_navigation_icon"),
        withParent(
            allOf(
                withResourceName("fixed_bottom_navigation_icon_container"),
                hasSibling(withText(id)),
            ),
        ),
    )

fun withSettingsCheckboxWidget(
    @StringRes id: Int,
): Matcher<View>? =
    allOf(
        withId(android.R.id.switch_widget),
        withParent(
            withSettingsCheckboxFrame(id),
        ),
    )

fun withSettingsCheckboxFrame(
    @StringRes id: Int,
): Matcher<View>? =
    allOf(
        withId(android.R.id.widget_frame),
        hasSibling(
            allOf(
                withClassName(Matchers.equalTo(RelativeLayout::class.java.name)),
                withChild(
                    withText(id),
                ),
            ),
        ),
    )

fun openMenu() {
    openActionBarOverflowOrOptionsMenu(
        ApplicationProvider.getApplicationContext<Context>(),
    )
}
