package bou.amine.apps.readerforselfossv2.android

import android.content.Context
import android.graphics.Bitmap
import android.os.Environment.DIRECTORY_PICTURES
import android.os.Environment.getExternalStoragePublicDirectory
import android.util.Log
import androidx.annotation.ArrayRes
import androidx.test.espresso.Espresso
import androidx.test.espresso.Espresso.onData
import androidx.test.espresso.Espresso.onView
import androidx.test.espresso.action.ViewActions.click
import androidx.test.espresso.action.ViewActions.replaceText
import androidx.test.espresso.action.ViewActions.typeTextIntoFocusedView
import androidx.test.espresso.assertion.ViewAssertions.doesNotExist
import androidx.test.espresso.assertion.ViewAssertions.matches
import androidx.test.espresso.base.DefaultFailureHandler
import androidx.test.espresso.matcher.RootMatchers.isDialog
import androidx.test.espresso.matcher.ViewMatchers.isChecked
import androidx.test.espresso.matcher.ViewMatchers.isDisplayed
import androidx.test.espresso.matcher.ViewMatchers.isNotChecked
import androidx.test.espresso.matcher.ViewMatchers.withId
import androidx.test.espresso.matcher.ViewMatchers.withText
import androidx.test.platform.app.InstrumentationRegistry.getInstrumentation
import androidx.test.uiautomator.UiDevice
import androidx.test.uiautomator.UiSelector
import org.hamcrest.CoreMatchers.allOf
import org.hamcrest.CoreMatchers.not
import org.hamcrest.Matchers.hasToString
import org.junit.BeforeClass
import java.io.BufferedOutputStream
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.util.Locale

// For now, do not move this as it is modified by the integration tests
const val DEFAULT_URL = "http://10.0.2.2:8888"

fun performLogin(someUrl: String? = null) {
    Log.i("AUTOMATION", "The url used will be ${if (!someUrl.isNullOrEmpty()) someUrl else DEFAULT_URL}")
    onView(withId(R.id.urlView)).perform(click()).perform(
        typeTextIntoFocusedView(
            if (!someUrl.isNullOrEmpty()) someUrl else DEFAULT_URL,
        ),
    )
    onView(withId(R.id.signInButton)).perform(click())
}

fun changeAndCancelSetting(
    oldValue: String,
    newValue: String,
    openSettingItem: () -> Unit,
) {
    openSettingItem()
    onView(
        withId(android.R.id.edit),
    ).perform(replaceText(newValue))
    onView(
        withId(android.R.id.button2),
    ).perform(click())
    openSettingItem()
    onView(
        withId(android.R.id.edit),
    ).check(matches(withText(oldValue)))
    onView(
        withText(newValue),
    ).check(doesNotExist())
    onView(
        withId(android.R.id.button2),
    ).perform(click())
}

fun changeAndSaveSetting(
    oldValue: String,
    newValue: String,
    openSettingItem: () -> Unit,
) {
    openSettingItem()
    onView(
        withId(android.R.id.edit),
    ).perform(replaceText(newValue))
    onView(
        withId(android.R.id.button1),
    ).perform(click())
    openSettingItem()
    onView(
        withId(android.R.id.edit),
    ).check(matches(withText(newValue)))
    if (oldValue.isNotEmpty()) {
        onView(
            withText(oldValue),
        ).check(doesNotExist())
    }
    onView(
        withId(android.R.id.button2),
    ).perform(click())
}

fun testPreferencesFromArray(
    context: Context,
    @ArrayRes arrayRes: Int,
    openSettingItem: () -> Unit,
) {
    openSettingItem()
    context.resources.getStringArray(arrayRes).forEach { res ->
        onView(withText(res)).check(matches(allOf(isDisplayed(), isNotChecked())))
        onView(withText(res)).perform(click())
        onView(withText(res)).check(doesNotExist())
        openSettingItem()
        onView(withText(res)).check(matches(allOf(isDisplayed(), isChecked())))
    }
}

fun goToSources() {
    openMenu()
    onView(withText(R.string.menu_home_sources))
        .perform(click())
}

fun testAddSourceWithUrl(
    url: String,
    sourceName: String,
) {
    onView(withId(R.id.fab))
        .perform(click())
    onView(withId(R.id.nameInput))
        .perform(click())
        .perform(typeTextIntoFocusedView(sourceName))
    onView(withId(R.id.sourceUri))
        .perform(click())
        .perform(typeTextIntoFocusedView(url))
    onView(withId(R.id.tags))
        .perform(click())
        .perform(typeTextIntoFocusedView("tag1,tag2,tag3"))
    onView(withId(R.id.spoutsSpinner))
        .perform(click())
    onData(hasToString("RSS Feed")).perform(click())
    onView(withId(R.id.saveBtn))
        .perform(click())
    onView(withText(sourceName)).check(matches(isDisplayed()))
}

fun checkHomeLoadingDone() {
    onView(withId(R.id.swipeRefreshLayout)).inRoot(not(isDialog())).perform(waitForRecyclerViewToStopLoading(300000))
}

@Suppress("detekt:UtilityClassWithPublicConstructor")
open class WithANRException {
    companion object {
        // Running count of the number of Android Not Responding dialogues to prevent endless dismissal.
        private var anrCount = 0

        // `RootViewWithoutFocusException` class is private, need to match the message (instead of using type matching).
        private val rootViewWithoutFocusExceptionMsg =
            java.lang.String.format(
                Locale.ROOT,
                "Waited for the root of the view hierarchy to have " +
                    "window focus and not request layout for 10 seconds. If you specified a non " +
                    "default root matcher, it may be picking a root that never takes focus. " +
                    "Root:",
            )
        private const val OTHER_EXCEPTION = "System Ul isn't responding"

        private fun handleAnrDialogue() {
            val device = UiDevice.getInstance(getInstrumentation())
            // If running the device in English Locale
            val waitButton = device.findObject(UiSelector().textContains("wait"))
            if (waitButton.exists()) waitButton.click()
        }

        @JvmStatic
        @BeforeClass
        fun setUpHandler() {
            Espresso.setFailureHandler { error, viewMatcher ->

                takeScreenshot()
                if (error.message!!.contains(OTHER_EXCEPTION)) {
                    handleAnrDialogue()
                } else if (error.message!!.contains(rootViewWithoutFocusExceptionMsg) &&
                    anrCount < 20
                ) {
                    anrCount++
                    handleAnrDialogue()
                } else { // chain all failures down to the default espresso handler
                    Log.e("AMINE", "AMINE : ${error.message}")
                    println("AMINE : ${error.message}")
                    DefaultFailureHandler(getInstrumentation().targetContext).handle(error, viewMatcher)
                }
            }
        }
    }
}

@Suppress("detekt:NestedBlockDepth")
fun takeScreenshot() {
    try {
        val bitmap = getInstrumentation().uiAutomation.takeScreenshot()

        val folder =
            File(
                File(
                    getExternalStoragePublicDirectory(DIRECTORY_PICTURES),
                    "selfoss_tests",
                ).absolutePath,
                "screenshots",
            )
        if (!folder.exists()) {
            folder.mkdirs()
        }

        var out: BufferedOutputStream? = null
        val size = folder.list().size + 1
        try {
            out = BufferedOutputStream(FileOutputStream(folder.path + "/" + size + ".png"))
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, out)
            Log.d("Screenshots", "Screenshot taken")
        } catch (e: IOException) {
            Log.e("Screenshots", "Could not save the screenshot", e)
        } finally {
            if (out != null) {
                try {
                    out.close()
                } catch (e: IOException) {
                    Log.e("Screenshots", "Could not save the screenshot", e)
                }
            }
        }
    } catch (ex: IOException) {
        Log.e("Screenshots", "Could not take the screenshot", ex)
    }
}
