@file:Suppress("detekt:TooManyFunctions")

package bou.amine.apps.readerforselfossv2.service

import com.russhwolf.settings.Settings

private const val DEFAULT_FONT_SIZE = 16

private const val DEFAULT_REFRESH_MINUTES = 360L

private const val MIN_REFRESH_MINUTES = 15L

private const val DEFAULT_API_TIMEOUT = 60L

private const val DEFAULT_ITEMS_NUMBER = 20

class AppSettingsService(
    acraSenderServiceProcess: Boolean = false,
) {
    val settings: Settings =
        if (acraSenderServiceProcess) {
            ACRASettings()
        } else {
            Settings()
        }

    // Api related
    private var apiVersion: Int = -1
    private var publicAccess: Boolean? = null
    private var selfSigned: Boolean? = null
    private var baseUrl: String = ""
    private var userName: String = ""
    private var basicUserName: String = ""
    private var password: String = ""
    private var basicPassword: String = ""

    // User settings related
    private var itemsCaching: Boolean? = null
    private var articleViewer: Boolean? = null
    private var shouldBeCardView: Boolean? = null
    private var displayUnreadCount: Boolean? = null
    private var displayAllCount: Boolean? = null
    private var fullHeightCards: Boolean? = null
    private var updateSources: Boolean? = null
    private var periodicRefresh: Boolean? = null
    private var refreshWhenChargingOnly: Boolean? = null
    private var infiniteLoading: Boolean? = null
    private var notifyNewItems: Boolean? = null
    private var itemsNumber: Int? = null
    private var apiTimeout: Long? = null
    private var refreshMinutes: Long = DEFAULT_REFRESH_MINUTES
    private var markOnScroll: Boolean? = null
    private var activeAlignment: Int? = null

    private var fontSize: Int? = null
    private var font: String = ""
    private var theme: Int? = null

    init {
        refreshApiSettings()
        refreshUserSettings()
    }

    fun getApiVersion(): Int {
        if (apiVersion == -1) {
            refreshApiVersion()
            return apiVersion
        }
        return apiVersion
    }

    fun updateApiVersion(apiMajorVersion: Int) {
        settings.putInt(API_VERSION_MAJOR, apiMajorVersion)
        refreshApiVersion()
    }

    private fun refreshApiVersion() {
        apiVersion = settings.getInt(API_VERSION_MAJOR, -1)
    }

    fun getPublicAccess(): Boolean {
        if (publicAccess == null) {
            refreshPublicAccess()
        }
        return publicAccess!!
    }

    fun updatePublicAccess(publicAccess: Boolean) {
        settings.putBoolean(API_PUBLIC_ACCESS, publicAccess)
        refreshPublicAccess()
    }

    private fun refreshPublicAccess() {
        publicAccess = settings.getBoolean(API_PUBLIC_ACCESS, false)
    }

    fun getSelfSigned(): Boolean {
        if (selfSigned == null) {
            refreshSelfSigned()
        }
        return selfSigned!!
    }

    fun updateSelfSigned(selfSigned: Boolean) {
        settings.putBoolean(API_SELF_SIGNED, selfSigned)
        refreshSelfSigned()
    }

    private fun refreshSelfSigned() {
        selfSigned = settings.getBoolean(API_SELF_SIGNED, false)
    }

    fun getBaseUrl(): String {
        if (baseUrl.isEmpty()) {
            refreshBaseUrl()
        }
        return baseUrl
    }

    fun getUserName(): String {
        if (userName.isEmpty()) {
            refreshUsername()
        }
        return userName
    }

    fun getPassword(): String {
        if (password.isEmpty()) {
            refreshPassword()
        }
        return password
    }

    fun getBasicUserName(): String {
        if (basicUserName.isEmpty()) {
            refreshBasicUsername()
        }
        return basicUserName
    }

    fun getBasicPassword(): String {
        if (basicPassword.isEmpty()) {
            refreshBasicPassword()
        }
        return basicPassword
    }

    fun getItemsNumber(): Int {
        if (itemsNumber == null) {
            refreshItemsNumber()
        }
        return itemsNumber!!
    }

    @Suppress("detekt:SwallowedException")
    private fun refreshItemsNumber() {
        itemsNumber =
            try {
                settings.getString(API_ITEMS_NUMBER, DEFAULT_ITEMS_NUMBER.toString()).toInt()
            } catch (e: Exception) {
                settings.remove(API_ITEMS_NUMBER)
                DEFAULT_ITEMS_NUMBER
            }
    }

    fun getApiTimeout(): Long {
        if (apiTimeout == null) {
            refreshApiTimeout()
        }
        return apiTimeout!!
    }

    @Suppress("detekt:MagicNumber")
    private fun secToMs(n: Long) = n * 1000

    @Suppress("detekt:SwallowedException")
    private fun refreshApiTimeout() {
        apiTimeout =
            secToMs(
                try {
                    val settingsTimeout = settings.getString(API_TIMEOUT, DEFAULT_API_TIMEOUT.toString())
                    if (settingsTimeout.toLong() > 0) {
                        settingsTimeout.toLong()
                    } else {
                        settings.remove(API_TIMEOUT)
                        DEFAULT_API_TIMEOUT
                    }
                } catch (e: Exception) {
                    settings.remove(API_TIMEOUT)
                    DEFAULT_API_TIMEOUT
                },
            )
    }

    private fun refreshBaseUrl() {
        baseUrl = settings.getString(BASE_URL, "")
    }

    private fun refreshUsername() {
        userName = settings.getString(LOGIN, "")
    }

    private fun refreshPassword() {
        password = settings.getString(PASSWORD, "")
    }

    private fun refreshBasicUsername() {
        basicUserName = settings.getString(BASIC_LOGIN, "")
    }

    private fun refreshBasicPassword() {
        basicPassword = settings.getString(BASIC_PASSWORD, "")
    }

    private fun refreshArticleViewerEnabled() {
        articleViewer = settings.getBoolean(PREFER_ARTICLE_VIEWER, true)
    }

    fun isArticleViewerEnabled(): Boolean {
        if (articleViewer != null) {
            refreshArticleViewerEnabled()
        }
        return articleViewer == true
    }

    private fun refreshShouldBeCardViewEnabled() {
        shouldBeCardView = settings.getBoolean(CARD_VIEW_ACTIVE, false)
    }

    fun isCardViewEnabled(): Boolean {
        if (shouldBeCardView != null) {
            refreshShouldBeCardViewEnabled()
        }
        return shouldBeCardView == true
    }

    private fun refreshDisplayUnreadCountEnabled() {
        displayUnreadCount = settings.getBoolean(DISPLAY_UNREAD_COUNT, true)
    }

    fun isDisplayUnreadCountEnabled(): Boolean {
        if (displayUnreadCount != null) {
            refreshDisplayUnreadCountEnabled()
        }
        return displayUnreadCount == true
    }

    private fun refreshDisplayAllCountEnabled() {
        displayAllCount = settings.getBoolean(DISPLAY_OTHER_COUNT, false)
    }

    fun isDisplayAllCountEnabled(): Boolean {
        if (displayAllCount != null) {
            refreshDisplayAllCountEnabled()
        }
        return displayAllCount == true
    }

    private fun refreshFullHeightCardsEnabled() {
        fullHeightCards = settings.getBoolean(FULL_HEIGHT_CARDS, false)
    }

    fun isFullHeightCardsEnabled(): Boolean {
        if (fullHeightCards != null) {
            refreshFullHeightCardsEnabled()
        }
        return fullHeightCards == true
    }

    private fun refreshUpdateSourcesEnabled() {
        updateSources = settings.getBoolean(UPDATE_SOURCES, true)
    }

    fun isUpdateSourcesEnabled(): Boolean {
        if (updateSources != null) {
            refreshUpdateSourcesEnabled()
        }
        return updateSources == true
    }

    private fun refreshPeriodicRefreshEnabled() {
        periodicRefresh = settings.getBoolean(PERIODIC_REFRESH, false)
    }

    fun isPeriodicRefreshEnabled(): Boolean {
        if (periodicRefresh != null) {
            refreshPeriodicRefreshEnabled()
        }
        return periodicRefresh == true
    }

    private fun refreshRefreshWhenChargingOnlyEnabled() {
        refreshWhenChargingOnly = settings.getBoolean(REFRESH_WHEN_CHARGING, false)
    }

    fun isRefreshWhenChargingOnlyEnabled(): Boolean {
        if (refreshWhenChargingOnly != null) {
            refreshRefreshWhenChargingOnlyEnabled()
        }
        return refreshWhenChargingOnly == true
    }

    private fun refreshRefreshMinutes() {
        refreshMinutes = settings.getString(PERIODIC_REFRESH_MINUTES, DEFAULT_REFRESH_MINUTES.toString()).toLong()
        if (refreshMinutes <= MIN_REFRESH_MINUTES) {
            refreshMinutes = MIN_REFRESH_MINUTES
        }
    }

    fun getRefreshMinutes(): Long {
        if (refreshMinutes != DEFAULT_REFRESH_MINUTES) {
            refreshRefreshMinutes()
        }
        return refreshMinutes
    }

    private fun refreshInfiniteLoadingEnabled() {
        infiniteLoading = settings.getBoolean(INFINITE_LOADING, false)
    }

    fun isInfiniteLoadingEnabled(): Boolean {
        if (infiniteLoading != null) {
            refreshInfiniteLoadingEnabled()
        }
        return infiniteLoading == true
    }

    private fun refreshItemCachingEnabled() {
        itemsCaching = settings.getBoolean(ITEMS_CACHING, false)
    }

    fun isItemCachingEnabled(): Boolean {
        if (itemsCaching != null) {
            refreshItemCachingEnabled()
        }
        return itemsCaching == true
    }

    private fun refreshNotifyNewItemsEnabled() {
        notifyNewItems = settings.getBoolean(NOTIFY_NEW_ITEMS, false)
    }

    fun isNotifyNewItemsEnabled(): Boolean {
        if (notifyNewItems != null) {
            refreshNotifyNewItemsEnabled()
        }
        return notifyNewItems == true
    }

    private fun refreshMarkOnScrollEnabled() {
        markOnScroll = settings.getBoolean(MARK_ON_SCROLL, false)
    }

    fun isMarkOnScrollEnabled(): Boolean {
        if (markOnScroll != null) {
            refreshMarkOnScrollEnabled()
        }
        return markOnScroll == true
    }

    private fun refreshActiveAllignment() {
        activeAlignment = settings.getInt(TEXT_ALIGN, JUSTIFY)
    }

    fun getActiveAllignment(): Int {
        if (activeAlignment != null) {
            refreshActiveAllignment()
        }
        return activeAlignment ?: JUSTIFY
    }

    fun changeAllignment(allignment: Int) {
        settings.putInt(TEXT_ALIGN, allignment)
        activeAlignment = allignment
    }

    private fun refreshFontSize() {
        fontSize = settings.getString(READER_FONT_SIZE, "16").toInt()
    }

    fun getFontSize(): Int {
        if (fontSize != null) {
            refreshFontSize()
        }
        return fontSize ?: DEFAULT_FONT_SIZE
    }

    private fun refreshFont() {
        font = settings.getString(READER_FONT, "")
    }

    fun getFont(): String {
        if (font.isEmpty()) {
            refreshFont()
        }
        return font
    }

    private fun refreshCurrentTheme() {
        theme = settings.getString(CURRENT_THEME, "-1").toInt()
    }

    fun getCurrentTheme(): Int {
        if (theme == null) {
            refreshCurrentTheme()
        }
        return theme ?: -1
    }

    fun refreshApiSettings() {
        refreshPassword()
        refreshUsername()
        refreshBasicUsername()
        refreshBasicPassword()
        refreshBaseUrl()
        refreshApiVersion()
        refreshPublicAccess()
        refreshSelfSigned()
    }

    fun refreshUserSettings() {
        refreshItemsNumber()
        refreshApiTimeout()
        refreshArticleViewerEnabled()
        refreshShouldBeCardViewEnabled()
        refreshDisplayUnreadCountEnabled()
        refreshDisplayAllCountEnabled()
        refreshFullHeightCardsEnabled()
        refreshUpdateSourcesEnabled()
        refreshPeriodicRefreshEnabled()
        refreshRefreshWhenChargingOnlyEnabled()
        refreshRefreshMinutes()
        refreshInfiniteLoadingEnabled()
        refreshItemCachingEnabled()
        refreshNotifyNewItemsEnabled()
        refreshMarkOnScrollEnabled()
        refreshActiveAllignment()
        refreshFontSize()
        refreshFont()
        refreshCurrentTheme()
    }

    fun refreshLoginInformation(
        url: String,
        login: String,
        password: String,
    ) {
        val regex = """\/\/(\S+):(\S+)@""".toRegex()
        val matchResult = regex.find(url)
        if (matchResult != null) {
            val (basicLogin, basicPassword) = matchResult.destructured
            settings.putString(BASIC_LOGIN, basicLogin)
            settings.putString(BASIC_PASSWORD, basicPassword)
            val urlWithoutBasicAuth = url.replace(regex, "//")
            settings.putString(BASE_URL, urlWithoutBasicAuth)
        } else {
            settings.putString(BASE_URL, url)
        }
        settings.putString(LOGIN, login)
        settings.putString(PASSWORD, password)
        refreshApiSettings()
    }

    fun resetLoginInformation() {
        settings.remove(BASE_URL)
        settings.remove(LOGIN)
        settings.remove(PASSWORD)
        settings.remove(BASIC_LOGIN)
        settings.remove(BASIC_PASSWORD)
        refreshApiSettings()
    }

    fun clearAll() {
        settings.clear()
        refreshApiSettings()
        refreshUserSettings()
    }

    fun disableArticleViewer() {
        settings.putBoolean(PREFER_ARTICLE_VIEWER, false)
        refreshArticleViewerEnabled()
    }

    companion object {
        const val TRANSLATION_URL = "https://crwd.in/readerforselfoss"

        const val SOURCE_URL = "https://gitea.amine-bouabdallaoui.fr/Louvorg/ReaderForSelfoss-multiplatform"

        const val BUG_URL = "https://gitea.amine-bouabdallaoui.fr/Louvorg/ReaderForSelfoss-multiplatform/issues"

        const val SYNC_CHANNEL_ID = "sync-channel-id"

        const val NEW_ITEMS_CHANNEL = "new-items-channel-id"

        const val JUSTIFY = 1

        const val ALIGN_LEFT = 2

        const val API_VERSION_MAJOR = "apiVersionMajor"

        const val API_PUBLIC_ACCESS = "apiPublicAccess"

        const val API_SELF_SIGNED = "apiSelfSigned"

        const val API_ITEMS_NUMBER = "prefer_api_items_number"

        const val API_TIMEOUT = "api_timeout"

        const val BASE_URL = "url"

        const val LOGIN = "login"

        const val PASSWORD = "password"

        const val BASIC_LOGIN = "basic_login"

        const val BASIC_PASSWORD = "basic_password"

        const val PREFER_ARTICLE_VIEWER = "prefer_article_viewer"

        const val CARD_VIEW_ACTIVE = "card_view_active"

        const val DISPLAY_UNREAD_COUNT = "display_unread_count"

        const val DISPLAY_OTHER_COUNT = "display_other_count"

        const val FULL_HEIGHT_CARDS = "full_height_cards"

        const val UPDATE_SOURCES = "update_sources"

        const val PERIODIC_REFRESH = "periodic_refresh"

        const val REFRESH_WHEN_CHARGING = "refresh_when_charging"

        const val READER_FONT = "reader_font"

        const val READER_FONT_SIZE = "reader_font_size"

        const val TEXT_ALIGN = "text_align"

        const val MARK_ON_SCROLL = "mark_on_scroll"

        const val NOTIFY_NEW_ITEMS = "notify_new_items"

        const val PERIODIC_REFRESH_MINUTES = "periodic_refresh_minutes"

        const val INFINITE_LOADING = "infinite_loading"

        const val ITEMS_CACHING = "items_caching"

        const val CURRENT_THEME = "currentMode"
    }
}
