@file:Suppress("detekt:LargeClass")

package bou.amine.apps.readerforselfossv2.tests.repository

import bou.amine.apps.readerforselfossv2.dao.ReaderForSelfossDB
import bou.amine.apps.readerforselfossv2.dao.SOURCE
import bou.amine.apps.readerforselfossv2.dao.TAG
import bou.amine.apps.readerforselfossv2.model.SelfossModel
import bou.amine.apps.readerforselfossv2.model.StatusAndData
import bou.amine.apps.readerforselfossv2.model.SuccessResponse
import bou.amine.apps.readerforselfossv2.repository.Repository
import bou.amine.apps.readerforselfossv2.rest.SelfossApi
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import bou.amine.apps.readerforselfossv2.service.ConnectivityService
import bou.amine.apps.readerforselfossv2.utils.ItemType
import bou.amine.apps.readerforselfossv2.utils.toView
import io.mockk.clearAllMocks
import io.mockk.coEvery
import io.mockk.coVerify
import io.mockk.every
import io.mockk.mockk
import io.mockk.verify
import junit.framework.TestCase.assertEquals
import junit.framework.TestCase.assertFalse
import junit.framework.TestCase.assertNotSame
import junit.framework.TestCase.assertSame
import junit.framework.TestCase.assertTrue
import kotlinx.coroutines.runBlocking
import org.junit.Assert.assertNotEquals
import org.junit.Before
import org.junit.Test

private const val BASE_URL = "https://test.com/selfoss/"

private const val USERNAME = "username"

private const val SPOUT = "spouts\\rss\\fulltextrss"

private const val IMAGE_URL = "b3aa8a664d08eb15d6ff1db2fa83e0d9.png"

private const val IMAGE_URL_2 = "d8c92cdb1ef119ea85c4b9205c879ca7.png"

private const val FEED_URL = "https://test.com/feed"

private const val TAGS = "Test, New"

private const val NUMBER_ARTICLES = 100
private const val NUMBER_UNREAD = 50
private const val NUMBER_STARRED = 20

class RepositoryTest {
    private val db = mockk<ReaderForSelfossDB>(relaxed = true)
    private val appSettingsService = mockk<AppSettingsService>()
    private val api = mockk<SelfossApi>()
    private val connectivityService = mockk<ConnectivityService>()
    private lateinit var repository: Repository

    private fun initializeRepository(isNetworkAvailable: Boolean = true) {
        every { connectivityService.isNetworkAvailable() } returns isNetworkAvailable
        repository = Repository(api, appSettingsService, connectivityService, db)

        runBlocking {
            repository.updateApiInformation()
        }
    }

    @Before
    fun setup() {
        clearAllMocks()
        every { appSettingsService.getApiVersion() } returns 4
        every { appSettingsService.getBaseUrl() } returns BASE_URL
        every { appSettingsService.getUserName() } returns USERNAME
        every { appSettingsService.isItemCachingEnabled() } returns false
        every { appSettingsService.isUpdateSourcesEnabled() } returns false

        coEvery { api.apiInformation() } returns
            StatusAndData(
                success = true,
                data =
                    SelfossModel.ApiInformation(
                        "2.19-ba1e8e3",
                        "4.0.0",
                        SelfossModel.ApiConfiguration(false, true),
                    ),
            )
        coEvery { api.stats() } returns
            StatusAndData(
                success = true,
                data = SelfossModel.Stats(NUMBER_ARTICLES, NUMBER_UNREAD, NUMBER_STARRED),
            )

        every { db.itemsQueries.deleteItemsWhereSource(any()) } returns Unit
        every { db.itemsQueries.items().executeAsList() } returns generateTestDBItems()
        every { db.tagsQueries.deleteAllTags() } returns Unit
        every { db.tagsQueries.transaction(any(), any()) } returns Unit
        every { db.tagsQueries.insertTag(any()) } returns Unit
    }

    @Test
    fun instantiate_repository() {
        initializeRepository()

        coVerify(exactly = 1) { api.apiInformation() }
    }

    @Test
    fun instantiate_repository_without_api_version() {
        every { appSettingsService.getApiVersion() } returns -1

        initializeRepository(false)

        coVerify(exactly = 0) { api.apiInformation() }
        coVerify(exactly = 0) { api.stats() }
    }

    @Test
    fun get_api_4_date_with_api_1_version_stored() {
        every { appSettingsService.getApiVersion() } returns 1
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = true, data = generateTestApiItem())
        every { appSettingsService.updateApiVersion(any()) } returns Unit

        initializeRepository()
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(repository.items.size, 1)
        verify(exactly = 1) { appSettingsService.updateApiVersion(4) }
    }

    @Test
    fun get_public_access() {
        every { appSettingsService.updatePublicAccess(any()) } returns Unit
        coEvery { api.apiInformation() } returns
            StatusAndData(
                success = true,
                data =
                    SelfossModel.ApiInformation(
                        "2.19-ba1e8e3",
                        "4.0.0",
                        SelfossModel.ApiConfiguration(true, true),
                    ),
            )
        every { appSettingsService.getUserName() } returns ""

        initializeRepository()

        coVerify(exactly = 1) { api.apiInformation() }
        coVerify(exactly = 1) { appSettingsService.updatePublicAccess(true) }
    }

    @Test
    fun get_public_access_username_not_empty() {
        every { appSettingsService.updatePublicAccess(any()) } returns Unit
        coEvery { api.apiInformation() } returns
            StatusAndData(
                success = true,
                data =
                    SelfossModel.ApiInformation(
                        "2.19-ba1e8e3",
                        "4.0.0",
                        SelfossModel.ApiConfiguration(true, true),
                    ),
            )
        every { appSettingsService.getUserName() } returns "username"

        initializeRepository()

        coVerify(exactly = 1) { api.apiInformation() }
        coVerify(exactly = 0) { appSettingsService.updatePublicAccess(true) }
    }

    @Test
    fun get_public_access_no_auth() {
        every { appSettingsService.updatePublicAccess(any()) } returns Unit
        coEvery { api.apiInformation() } returns
            StatusAndData(
                success = true,
                data =
                    SelfossModel.ApiInformation(
                        "2.19-ba1e8e3",
                        "4.0.0",
                        SelfossModel.ApiConfiguration(true, false),
                    ),
            )
        every { appSettingsService.getUserName() } returns ""

        initializeRepository()

        coVerify(exactly = 1) { api.apiInformation() }
        coVerify(exactly = 0) { appSettingsService.updatePublicAccess(true) }
    }

    @Test
    fun get_public_access_disabled() {
        every { appSettingsService.updatePublicAccess(any()) } returns Unit
        coEvery { api.apiInformation() } returns
            StatusAndData(
                success = true,
                data =
                    SelfossModel.ApiInformation(
                        "2.19-ba1e8e3",
                        "4.0.0",
                        SelfossModel.ApiConfiguration(false, true),
                    ),
            )
        every { appSettingsService.getUserName() } returns ""

        initializeRepository()

        coVerify(exactly = 1) { api.apiInformation() }
        coVerify(exactly = 0) { appSettingsService.updatePublicAccess(true) }
    }

    @Test
    fun get_api_1_date_with_api_4_version_stored() {
        every { appSettingsService.getApiVersion() } returns 4
        coEvery { api.apiInformation() } returns StatusAndData(success = false, null)
        val itemParameters = FakeItemParameters()
        itemParameters.datetime = "2021-04-23 11:45:32"
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(
                success = true,
                data = generateTestApiItem(itemParameters),
            )

        initializeRepository()
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(1, repository.items.size)
    }

    @Test
    fun get_newer_items() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = true, data = generateTestApiItem())

        initializeRepository()
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(repository.items.size, 1)
        coVerify(exactly = 1) { api.getItems("unread", 0, null, null, null, null, any()) }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_all_newer_items() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = true, data = generateTestApiItem())

        initializeRepository()
        repository.displayedItems = ItemType.ALL
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(repository.items.size, 1)
        coVerify(exactly = 1) { api.getItems("all", 0, null, null, null, null, any()) }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_newer_starred_items() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = true, data = generateTestApiItem())

        initializeRepository()
        repository.displayedItems = ItemType.STARRED
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(repository.items.size, 1)
        coVerify(exactly = 1) { api.getItems("starred", 0, null, null, null, null, any()) }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_newer_items_without_connectivity() {
        every { appSettingsService.isItemCachingEnabled() } returns true

        initializeRepository(false)
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(repository.items.size, 1)
        coVerify(exactly = 0) { api.getItems("unread", 0, null, null, null, null, any()) }
        verify(atLeast = 1) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_newer_items_without_connectivity_and_tag_filter() {
        val itemParameter1 = FakeItemParameters()
        val itemParameter2 = FakeItemParameters()
        val itemParameter3 = FakeItemParameters()
        itemParameter2.tags = "Test, Stuff"
        itemParameter2.id = "2"
        itemParameter3.tags = "Other, Tag"
        itemParameter3.id = "3"
        coEvery { db.itemsQueries.items().executeAsList() } returns generateTestDBItems(
            itemParameter1,
        ) +
            generateTestDBItems(itemParameter2) +
            generateTestDBItems(itemParameter3)

        every { appSettingsService.isItemCachingEnabled() } returns true

        initializeRepository(false)
        repository.setTagFilter(SelfossModel.Tag("Test", "red", 3))
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(repository.items.size, 1)
        coVerify(exactly = 0) { api.getItems("unread", 0, null, null, null, null, any()) }
        verify(atLeast = 1) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_newer_items_without_connectivity_and_source_filter() {
        val itemParameter1 = FakeItemParameters()
        val itemParameter2 = FakeItemParameters()
        val itemParameter3 = FakeItemParameters()
        itemParameter2.sourcetitle = "Test"
        itemParameter2.id = "2"
        itemParameter3.sourcetitle = "Other"
        itemParameter3.id = "3"
        coEvery { db.itemsQueries.items().executeAsList() } returns generateTestDBItems(
            itemParameter1,
        ) +
            generateTestDBItems(itemParameter2) +
            generateTestDBItems(itemParameter3)

        every { appSettingsService.isItemCachingEnabled() } returns true

        initializeRepository(false)
        repository.setSourceFilter(
            SelfossModel.SourceDetail(
                1,
                "Test",
                null,
                listOf("tags"),
                SPOUT,
                "",
                IMAGE_URL,
                SelfossModel.SourceParams("url"),
            ),
        )
        runBlocking {
            repository.getNewerItems()
        }

        assertSame(repository.items.size, 1)
        coVerify(exactly = 0) { api.getItems("unread", 0, null, null, null, null, any()) }
        verify(atLeast = 1) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_older_items() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = true, data = generateTestApiItem())

        initializeRepository()
        repository.items = ArrayList(generateTestApiItem())
        runBlocking {
            repository.getOlderItems()
        }

        assertSame(repository.items.size, 2)
        coVerify(exactly = 1) { api.getItems("unread", 1, null, null, null, null, any()) }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_all_older_items() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = true, data = generateTestApiItem())

        initializeRepository()
        repository.items = ArrayList(generateTestApiItem())
        repository.displayedItems = ItemType.ALL
        runBlocking {
            repository.getOlderItems()
        }

        assertSame(repository.items.size, 2)
        coVerify(exactly = 1) { api.getItems("all", 1, null, null, null, null, any()) }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_older_starred_items() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = true, data = generateTestApiItem())

        initializeRepository()
        repository.displayedItems = ItemType.STARRED
        repository.items = ArrayList(generateTestApiItem())
        runBlocking {
            repository.getOlderItems()
        }

        assertSame(repository.items.size, 2)
        coVerify(exactly = 1) { api.getItems("starred", 1, null, null, null, null, any()) }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun reload_badges() {
        var success: Boolean

        initializeRepository()
        runBlocking {
            success = repository.reloadBadges()
        }

        assertSame(true, success)
        assertEquals(NUMBER_ARTICLES, repository.badgeAll.value)
        assertEquals(NUMBER_UNREAD, repository.badgeUnread.value)
        assertEquals(NUMBER_STARRED, repository.badgeStarred.value)
        coVerify(atLeast = 1) { api.stats() }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun reload_badges_without_response() {
        coEvery { api.stats() } returns StatusAndData(success = false, data = null)

        var success: Boolean

        initializeRepository()
        runBlocking {
            success = repository.reloadBadges()
        }

        assertSame(false, success)
        assertSame(0, repository.badgeAll.value)
        assertSame(0, repository.badgeUnread.value)
        assertSame(0, repository.badgeStarred.value)
        coVerify(atLeast = 1) { api.stats() }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun reload_badges_without_connection() {
        every { appSettingsService.isItemCachingEnabled() } returns true
        every { db.itemsQueries.items().executeAsList() } returns generateTestDBItems()

        var success: Boolean

        initializeRepository(false)
        runBlocking {
            success = repository.reloadBadges()
        }

        assertTrue(success)
        assertEquals(1, repository.badgeAll.value)
        assertEquals(1, repository.badgeUnread.value)
        assertEquals(1, repository.badgeStarred.value)
        coVerify(exactly = 0) { api.stats() }
        verify(atLeast = 1) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun reload_badges_without_connection_and_items_caching_disabled() {
        every { appSettingsService.isItemCachingEnabled() } returns false
        every { appSettingsService.isUpdateSourcesEnabled() } returns true

        var success: Boolean

        initializeRepository(false)
        runBlocking {
            success = repository.reloadBadges()
        }

        assertFalse(success)
        assertSame(0, repository.badgeAll.value)
        assertSame(0, repository.badgeUnread.value)
        assertSame(0, repository.badgeStarred.value)
        coVerify(exactly = 0) { api.stats() }
        verify(exactly = 0) { db.itemsQueries.items().executeAsList() }
    }

    @Test
    fun get_tags() {
        val (tags, tagsDB) = prepareTags()

        coEvery { api.tags() } returns StatusAndData(success = true, data = tags)
        coEvery { db.tagsQueries.tags().executeAsList() } returns tagsDB
        every { appSettingsService.isUpdateSourcesEnabled() } returns true
        every { appSettingsService.isItemCachingEnabled() } returns true

        initializeRepository()
        var testTags: List<SelfossModel.Tag>?
        runBlocking {
            testTags = repository.getTags()
        }

        assertSame(tags, testTags)
        assertNotSame(tagsDB.map { it.toView() }, testTags)
        coVerify(exactly = 1) { api.tags() }
    }

    @Test
    fun get_tags_with_sources_update_disabled() {
        val (tags, tagsDB) = prepareTags()
        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        every { appSettingsService.isItemCachingEnabled() } returns true

        initializeRepository()
        var testTags: List<SelfossModel.Tag>
        runBlocking {
            testTags = repository.getTags()
            // Tags will be fetched from the database on the second call, thus testTags != tags
            testTags = repository.getTags()
        }

        coVerify(exactly = 1) { api.tags() }
        assertNotSame(tags, testTags)
        assertEquals(tagsDB.map { it.toView() }, testTags)
        verify(atLeast = 1) { db.tagsQueries.tags().executeAsList() }
    }

    @Test
    fun get_tags_with_items_caching_disabled() {
        val (tags, _) = prepareTags()
        every { appSettingsService.isUpdateSourcesEnabled() } returns true
        every { appSettingsService.isItemCachingEnabled() } returns false

        initializeRepository()
        var testTags: List<SelfossModel.Tag>
        runBlocking {
            testTags = repository.getTags()
        }

        assertSame(tags, testTags)
        coVerify(exactly = 1) { api.tags() }
        verify(exactly = 0) { db.tagsQueries.tags().executeAsList() }
    }

    @Test
    fun get_tags_with_sources_update_and_items_caching_disabled() {
        val (tags, tagsDB) = prepareTags()
        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        every { appSettingsService.isItemCachingEnabled() } returns false

        initializeRepository()
        var testTags: List<SelfossModel.Tag>
        runBlocking {
            testTags = repository.getTags()
            testTags = repository.getTags()
        }

        coVerify(exactly = 1) { api.tags() }
        assertNotSame(tags, testTags)
        assertEquals(tagsDB.map { it.toView() }, testTags)
        verify(atLeast = 1) { db.tagsQueries.tags().executeAsList() }
    }

    @Test
    fun get_tags_without_connection() {
        val (tags, tagsDB) = prepareTags()
        every { appSettingsService.isUpdateSourcesEnabled() } returns true
        every { appSettingsService.isItemCachingEnabled() } returns true

        initializeRepository(false)
        var testTags: List<SelfossModel.Tag>
        runBlocking {
            testTags = repository.getTags()
        }

        assertNotSame(tags, testTags)
        assertEquals(tagsDB.map { it.toView() }, testTags)
        coVerify(exactly = 0) { api.tags() }
        verify(atLeast = 1) { db.tagsQueries.tags().executeAsList() }
    }

    @Test
    fun get_tags_without_connection_and_items_caching_disabled() {
        prepareTags()
        every { appSettingsService.isItemCachingEnabled() } returns false
        every { appSettingsService.isUpdateSourcesEnabled() } returns true

        initializeRepository(false)
        var testTags: List<SelfossModel.Tag>
        runBlocking {
            testTags = repository.getTags()
        }

        assertSame(emptyList<SelfossModel.Tag>(), testTags)
        coVerify(exactly = 0) { api.tags() }
        verify(exactly = 0) { db.tagsQueries.tags().executeAsList() }
    }

    @Test
    fun get_tags_without_connection_and_sources_update_disabled() {
        val (tags, tagsDB) = prepareTags()
        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        every { appSettingsService.isItemCachingEnabled() } returns true

        initializeRepository(false)
        var testTags: List<SelfossModel.Tag>
        runBlocking {
            testTags = repository.getTags()
        }

        assertNotSame(tags, testTags)
        assertEquals(tagsDB.map { it.toView() }, testTags)
        coVerify(exactly = 0) { api.tags() }
        verify(atLeast = 1) { db.tagsQueries.tags().executeAsList() }
    }

    @Test
    fun get_tags_without_connection_and_sources_update_and_items_caching_disabled() {
        val (_, tagsDB) = prepareTags()
        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        every { appSettingsService.isItemCachingEnabled() } returns false

        initializeRepository(false)
        var testTags: List<SelfossModel.Tag>
        runBlocking {
            testTags = repository.getTags()
        }

        assertEquals(tagsDB.map { it.toView() }, testTags)
        coVerify(exactly = 0) { api.tags() }
        verify(atLeast = 1) { db.tagsQueries.tags().executeAsList() }
    }

    private fun prepareTags(): Pair<List<SelfossModel.Tag>, List<TAG>> {
        val tags =
            listOf(
                SelfossModel.Tag("test", "red", 6),
                SelfossModel.Tag("second", "yellow", 0),
            )
        val tagsDB =
            listOf(
                TAG("test_DB", "red", 6),
                TAG("second_DB", "yellow", 0),
            )

        coEvery { api.tags() } returns StatusAndData(success = true, data = tags)
        coEvery { db.tagsQueries.tags().executeAsList() } returns tagsDB
        return Pair(tags, tagsDB)
    }

    @Test
    fun get_sources() {
        val (sources, sourcesDB) = prepareSources()
        initializeRepository()
        var testSources: List<SelfossModel.Source>
        runBlocking {
            testSources = repository.getSourcesDetails()
        }

        assertEquals(sources, testSources)
        assertNotEquals(sourcesDB.map { it.toView() }, testSources)
        coVerify(exactly = 1) { api.sourcesDetailed() }
    }

    private fun prepareSources(): Pair<ArrayList<SelfossModel.SourceDetail>, List<SOURCE>> {
        val sources =
            arrayListOf(
                SelfossModel.SourceDetail(
                    1,
                    "First source",
                    null,
                    listOf("Test", "second"),
                    SPOUT,
                    "",
                    IMAGE_URL_2,
                    SelfossModel.SourceParams("url"),
                ),
                SelfossModel.SourceDetail(
                    2,
                    "Second source",
                    null,
                    listOf("second"),
                    SPOUT,
                    "",
                    IMAGE_URL,
                    SelfossModel.SourceParams("url"),
                ),
            )
        val sourcesDB =
            listOf(
                SOURCE(
                    "1",
                    "First DB source",
                    "Test,second",
                    SPOUT,
                    "",
                    IMAGE_URL_2,
                    "url",
                ),
                SOURCE(
                    "2",
                    "Second source",
                    "second",
                    SPOUT,
                    "",
                    IMAGE_URL,
                    "url",
                ),
            )

        coEvery { api.sourcesDetailed() } returns StatusAndData(success = true, data = sources)
        every { db.sourcesQueries.sources().executeAsList() } returns sourcesDB
        return Pair(sources, sourcesDB)
    }

    @Test
    fun get_sources_with_sources_update_disabled() {
        val (sources, sourcesDB) = prepareSources()

        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        every { appSettingsService.isItemCachingEnabled() } returns true
        initializeRepository()
        var testSources: List<SelfossModel.Source>?
        runBlocking {
            testSources = repository.getSourcesDetails()
            // Sources will be fetched from the database on the second call, thus testSources != sources
            testSources = repository.getSourcesDetails()
        }

        coVerify(exactly = 1) { api.sourcesDetailed() }
        assertNotEquals(sources, testSources)
        assertEquals(sourcesDB.map { it.toView() }, testSources)
        verify(atLeast = 1) { db.sourcesQueries.sources().executeAsList() }
    }

    @Test
    fun get_sources_with_items_caching_disabled() {
        val (sources, _) = prepareSources()

        every { appSettingsService.isUpdateSourcesEnabled() } returns true
        every { appSettingsService.isItemCachingEnabled() } returns false
        initializeRepository()
        var testSources: List<SelfossModel.Source>
        runBlocking {
            testSources = repository.getSourcesDetails()
        }

        assertEquals(sources, testSources)
        coVerify(exactly = 1) { api.sourcesDetailed() }
        verify(exactly = 0) { db.sourcesQueries }
    }

    @Test
    fun get_sources_with_sources_update_and_items_caching_disabled() {
        val (sources, _) = prepareSources()

        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        every { appSettingsService.isItemCachingEnabled() } returns false
        initializeRepository()
        var testSources: List<SelfossModel.Source>
        runBlocking {
            testSources = repository.getSourcesDetails()
        }

        assertEquals(sources, testSources)
        coVerify(exactly = 1) { api.sourcesDetailed() }
        verify(atLeast = 1) { db.sourcesQueries }
    }

    @Test
    fun get_sources_without_connection() {
        val (_, sourcesDB) = prepareSources()
        initializeRepository(false)
        var testSources: List<SelfossModel.Source>
        runBlocking {
            testSources = repository.getSourcesDetails()
        }

        assertEquals(sourcesDB.map { it.toView() }, testSources)
        coVerify(exactly = 0) { api.sourcesDetailed() }
        verify(atLeast = 1) { db.sourcesQueries.sources().executeAsList() }
    }

    @Test
    fun get_sources_without_connection_and_items_caching_disabled() {
        prepareSources()

        every { appSettingsService.isItemCachingEnabled() } returns false
        every { appSettingsService.isUpdateSourcesEnabled() } returns true
        initializeRepository(false)
        var testSources: List<SelfossModel.Source>
        runBlocking {
            testSources = repository.getSourcesDetails()
        }

        assertEquals(emptyList<SelfossModel.Source>(), testSources)
        coVerify(exactly = 0) { api.sourcesDetailed() }
        verify(exactly = 0) { db.sourcesQueries.sources().executeAsList() }
    }

    @Test
    fun get_sources_without_connection_and_sources_update_disabled() {
        val (_, sourcesDB) = prepareSources()

        every { appSettingsService.isItemCachingEnabled() } returns true
        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        initializeRepository(false)
        var testSources: List<SelfossModel.Source>
        runBlocking {
            testSources = repository.getSourcesDetails()
        }

        assertEquals(sourcesDB.map { it.toView() }, testSources)
        coVerify(exactly = 0) { api.sourcesDetailed() }
        verify(atLeast = 1) { db.sourcesQueries.sources().executeAsList() }
    }

    @Test
    fun get_sources_without_connection_and_items_caching_and_sources_update_disabled() {
        val (_, sourcesDB) = prepareSources()

        every { appSettingsService.isItemCachingEnabled() } returns false
        every { appSettingsService.isUpdateSourcesEnabled() } returns false
        initializeRepository(false)
        var testSources: List<SelfossModel.Source>
        runBlocking {
            testSources = repository.getSourcesDetails()
        }

        assertEquals(sourcesDB.map { it.toView() }, testSources)
        coVerify(exactly = 0) { api.sourcesDetailed() }
        verify(atLeast = 1) { db.sourcesQueries.sources().executeAsList() }
    }

    @Test
    fun create_source() {
        coEvery { api.createSourceForVersion(any(), any(), any(), any()) } returns
            SuccessResponse(true)

        initializeRepository()
        var response: Boolean
        runBlocking {
            response =
                repository.createSource(
                    "test",
                    FEED_URL,
                    SPOUT,
                    TAGS,
                )
        }

        coVerify(exactly = 1) {
            api.createSourceForVersion(
                any(),
                any(),
                any(),
                any(),
            )
        }
        assertSame(true, response)
    }

    @Test
    fun create_source_but_response_fails() {
        coEvery { api.createSourceForVersion(any(), any(), any(), any()) } returns
            SuccessResponse(false)

        initializeRepository()
        var response: Boolean
        runBlocking {
            response =
                repository.createSource(
                    "test",
                    FEED_URL,
                    SPOUT,
                    TAGS,
                )
        }

        coVerify(exactly = 1) {
            api.createSourceForVersion(
                any(),
                any(),
                any(),
                any(),
            )
        }
        assertSame(false, response)
    }

    @Test
    fun create_source_without_connection() {
        coEvery { api.createSourceForVersion(any(), any(), any(), any()) } returns
            SuccessResponse(true)

        initializeRepository(false)
        var response: Boolean
        runBlocking {
            response =
                repository.createSource(
                    "test",
                    FEED_URL,
                    SPOUT,
                    TAGS,
                )
        }

        coVerify(exactly = 0) {
            api.createSourceForVersion(
                any(),
                any(),
                any(),
                any(),
            )
        }
        assertSame(false, response)
    }

    @Test
    fun delete_source() {
        coEvery { api.deleteSource(any()) } returns SuccessResponse(true)

        initializeRepository()
        var response: Boolean
        runBlocking {
            response = repository.deleteSource(5, "src")
        }

        coVerify(exactly = 1) { api.deleteSource(5) }
        coVerify(exactly = 1) { db.itemsQueries.deleteItemsWhereSource("src") }
        assertSame(true, response)
    }

    @Test
    fun delete_source_but_response_fails() {
        coEvery { api.deleteSource(any()) } returns SuccessResponse(false)

        initializeRepository()
        var response: Boolean
        runBlocking {
            response = repository.deleteSource(5, "src")
        }

        coVerify(exactly = 1) { api.deleteSource(5) }
        coVerify(exactly = 0) { db.itemsQueries.deleteItemsWhereSource("src") }
        assertSame(false, response)
    }

    @Test
    fun delete_source_without_connection() {
        coEvery { api.deleteSource(any()) } returns SuccessResponse(false)

        initializeRepository(false)
        var response: Boolean
        runBlocking {
            response = repository.deleteSource(5, "src")
        }

        coVerify(exactly = 0) { api.deleteSource(5) }
        coVerify(exactly = 1) { db.itemsQueries.deleteItemsWhereSource("src") }
        assertSame(false, response)
    }

    @Test
    fun update_remote() {
        coEvery { api.update() } returns
            StatusAndData(
                success = true,
                data = "finished",
            )

        initializeRepository()
        var response: Boolean
        runBlocking {
            response = repository.updateRemote()
        }

        coVerify(exactly = 1) { api.update() }
        assertTrue(response)
    }

    @Test
    fun update_remote_but_response_fails() {
        coEvery { api.update() } returns
            StatusAndData(
                success = false,
                data = "unallowed access",
            )

        initializeRepository()
        var response: Boolean
        runBlocking {
            response = repository.updateRemote()
        }

        coVerify(exactly = 1) { api.update() }
        assertSame(false, response)
    }

    @Test
    fun update_remote_with_unallowed_access() {
        coEvery { api.update() } returns
            StatusAndData(
                success = true,
                data = "unallowed access",
            )

        initializeRepository()
        var response: Boolean
        runBlocking {
            response = repository.updateRemote()
        }

        coVerify(exactly = 1) { api.update() }
        assertSame(false, response)
    }

    @Test
    fun update_remote_without_connection() {
        coEvery { api.update() } returns
            StatusAndData(
                success = true,
                data = "undocumented...",
            )

        initializeRepository(false)
        var response: Boolean
        runBlocking {
            response = repository.updateRemote()
        }

        coVerify(exactly = 0) { api.update() }
        assertSame(false, response)
    }

    @Test
    fun login() {
        coEvery { api.login() } returns SuccessResponse(success = true)

        initializeRepository()
        var response: Boolean
        runBlocking {
            response = repository.login()
        }

        coVerify(exactly = 1) { api.login() }
        assertSame(true, response)
    }

    @Test
    fun login_but_response_fails() {
        coEvery { api.login() } returns SuccessResponse(success = false)

        initializeRepository()
        var response: Boolean
        runBlocking {
            response = repository.login()
        }

        coVerify(exactly = 1) { api.login() }
        assertSame(false, response)
    }

    @Test
    fun login_but_without_connection() {
        coEvery { api.login() } returns SuccessResponse(success = true)

        initializeRepository(false)
        var response: Boolean
        runBlocking {
            response = repository.login()
        }

        coVerify(exactly = 0) { api.login() }
        assertSame(false, response)
    }

    @Test
    fun refresh_login_information() {
        coEvery { api.refreshLoginInformation() } returns Unit
        coEvery { appSettingsService.refreshLoginInformation(any(), any(), any()) } returns Unit

        initializeRepository()
        repository.refreshLoginInformation(BASE_URL, "login", "password")

        coVerify(exactly = 1) { api.refreshLoginInformation() }
        coVerify(exactly = 1) {
            appSettingsService.refreshLoginInformation(
                BASE_URL,
                "login",
                "password",
            )
        }
    }

    @Test
    fun cache_items() {
        val itemParameter1 = FakeItemParameters()
        val itemParameter2 = FakeItemParameters()
        val itemParameter3 = FakeItemParameters()
        itemParameter2.id = "2"
        itemParameter3.id = "3"
        coEvery {
            api.getItems(
                any(),
                any(),
                any(),
                any(),
                any(),
                any(),
                any(),
            )
        } returnsMany
            listOf(
                StatusAndData(success = true, data = generateTestApiItem(itemParameter1)),
                StatusAndData(success = true, data = generateTestApiItem(itemParameter2)),
                StatusAndData(success = true, data = generateTestApiItem(itemParameter1)),
            )

        initializeRepository()
        prepareSearch()
        runBlocking {
            repository.tryToCacheItemsAndGetNewOnes()
        }

        coVerify(exactly = 3) { api.getItems(any(), 0, null, null, null, null, 200) }
    }

    @Test
    fun cache_items_but_response_fails() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = false, data = generateTestApiItem())

        initializeRepository()
        prepareSearch()
        runBlocking {
            repository.tryToCacheItemsAndGetNewOnes()
        }

        coVerify(exactly = 3) { api.getItems(any(), 0, null, null, null, null, 200) }
    }

    @Test
    fun cache_items_without_connection() {
        coEvery { api.getItems(any(), any(), any(), any(), any(), any(), any()) } returns
            StatusAndData(success = false, data = generateTestApiItem())

        initializeRepository(false)
        prepareSearch()
        runBlocking {
            repository.tryToCacheItemsAndGetNewOnes()
        }

        coVerify(exactly = 0) { api.getItems(any(), 0, null, null, null, null, 200) }
    }

    private fun prepareSearch() {
        repository.setTagFilter(SelfossModel.Tag("Tag", "read", 0))
        repository.setSourceFilter(
            SelfossModel.SourceDetail(
                1,
                "First source",
                5,
                listOf("Test", "second"),
                SPOUT,
                "",
                IMAGE_URL_2,
                SelfossModel.SourceParams("url"),
            ),
        )
        repository.searchFilter = "search"
    }
}
