package bou.amine.apps.readerforselfossv2.android.utils.glide

import android.content.Context
import android.graphics.Bitmap
import android.graphics.drawable.Drawable
import android.webkit.WebView
import android.widget.ImageView
import bou.amine.apps.readerforselfossv2.android.utils.CircleImageView
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import bou.amine.apps.readerforselfossv2.utils.isEmptyOrNullOrNullString
import com.bumptech.glide.Glide
import com.bumptech.glide.load.engine.DiskCacheStrategy
import com.bumptech.glide.load.model.GlideUrl
import com.bumptech.glide.load.model.LazyHeaders
import com.bumptech.glide.request.RequestOptions
import com.bumptech.glide.request.target.ViewTarget
import com.google.android.material.chip.Chip
import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.io.InputStream
import kotlin.io.encoding.Base64
import kotlin.io.encoding.ExperimentalEncodingApi

private const val PRELOAD_IMAGE_TIMEOUT = 10000

@Suppress("detekt:ReturnCount")
@OptIn(ExperimentalEncodingApi::class)
fun String.toGlideUrl(appSettingsService: AppSettingsService): Any { // GlideUrl Or String
    if (this.isEmptyOrNullOrNullString()) {
        return ""
    }
    if (appSettingsService.getBasicUserName().isNotEmpty()) {
        val authString = "${appSettingsService.getBasicUserName()}:${appSettingsService.getBasicPassword()}"
        val authBuf = Base64.encode(authString.toByteArray(Charsets.UTF_8))

        return GlideUrl(
            this,
            LazyHeaders
                .Builder()
                .addHeader("Authorization", "Basic $authBuf")
                .build(),
        )
    } else {
        return GlideUrl(
            this,
        )
    }
}

fun WebView.getGlideImageForResource(
    url: String,
    appSettingsService: AppSettingsService,
) = Glide
    .with(this)
    .asBitmap()
    .apply(RequestOptions.diskCacheStrategyOf(DiskCacheStrategy.ALL))
    .load(url.toGlideUrl(appSettingsService))
    .submit()
    .get()

fun Context.preloadImage(
    url: String,
    appSettingsService: AppSettingsService,
) = Glide
    .with(this)
    .asBitmap()
    .apply(RequestOptions.diskCacheStrategyOf(DiskCacheStrategy.ALL).timeout(PRELOAD_IMAGE_TIMEOUT))
    .load(url.toGlideUrl(appSettingsService))
    .submit()

fun Context.imageIntoViewTarget(
    url: String,
    target: ViewTarget<Chip?, Drawable?>,
    appSettingsService: AppSettingsService,
) = Glide
    .with(this)
    .load(url.toGlideUrl(appSettingsService))
    .into(target)

fun Context.bitmapWithCache(
    url: String,
    iv: ImageView,
    appSettingsService: AppSettingsService,
) = Glide
    .with(this)
    .asBitmap()
    .apply(RequestOptions.diskCacheStrategyOf(DiskCacheStrategy.ALL))
    .load(url.toGlideUrl(appSettingsService))
    .into(iv)

fun Context.bitmapCenterCrop(
    url: String,
    iv: ImageView,
    appSettingsService: AppSettingsService,
) = Glide
    .with(this)
    .asBitmap()
    .load(url.toGlideUrl(appSettingsService))
    .apply(RequestOptions.centerCropTransform())
    .into(iv)

fun Context.bitmapFitCenter(
    url: String,
    iv: ImageView,
    appSettingsService: AppSettingsService,
) = Glide
    .with(this)
    .asBitmap()
    .load(url.toGlideUrl(appSettingsService))
    .apply(RequestOptions.fitCenterTransform())
    .into(iv)

fun Context.circularDrawable(
    url: String,
    view: CircleImageView,
    appSettingsService: AppSettingsService,
) {
    view.textView.text = ""

    Glide
        .with(this)
        .load(url.toGlideUrl(appSettingsService))
        .into(view.imageView)
}

private const val BITMAP_INPUT_STREAM_COMPRESSION_QUALITY = 80

fun getBitmapInputStream(
    bitmap: Bitmap,
    compressFormat: Bitmap.CompressFormat,
): InputStream {
    val byteArrayOutputStream = ByteArrayOutputStream()
    bitmap.compress(compressFormat, BITMAP_INPUT_STREAM_COMPRESSION_QUALITY, byteArrayOutputStream)
    val bitmapData: ByteArray = byteArrayOutputStream.toByteArray()
    return ByteArrayInputStream(bitmapData)
}
