package bou.amine.apps.readerforselfossv2.android.settings

import android.os.Bundle
import android.text.Editable
import android.text.InputFilter
import android.text.InputType
import android.text.TextWatcher
import android.widget.Toast
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate
import androidx.core.widget.addTextChangedListener
import androidx.preference.EditTextPreference
import androidx.preference.Preference
import androidx.preference.PreferenceFragmentCompat
import bou.amine.apps.readerforselfossv2.android.R
import bou.amine.apps.readerforselfossv2.android.databinding.ActivitySettingsBinding
import bou.amine.apps.readerforselfossv2.android.utils.acra.sendSilentlyWithAcraWithName
import bou.amine.apps.readerforselfossv2.android.utils.openUrlInBrowser
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import bou.amine.apps.readerforselfossv2.service.AppSettingsService.Companion.API_ITEMS_NUMBER
import bou.amine.apps.readerforselfossv2.service.AppSettingsService.Companion.CURRENT_THEME
import bou.amine.apps.readerforselfossv2.service.AppSettingsService.Companion.READER_FONT_SIZE
import com.mikepenz.aboutlibraries.LibsBuilder
import org.kodein.di.DIAware
import org.kodein.di.android.closestDI

private const val TITLE_TAG = "settingsActivityTitle"

const val MAX_ITEMS_NUMBER = 200

private const val MIN_ITEMS_NUMBER = 1

class SettingsActivity :
    AppCompatActivity(),
    PreferenceFragmentCompat.OnPreferenceStartFragmentCallback,
    DIAware {
    override val di by closestDI()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val binding = ActivitySettingsBinding.inflate(layoutInflater)

        setContentView(binding.root)
        if (savedInstanceState == null) {
            supportFragmentManager
                .beginTransaction()
                .replace(R.id.settings, MainPreferenceFragment())
                .commit()
        } else {
            title = savedInstanceState.getCharSequence(TITLE_TAG)
        }
        supportFragmentManager.addOnBackStackChangedListener {
            if (supportFragmentManager.backStackEntryCount == 0) {
                setTitle(R.string.title_activity_settings)
            }
        }

        setSupportActionBar(binding.toolbar)

        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.setDisplayShowHomeEnabled(true)
        supportActionBar?.title = title
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        // Save current activity title so we can set it again after a configuration change
        outState.putCharSequence(TITLE_TAG, title)
    }

    override fun onSupportNavigateUp(): Boolean =
        if (supportFragmentManager.popBackStackImmediate()) {
            supportActionBar?.title = getText(R.string.title_activity_settings)
            false
        } else {
            super.onBackPressed()
            true
        }

    override fun onPreferenceStartFragment(
        caller: PreferenceFragmentCompat,
        pref: Preference,
    ): Boolean {
        // Instantiate the new Fragment
        val args = pref.extras
        val fragment =
            supportFragmentManager.fragmentFactory
                .instantiate(
                    classLoader,
                    pref.fragment.toString(),
                ).apply {
                    arguments = args
                    setTargetFragment(caller, 0)
                }
        // Replace the existing Fragment with the new Fragment
        supportFragmentManager
            .beginTransaction()
            .replace(R.id.settings, fragment)
            .addToBackStack(null)
            .commit()
        title = pref.title
        supportActionBar?.title = title
        return true
    }

    class MainPreferenceFragment : PreferenceFragmentCompat() {
        override fun onCreatePreferences(
            savedInstanceState: Bundle?,
            rootKey: String?,
        ) {
            setPreferencesFromResource(R.xml.pref_main, rootKey)

            preferenceManager.findPreference<Preference>(CURRENT_THEME)?.onPreferenceChangeListener =
                Preference.OnPreferenceChangeListener { _, newValue ->
                    AppCompatDelegate.setDefaultNightMode(
                        newValue.toString().toInt(),
                    ) // ListPreference Only takes string-arrays ¯\_(ツ)_/¯
                    true
                }

            preferenceManager.findPreference<Preference>("action_about")?.onPreferenceClickListener =
                Preference.OnPreferenceClickListener { _ ->
                    context?.let {
                        LibsBuilder()
                            .withAboutIconShown(true)
                            .withAboutVersionShown(true)
                            .withShowLoadingProgress(false)
                            .start(it)
                    }
                    true
                }
        }
    }

    class GeneralPreferenceFragment : PreferenceFragmentCompat() {
        override fun onCreatePreferences(
            savedInstanceState: Bundle?,
            rootKey: String?,
        ) {
            setPreferencesFromResource(R.xml.pref_general, rootKey)

            val editTextPreference =
                preferenceManager.findPreference<EditTextPreference>(API_ITEMS_NUMBER)
            editTextPreference?.setOnBindEditTextListener { editText ->
                editText.inputType = InputType.TYPE_CLASS_NUMBER
                editText.filters =
                    arrayOf(
                        InputFilter { source, _, _, dest, _, _ ->
                            try {
                                val input: Int = (dest.toString() + source.toString()).toInt()
                                if (input in MIN_ITEMS_NUMBER..MAX_ITEMS_NUMBER) return@InputFilter null
                            } catch (nfe: NumberFormatException) {
                                Toast
                                    .makeText(
                                        activity,
                                        R.string.items_number_should_be_number,
                                        Toast.LENGTH_LONG,
                                    ).show()
                            }
                            ""
                        },
                    )
            }
        }
    }

    class ArticleViewerPreferenceFragment : PreferenceFragmentCompat() {
        override fun onCreatePreferences(
            savedInstanceState: Bundle?,
            rootKey: String?,
        ) {
            setPreferencesFromResource(R.xml.pref_viewer, rootKey)

            val fontSize = preferenceManager.findPreference<EditTextPreference>(READER_FONT_SIZE)
            fontSize?.setOnBindEditTextListener { editText ->
                editText.inputType = InputType.TYPE_CLASS_NUMBER
                editText.addTextChangedListener {
                    object : TextWatcher {
                        override fun beforeTextChanged(
                            charSequence: CharSequence,
                            i: Int,
                            i1: Int,
                            i2: Int,
                        ) {
                            // We do nothing
                        }

                        override fun onTextChanged(
                            charSequence: CharSequence,
                            i: Int,
                            i1: Int,
                            i2: Int,
                        ) {
                            // We do nothing
                        }

                        override fun afterTextChanged(editable: Editable) {
                            try {
                                editText.textSize = editable.toString().toInt().toFloat()
                            } catch (e: NumberFormatException) {
                                e.sendSilentlyWithAcraWithName("ArticleViewerPreferenceFragment > afterTextChanged")
                            }
                        }
                    }
                }
                editText.filters =
                    arrayOf(
                        InputFilter { source, _, _, dest, _, _ ->
                            try {
                                val input = (dest.toString() + source.toString()).toInt()
                                if (input > 0) return@InputFilter null
                            } catch (nfe: NumberFormatException) {
                                nfe.sendSilentlyWithAcraWithName("ArticleViewerPreferenceFragment > filters")
                            }
                            ""
                        },
                    )
            }
        }
    }

    class OfflinePreferenceFragment : PreferenceFragmentCompat() {
        override fun onCreatePreferences(
            savedInstanceState: Bundle?,
            rootKey: String?,
        ) {
            setPreferencesFromResource(R.xml.pref_offline, rootKey)
        }
    }

    class LinksPreferenceFragment : PreferenceFragmentCompat() {
        private fun openUrl(url: String) {
            context?.openUrlInBrowser(url)
        }

        override fun onCreatePreferences(
            savedInstanceState: Bundle?,
            rootKey: String?,
        ) {
            setPreferencesFromResource(R.xml.pref_links, rootKey)

            preferenceManager.findPreference<Preference>("trackerLink")?.onPreferenceClickListener =
                Preference.OnPreferenceClickListener {
                    openUrl(AppSettingsService.BUG_URL)
                    true
                }

            preferenceManager.findPreference<Preference>("sourceLink")?.onPreferenceClickListener =
                Preference.OnPreferenceClickListener {
                    openUrl(AppSettingsService.SOURCE_URL)
                    false
                }

            preferenceManager.findPreference<Preference>("translation")?.onPreferenceClickListener =
                Preference.OnPreferenceClickListener {
                    openUrl(AppSettingsService.TRANSLATION_URL)
                    false
                }
        }
    }

    class ExperimentalPreferenceFragment : PreferenceFragmentCompat() {
        override fun onCreatePreferences(
            savedInstanceState: Bundle?,
            rootKey: String?,
        ) {
            setPreferencesFromResource(R.xml.pref_experimental, rootKey)
        }
    }
}
