package bou.amine.apps.readerforselfossv2.android

import android.graphics.Color
import android.os.Bundle
import android.view.KeyEvent
import android.view.Menu
import android.view.MenuItem
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.viewpager2.adapter.FragmentStateAdapter
import androidx.viewpager2.widget.ViewPager2
import bou.amine.apps.readerforselfossv2.android.databinding.ActivityReaderBinding
import bou.amine.apps.readerforselfossv2.android.fragments.ArticleFragment
import bou.amine.apps.readerforselfossv2.model.SelfossModel
import bou.amine.apps.readerforselfossv2.repository.Repository
import bou.amine.apps.readerforselfossv2.service.AppSettingsService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.kodein.di.DIAware
import org.kodein.di.android.closestDI
import org.kodein.di.instance

class ReaderActivity :
    AppCompatActivity(),
    DIAware {
    private var currentItem: Int = 0

    private var toolbarMenu: Menu? = null

    private lateinit var binding: ActivityReaderBinding

    private var allItems: ArrayList<SelfossModel.Item> = ArrayList()

    override val di by closestDI()
    private val repository: Repository by instance()
    private val appSettingsService: AppSettingsService by instance()

    @Suppress("detekt:SwallowedException")
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityReaderBinding.inflate(layoutInflater)
        val view = binding.root

        setContentView(view)

        setSupportActionBar(binding.toolBar)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        supportActionBar?.setDisplayShowHomeEnabled(true)

        currentItem = intent.getIntExtra("currentItem", 0)

        allItems = repository.getReaderItems()

        if (allItems.isEmpty() || currentItem > allItems.size) {
            finish()
        }

        readItem()

        binding.pager.adapter = ScreenSlidePagerAdapter(this)
        binding.pager.setCurrentItem(currentItem, false)

        binding.pager.registerOnPageChangeCallback(
            object : ViewPager2.OnPageChangeCallback() {
                override fun onPageSelected(position: Int) {
                    super.onPageSelected(position)
                    currentItem = position
                    updateStarIcon()
                    readItem()
                }
            },
        )
    }

    override fun onResume() {
        super.onResume()

        binding.indicator.setViewPager(binding.pager)
    }

    private fun readItem() {
        val item = allItems.getOrNull(currentItem)
        if (appSettingsService.isMarkOnScrollEnabled() && !appSettingsService.getPublicAccess() && item != null) {
            CoroutineScope(Dispatchers.IO).launch {
                repository.markAsRead(item)
            }
        }
    }

    private fun updateStarIcon() {
        if (toolbarMenu != null) {
            val isStarred = allItems.getOrNull(currentItem)?.starred ?: false
            toolbarMenu!!.findItem(R.id.star)?.icon?.setTint(if (isStarred) Color.RED else Color.WHITE)
        }
    }

    override fun onSaveInstanceState(oldInstanceState: Bundle) {
        super.onSaveInstanceState(oldInstanceState)
        oldInstanceState.clear()
    }

    private inner class ScreenSlidePagerAdapter(
        fa: FragmentActivity,
    ) : FragmentStateAdapter(fa) {
        override fun getItemCount(): Int = allItems.size

        override fun createFragment(position: Int): Fragment = ArticleFragment.newInstance(allItems[position])
    }

    override fun onKeyDown(
        keyCode: Int,
        event: KeyEvent?,
    ): Boolean =
        when (keyCode) {
            KeyEvent.KEYCODE_VOLUME_DOWN -> {
                val currentFragment =
                    supportFragmentManager.findFragmentByTag("f" + binding.pager.currentItem) as ArticleFragment
                currentFragment.volumeButtonScrollDown()
                true
            }

            KeyEvent.KEYCODE_VOLUME_UP -> {
                val currentFragment =
                    supportFragmentManager.findFragmentByTag("f" + binding.pager.currentItem) as ArticleFragment
                currentFragment.volumeButtonScrollUp()
                true
            }

            else -> {
                super.onKeyDown(keyCode, event)
            }
        }

    private fun alignmentMenu() {
        val showJustify = appSettingsService.getActiveAllignment() == AppSettingsService.ALIGN_LEFT
        if (toolbarMenu != null) {
            toolbarMenu!!.findItem(R.id.align_left).isVisible = !showJustify
            toolbarMenu!!.findItem(R.id.align_justify).isVisible = showJustify
        }
    }

    override fun onCreateOptionsMenu(menu: Menu): Boolean {
        menuInflater.inflate(R.menu.reader_menu, menu)
        toolbarMenu = menu

        alignmentMenu()

        if (appSettingsService.getPublicAccess()) {
            menu.removeItem(R.id.star)
        } else {
            updateStarIcon()
        }

        return true
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        when (item.itemId) {
            android.R.id.home -> onBackPressedDispatcher.onBackPressed()
            R.id.star -> toggleFavorite()
            R.id.align_left -> switchAlignmentSetting(AppSettingsService.ALIGN_LEFT)
            R.id.align_justify -> switchAlignmentSetting(AppSettingsService.JUSTIFY)
        }
        return super.onOptionsItemSelected(item)
    }

    private fun toggleFavorite() {
        val item = allItems.getOrNull(currentItem) ?: return

        val starred = item.starred

        CoroutineScope(Dispatchers.IO).launch {
            if (starred) {
                repository.unstarr(item)
            } else {
                repository.starr(item)
            }
        }

        item.toggleStar()
        updateStarIcon()
    }

    private fun switchAlignmentSetting(alignment: Int) {
        appSettingsService.changeAllignment(alignment)
        alignmentMenu()

        val fragmentManager = supportFragmentManager
        val fragments = fragmentManager.fragments

        for (fragment in fragments) {
            if (fragment is ArticleFragment) {
                fragment.refreshAlignment()
            }
        }
    }
}
