import 'package:panoramax_mobile/main.dart';
import 'package:panoramax_mobile/service/plugin/gps_location_service.dart';
import 'package:panoramax_mobile/service/plugin/magnetometer_calibration_service.dart';
import 'package:test/test.dart';

void main() {
  group('interpolatePhoto', () {
    late List <PhotoToBeRelocated> photoToInterpolateGPS;

    setUp(() {
      photoToInterpolateGPS = fakePhotosToInterpolateGPS();
    });

    test('Should interpolate point between two captures unitary', () {
      expect(
        CaptureService.interpolateBetweenGpsCaptures(
          photoTime: photoToInterpolateGPS[0].captureTime,
          before: fakeGpsCourse.captures[0],
          after: fakeGpsCourse.captures[1]
        ),
        GpsLocation.fromMap({
          'latitude': 1.0,
          'longitude': 1.0,
          'altitude': 10.0,
          'timestamp': photoToInterpolateGPS[0].captureTime.millisecondsSinceEpoch
        })
      );

      expect(
        CaptureService.interpolateBetweenGpsCaptures(
          photoTime: photoToInterpolateGPS[1].captureTime,
          before: fakeGpsCourse.captures[0],
          after: fakeGpsCourse.captures[1]
        ),
        GpsLocation.fromMap({
          'latitude': 3.0,
          'longitude': 4.0,
          'altitude': 10.0,
          'timestamp': photoToInterpolateGPS[1].captureTime.millisecondsSinceEpoch
        })
      );

      expect(
        CaptureService.interpolateBetweenGpsCaptures(
          photoTime: photoToInterpolateGPS[2].captureTime,
          before: fakeGpsCourse.captures[0],
          after: fakeGpsCourse.captures[1]
        ),
        GpsLocation.fromMap({
          'latitude': 5.0,
          'longitude': 7.0,
          'altitude': 10.0,
          'timestamp': photoToInterpolateGPS[2].captureTime.millisecondsSinceEpoch
        })
      );

      expect(
        CaptureService.interpolateBetweenGpsCaptures(
          photoTime: photoToInterpolateGPS[3].captureTime,
          before: fakeGpsCourse.captures[0],
          after: fakeGpsCourse.captures[1]
        ),
        GpsLocation.fromMap({
          'latitude': 9.0,
          'longitude': 13.0,
          'altitude': 10.0,
          'timestamp': photoToInterpolateGPS[3].captureTime.millisecondsSinceEpoch
        })
      );

      expect(
        CaptureService.interpolateBetweenGpsCaptures(
          photoTime: photoToInterpolateGPS[4].captureTime,
          before: fakeGpsCourse.captures[0],
          after: fakeGpsCourse.captures[1]
        ),
        GpsLocation.fromMap({
          'latitude': 11.0,
          'longitude': 16.0,
          'altitude': 10.0,
          'timestamp': photoToInterpolateGPS[4].captureTime.millisecondsSinceEpoch
        })
      );
    });
  });

  group('interpolatePhotoSet', () {
    late List <PhotoToBeRelocated> photoToInterpolateGPS;

    setUp(() {
      photoToInterpolateGPS = fakePhotosToInterpolateGPS();
    });

    test('Should interpolate photos group between two captures', () {
      expect(
        CaptureService.interpolatePhoto(
          photos: photoToInterpolateGPS,
          gpsCourse: fakeGpsCourse,
          lastInterpolationDate: DateTime(2025, 4, 8, 8, 0, 0)
        ).items,
        [
          PhotoGpsInterpolationResultItem(
            photo: photoToInterpolateGPS[0],
            newLocation: GpsLocation.fromMap({
              'latitude': 1.0,
              'longitude': 1.0,
              'altitude': 10.0,
              'timestamp': photoToInterpolateGPS[0].captureTime.millisecondsSinceEpoch
            })
          ),
          PhotoGpsInterpolationResultItem(
              photo: photoToInterpolateGPS[1],
              newLocation: GpsLocation.fromMap({
                'latitude': 3.0,
                'longitude': 4.0,
                'altitude': 10.0,
                'timestamp': photoToInterpolateGPS[1].captureTime.millisecondsSinceEpoch
              })
          ),
          PhotoGpsInterpolationResultItem(
              photo: photoToInterpolateGPS[2],
              newLocation: GpsLocation.fromMap({
                'latitude': 5.0,
                'longitude': 7.0,
                'altitude': 10.0,
                'timestamp': photoToInterpolateGPS[2].captureTime.millisecondsSinceEpoch
              })
          ),
          PhotoGpsInterpolationResultItem(
              photo: photoToInterpolateGPS[3],
              newLocation: GpsLocation.fromMap({
                'latitude': 9.0,
                'longitude': 13.0,
                'altitude': 10.0,
                'timestamp': photoToInterpolateGPS[3].captureTime.millisecondsSinceEpoch
              })
          ),
          PhotoGpsInterpolationResultItem(
              photo: photoToInterpolateGPS[4],
              newLocation: GpsLocation.fromMap({
                'latitude': 11.0,
                'longitude': 16.0,
                'altitude': 10.0,
                'timestamp': photoToInterpolateGPS[4].captureTime.millisecondsSinceEpoch
              })
          ),
        ]
      );
    });

  test('Should interpolate part of photos group between two captures', () {
    expect(
      CaptureService.interpolatePhoto(
        photos: photoToInterpolateGPS,
        gpsCourse: fakeGpsCourse,
        lastInterpolationDate: DateTime(2025, 4, 8, 8, 0, 2, 500)
      ).items,
      [
        PhotoGpsInterpolationResultItem(
          photo: photoToInterpolateGPS[3],
          newLocation: GpsLocation.fromMap({
            'latitude': 9.0,
            'longitude': 13.0,
            'altitude': 10.0,
            'timestamp': photoToInterpolateGPS[3].captureTime.millisecondsSinceEpoch
          })
        ),
        PhotoGpsInterpolationResultItem(
          photo: photoToInterpolateGPS[4],
          newLocation: GpsLocation.fromMap({
            'latitude': 11.0,
            'longitude': 16.0,
            'altitude': 10.0,
            'timestamp': photoToInterpolateGPS[4].captureTime.millisecondsSinceEpoch
          })
        ),
      ]
    );
  });

  });

  group('getInterpolationIntervals', () {
    late GpsCourse generatedGpsCourse;
    late DateTime baseDateTime;

    setUp(() {
      generatedGpsCourse = GpsCourse();
      baseDateTime = DateTime(2025, 4, 8, 12, 0, 0);

      for (int i = 0; i < 5; i++) {
        generatedGpsCourse.captures.add(
          GpsCapture(
            location: fakeLocation(10.0 + i, 20.0 + i),
            date: baseDateTime.add(Duration(seconds: i * 10)),
          ),
        );
      }
    });

    test('returns all intervals starting at an exact match date', () {
      final intervals = CaptureService.getInterpolationIntervals(generatedGpsCourse, baseDateTime);

      expect(intervals.length, 4);
      expect(intervals.first.$1.date, baseDateTime);
      expect(intervals.first.$2.date, baseDateTime.add(Duration(seconds: 10)));
    });

    test('returns only intervals after a mid-range date', () {
      final from = baseDateTime.add(Duration(seconds: 15)); // between index 1 and 2
      final intervals = CaptureService.getInterpolationIntervals(generatedGpsCourse, from);

      expect(intervals.length, 3);
      expect(intervals.first.$1.date, baseDateTime.add(Duration(seconds: 10)));
    });

    test('returns empty list if date is after all captures', () {
      final from = baseDateTime.add(Duration(seconds: 100));
      final intervals = CaptureService.getInterpolationIntervals(generatedGpsCourse, from);

      expect(intervals, isEmpty);
    });

    test('returns all intervals if date is far before any captures', () {
      final from = baseDateTime.subtract(Duration(minutes: 1));
      final intervals = CaptureService.getInterpolationIntervals(generatedGpsCourse, from);

      expect(intervals.length, 4);
    });
  });
}

GpsLocation fakeLocation(double lat, double lon) => GpsLocation.fromMap({
  'latitude': lat,
  'longitude': lon,
  'accuracy': 5.0,
  'altitude': 0.0,
  'speed': 0.0,
  'speed_accuracy': 0.0,
  'heading': 0.0,
  'timestamp': 0,
  'isMock': false,
});

List<PhotoToBeRelocated> fakePhotosToInterpolateGPS() => [
  PhotoToBeRelocated(
      filePath: 'firstPhoto',
      captureTime: DateTime(2025, 4, 8, 8, 0, 0),
      cameraDirection: 10,
      initialPosition: fakeLocation(0, 0),
      mustInterpolatePosition: true,
      calibrationStatus: CalibrationStatus.ACCURACY_HIGH
  ),
  PhotoToBeRelocated(
      filePath: 'secondPhoto',
      captureTime: DateTime(2025, 4, 8, 8, 0, 1),
      cameraDirection: 10,
      initialPosition: fakeLocation(0, 0),
      mustInterpolatePosition: true,
      calibrationStatus: CalibrationStatus.ACCURACY_HIGH
  ),
  PhotoToBeRelocated(
      filePath: 'thirdPhoto',
      captureTime: DateTime(2025, 4, 8, 8, 0, 2),
      cameraDirection: 10,
      initialPosition: fakeLocation(0, 0),
      mustInterpolatePosition: true,
      calibrationStatus: CalibrationStatus.ACCURACY_HIGH
  ),
  PhotoToBeRelocated(
      filePath: 'fourthPhoto',
      captureTime: DateTime(2025, 4, 8, 8, 0, 4),
      cameraDirection: 10,
      initialPosition: fakeLocation(0, 0),
      mustInterpolatePosition: true,
      calibrationStatus: CalibrationStatus.ACCURACY_HIGH
  ),
  PhotoToBeRelocated(
      filePath: 'firthPhoto',
      captureTime: DateTime(2025, 4, 8, 8, 0, 5),
      cameraDirection: 10,
      initialPosition: fakeLocation(0, 0),
      mustInterpolatePosition: true,
      calibrationStatus: CalibrationStatus.ACCURACY_HIGH
  )
];

GpsCourse fakeGpsCourse = new GpsCourse()
  ..captures.addAll([
    GpsCapture(
        location: GpsLocation.fromMap({
          'latitude': 1.0,
          'longitude': 1.0,
          'altitude': 10.0,
          'accuracy': 0.0,
          'timestamp': 0
        }),
        date: DateTime(2025, 4, 8, 8, 0, 0)
    ),
    GpsCapture(
        location: GpsLocation.fromMap({
          'latitude': 6.0,
          'longitude': 8.5,
          'altitude': 10.0,
          'accuracy': 0.0,
          'timestamp': 0
        }),
        date: DateTime(2025, 4, 8, 8, 0, 2, 500)
    ),
    GpsCapture(
        location: GpsLocation.fromMap({
          'latitude': 11.0,
          'longitude': 16.0,
          'altitude': 10.0,
          'accuracy': 0.0,
          'timestamp': 0
        }),
        date: DateTime(2025, 4, 8, 8, 0, 5)
    )
  ]);