import 'package:flutter/services.dart';
import '../../main.dart';

class GpsLocator {
  static const EventChannel _eventChannel =
      EventChannel('app.panoramax.beta/gps_location_stream');

  static Stream<GpsLocation> get locationStream => _eventChannel
          .receiveBroadcastStream()
          .map((event) => GpsLocation.fromMap(Map<String, dynamic>.from(event)))
          .handleError((error) {
        Logger.getInstance().error('Error in localisation flow : $error');
      });
}

class GpsLocation {
  final double latitude;
  final double longitude;
  final double? accuracy; // en mètres
  final double altitude; // en mètres
  final DateTime? timestamp;

  GpsLocation({
    required this.latitude,
    required this.longitude,
    required this.accuracy,
    required this.altitude,
    required this.timestamp,
  });

  factory GpsLocation.fromMap(Map<dynamic, dynamic> map) => GpsLocation(
        latitude: map['latitude'] as double,
        longitude: map['longitude'] as double,
        accuracy: map['accuracy'] != null ? map['accuracy'] as double : null,
        altitude: map['altitude'] as double,
        timestamp: map['timestamp'] != null
            ? DateTime.fromMillisecondsSinceEpoch(map['timestamp'] as int)
            : null,
      );

  Map<String, dynamic> toMap() => {
        'latitude': latitude,
        'longitude': longitude,
        'accuracy': accuracy,
        'altitude': altitude,
        'timestamp': timestamp?.millisecondsSinceEpoch,
      };

  @override
  String toString() =>
      'GpsPosition(lat: $latitude, lon: $longitude, acc: ${accuracy}, alt: ${altitude}, time: $timestamp)';

  @override
  bool operator ==(Object other) =>
      identical(this, other) ||
      other is GpsLocation &&
          runtimeType == other.runtimeType &&
          latitude == other.latitude &&
          longitude == other.longitude &&
          accuracy == other.accuracy &&
          altitude == other.altitude &&
          timestamp == other.timestamp;

  @override
  int get hashCode =>
      latitude.hashCode ^
      longitude.hashCode ^
      accuracy.hashCode ^
      altitude.hashCode ^
      timestamp.hashCode;
}
