part of panoramax;

final TalkerSettings talkerSettings = TalkerSettings(
  useHistory: true,
  enabled: true,
  timeFormat: TimeFormat.yearMonthDayAndTime,
  maxHistoryItems: 10000,
);

final AnsiPen routePen = AnsiPen()..xterm(135);

class Logger {
  static final Talker _instance = initTalker();

  static Talker initTalker() {
    final talker = Talker(
        settings: talkerSettings,
        logger: TalkerLogger(
          settings: TalkerLoggerSettings(),
        ),
        history: CustomHistory(talkerSettings));
    return talker;
  }

  static Talker getInstance() => _instance;
}

extension TalkerDataExtension on TalkerData {
  /// Convert `TalkerData` to JSON
  Map<String, dynamic> toJson() => {
        'message': message,
        'title': title,
        'logLevel': logLevel?.name,
        'error': error?.toString(),
        'exception': exception?.toString().replaceAll('\n', '_____N'),
        'stackTrace': stackTrace?.toString().replaceAll('\n', '_____N'),
        'time': time.toIso8601String(),
      };

  /// Convert JSON to `TalkerData`
  static TalkerData fromJson(Map<String, dynamic> json) {
    final logLevel = LogLevel.values.firstWhere(
      (e) => e.name == json['logLevel'],
      orElse: () => LogLevel.info,
    );
    final type = TalkerLogType.fromLogLevel(logLevel);
    return TalkerData(
      json['message'] ?? 'No message',
      logLevel: logLevel,
      pen: json['title'] == 'route'
          ? routePen
          : talkerSettings.getPenByLogKey(type.key),
      error: AssertionError(json['error']),
      title: json['title'],
      exception: json['exception'] != null
          ? Exception(json['exception'].replaceAll('_____N', '\n'))
          : null,
      stackTrace: json['stackTrace'] != null
          ? StackTrace.fromString(json['stackTrace'].replaceAll('_____N', '\n'))
          : null,
      time: DateTime.parse(json['time']),
    );
  }
}

class CustomHistory implements TalkerHistory {
  final TalkerSettings settings;
  final List<TalkerData> _history = [];
  final StreamController<TalkerData> _logController =
      StreamController<TalkerData>();

  @override
  List<TalkerData> get history => _history;

  CustomHistory(this.settings, {List<TalkerData>? history}) {
    Future.microtask(() => loadHistory(history: history));
    _logController.stream.listen((logData) {
      _writeToFile(logData);
    });
  }

  void loadHistory({List<TalkerData>? history}) async {
    final File logFile = await getLogFile();
    if (!logFile.existsSync()) {
      logFile.createSync();
    }
    if (history != null) {
      _history.addAll(history);
    }
    _history.addAll(await readTalkerDataFromFile());
  }

  Future<File> getLogFile() async {
    Directory docDir;
    if (Platform.isAndroid) {
      docDir = await getExternalStorageDirectory() ??
          await getApplicationSupportDirectory();
    } else {
      docDir = await getApplicationDocumentsDirectory();
    }
    return File(path.join(docDir.path, 'history.log'));
  }

  @override
  void clean() async {
    if (settings.useHistory) {
      _history.clear();
      final File logFile = await getLogFile();
      if (logFile.existsSync()) {
        logFile.deleteSync();
      }
    }
  }

  @override
  void write(TalkerData data) {
    _history.add(data);
    _logController.add(data);
  }

  void _writeToFile(TalkerData newLog) async {
    final File file = await getLogFile();

    final String jsonEntry = jsonEncode(newLog.toJson());

    file.writeAsStringSync('$jsonEntry\n', mode: FileMode.append, flush: true);
  }

  Future<List<TalkerData>> readTalkerDataFromFile() async {
    final File file = await getLogFile();

    if (!file.existsSync()) {
      return [];
    }

    final List<String> lines = await file.readAsLines();

    return lines
        .where((line) =>
            line.trim().isNotEmpty &&
            line.trim().startsWith('{"')) // Ignore empty lines
        .map((line) {
          try {
            var fromJson = TalkerDataExtension.fromJson(jsonDecode(line));
            return fromJson;
          } on Exception catch (e) {
            debugPrint(e.toString());
            return null;
          }
        })
        .whereType<TalkerData>()
        .toList();
  }
}
