import 'package:drift/drift.dart' as drift;

import '../../main.dart';
import 'database.dart';
import 'entities.dart';

class SequenceRepository {
  static late drift.DriftDatabase driftDatabase;
  static late PanoramaxDatabase _database;

  static Future<void> initializeDatabase() async {
    _database = PanoramaxDatabase();
  }

  static Future<SequenceDto> createSequence(String name, DateTime creationDate) async {
    final companion = SequencesCompanion.insert(
      name: name,
      creationDate: creationDate,
      panoramaxSequenceId: const drift.Value(null),
    );

    final id = await _database.into(_database.sequences).insert(companion);
    return (await getSequence(id))!;
  }

  static Future<SequenceDto?> getSequence(int idSequence) async {
    final sequenceQuery = _database.select(_database.sequences)
      ..where((s) => s.id.equals(idSequence));

    final sequenceResult = await sequenceQuery.getSingleOrNull();
    if (sequenceResult == null) return null;

    final picturesQuery = _database.select(_database.sequencePictures)
      ..where((p) => p.sequenceId.equals(idSequence));
    final picturesResult = await picturesQuery.get();

    return SequenceDto.fromDrift(sequenceResult, picturesResult);
  }

  static Future<List<SequenceDto>> getAllSequences() async {
    final sequencesQuery = _database.select(_database.sequences);
    final sequencesResult = await sequencesQuery.get();

    List<SequenceDto> result = [];
    for (var seq in sequencesResult) {
      final picturesQuery = _database.select(_database.sequencePictures)
        ..where((p) => p.sequenceId.equals(seq.id));
      final picturesResult = await picturesQuery.get();

      result.add(SequenceDto.fromDrift(seq, picturesResult));
    }

    return result;
  }

  static Future<SequenceDto> updateSequence(SequenceDto sequenceToUpdate) async {
    await _database.update(_database.sequences).replace(
        Sequence(
          id: sequenceToUpdate.id,
          panoramaxSequenceId: sequenceToUpdate.panoramaxSequenceId,
          name: sequenceToUpdate.name,
          creationDate: sequenceToUpdate.creationDate,
        )
    );

    for (var picture in sequenceToUpdate.pictures) {
      if (picture.id > 0) {
        await _database.update(_database.sequencePictures).replace(
            SequencePicture(
              id: picture.id,
              localFilePath: picture.localFilePath,
              isToSend: picture.isToSend,
              hasBeenSent: picture.hasBeenSent,
              doesImageHaveCorrectExifTags: picture.doesImageHaveCorrectExifTags,
              sequenceId: sequenceToUpdate.id,
            )
        );
      } else {
        final pictureId = await _database.into(_database.sequencePictures).insert(
          SequencePicturesCompanion.insert(
            localFilePath: picture.localFilePath,
            sequenceId: sequenceToUpdate.id,
            isToSend: drift.Value(picture.isToSend),
            hasBeenSent: drift.Value(picture.hasBeenSent),
            doesImageHaveCorrectExifTags: drift.Value(picture.doesImageHaveCorrectExifTags),
          )
        );
        picture.id = pictureId;
      }
    }

    return sequenceToUpdate;
  }

  static Future<bool> deleteSequence(int id) async {
    try {
      final deletePictures = _database.delete(_database.sequencePictures)
        ..where((p) => p.sequenceId.equals(id));
      await deletePictures.go();

      final deleteSequence = _database.delete(_database.sequences)
        ..where((s) => s.id.equals(id));
      final rowsAffected = await deleteSequence.go();

      return rowsAffected > 0;
    } on Exception catch (e) {
      Logger.getInstance().info('Erreur lors de la suppression de la séquence: $e');
      return false;
    }
  }
}

// Classe factice pour maintenir la compatibilité de l'API
class Box<T> {
  Box();

  Future<T> putAndGetAsync(T object) async {
    throw UnimplementedError('Cette méthode est uniquement présente pour la compatibilité');
  }

  T? get(int id) {
    throw UnimplementedError('Cette méthode est uniquement présente pour la compatibilité');
  }

  List<T> getAll() {
    throw UnimplementedError('Cette méthode est uniquement présente pour la compatibilité');
  }

  bool remove(int id) {
    throw UnimplementedError('Cette méthode est uniquement présente pour la compatibilité');
  }
}