part of panoramax;

class CaptureService {

  static Future<SequenceDto> createSequenceDto(BuildContext context) {
    var creationDate = DateTime.now();
    final collectionName = getSequenceName(context, creationDate);
    return SequenceRepository.createSequence(
        collectionName,
        creationDate
    );
  }

  static String getSequenceName(BuildContext context, DateTime creationDate) =>
      '${AppLocalizations.of(context)!.newSequenceNamePreffix} ${DateFormat(AppLocalizations.of(context)!.newSequenceNameDateFormat).format(creationDate)}';

  static Future<SequenceDto> addImageSequence(SequenceDto sequenceToEnriched, String rawImagePath,
      {doesImageHaveCorrectExifTags = true}) async {
    sequenceToEnriched.pictures.add(
        SequencePictureDto()
          ..localFilePath = rawImagePath
          ..doesImageHaveCorrectExifTags = doesImageHaveCorrectExifTags
    );
    return SequenceRepository.updateSequence(sequenceToEnriched);
  }

  static CaptureButtonState getCaptureButtonState(AvailableCaptorsData? captorsData, bool isBurstMode, bool isBurstPlay, bool isProcessing) {
    if (captorsData?.location.accuracy == null) {
      return CaptureButtonState.GPS_NOT_GRANTED;
    }
    if (captorsData?.location == null) {
      return CaptureButtonState.GPS_IN_ACQUISITION;
    }
    if (!isGpsAccuracyGood(captorsData?.location.accuracy) || !CaptureService.isPhoneCalibrationGood(captorsData?.calibrationStatus)) {
      return CaptureButtonState.GPS_NOT_PRECISE;
    }
    if (isBurstMode) {
      return (isBurstPlay) ? CaptureButtonState.SEQUENCE_PHOTO_IN_PROGRESS : CaptureButtonState.SEQUENCE_PHOTO_AVAILABLE;
    } else {
      return (isProcessing) ? CaptureButtonState.SINGLE_PHOTO_IN_PROGRESS : CaptureButtonState.SINGLE_PHOTO_AVAILABLE;
    }
  }

  static bool isGpsAccuracyGood(double? accuracy) => accuracy != null && accuracy < GPS_ACCURACY_THRESHOLD;

  static bool isPhoneCalibrationGood(CalibrationStatus? calibrationStatus) => calibrationStatus == CalibrationStatus.ACCURACY_MEDIUM || calibrationStatus == CalibrationStatus.ACCURACY_HIGH;

  static double radiansToDegrees(double radians) => radians * (180.0 / pi);

  static Future<void> processImage(String imagePath, GpsLocation currentLocation, double? currentDirection, CalibrationStatus calibrationStatus, GyroscopeEvent? accelerometerValues) async {
    final XFile rawImage = XFile(imagePath);

    Logger.getInstance().debug('Picture taken');
    Logger.getInstance().debug('addExifTags');
    Logger.getInstance().debug('currentLocation: ' + currentLocation.toString());
    Logger.getInstance().debug('currentDirection: ' + currentDirection.toString());
    Logger.getInstance().debug('accelerometerValues: ' + accelerometerValues.toString());
    await addExifTags(rawImage, currentLocation, currentDirection, calibrationStatus, accelerometerValues);
    Logger.getInstance().debug('ExifTags added');
  }

  static Future<void> addExifTags(XFile rawImage, GpsLocation currentLocation, double? currentDirection, CalibrationStatus calibrationStatus, GyroscopeEvent? accelerometerValues) async {
    Logger.getInstance().debug('add exif tag : ' +
        currentLocation.latitude.toString() +
        ' ' +
        currentLocation.longitude.toString() +
        ' ' +
        currentLocation.altitude.toString());

      if (Platform.isIOS) {
        var exif = await Exif.fromPath(rawImage.path);
        await exif.writeAttributes({
          'GPSLatitude': currentLocation.latitude,
          'GPSLongitude': currentLocation.longitude,
          'GPSAltitude': currentLocation.altitude,
          'Software': await GeneralService.getFullAppVersion()
        });
        if (currentDirection != null && isPhoneCalibrationGood(calibrationStatus)) {
          Logger.getInstance().debug('Adding GPSImgDirection');
          final directionAttributeValue = '${(currentDirection * 100).truncate()}/100';
          await exif.writeAttributes({'GPSImgDirection': directionAttributeValue});
          Logger.getInstance().debug('GPSImgDirection ${directionAttributeValue} added');
        }
      } else {
        final exif = FlutterExif.fromPath(rawImage.path);
        await exif.setLatLong(
            currentLocation.latitude,
            currentLocation.longitude
        );
        await exif.setAltitude(currentLocation.altitude);
        await exif.setAttribute('Software', await GeneralService.getFullAppVersion());
        if (currentDirection != null && isPhoneCalibrationGood(calibrationStatus)) {
          Logger.getInstance().debug('Adding GPSImgDirection');
          final directionAttributeValue = '${(currentDirection * 100).truncate()}/100';
          await exif.setAttribute('GPSImgDirection', directionAttributeValue);
          Logger.getInstance().debug('GPSImgDirection ${directionAttributeValue} added');
        }
        await exif.saveAttributes();

        if (accelerometerValues != null) {
          Logger.getInstance().debug('Adding XMP tag', '${(currentDirection! * 100).truncate()}/100');
          // double yaw = 0, pitch = 0, roll = 0;
          // pitch = CaptureService.radiansToDegrees(atan2(accelerometerValues.x, sqrt(accelerometerValues.y * accelerometerValues.y + accelerometerValues.z * accelerometerValues.z)));
          // roll = CaptureService.radiansToDegrees(atan2(-accelerometerValues.y, accelerometerValues.z));
          // addXMPToImage(rawImage.path, yaw, pitch, roll);
          Logger.getInstance().debug('XMP tag added');
        }
      }

    await debugPrintImageExifTags(rawImage);
  }

  static String getPictureFilename(DateTime captureTime) => 'PX_${filenameDateFormat.format(captureTime)}.jpg';

  static void addXMPToImage(String imagePath, double yaw, double pitch, double roll) {
    String xmpData = '''
<?xpacket begin="" id="W5M0MpCehiHzreSzNTczkc9d"?>
<x:xmpmeta xmlns:x="adobe:ns:meta/">
  <rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#">
    <rdf:Description rdf:about="" xmlns:Camera="http://pix4d.com/camera/1.0" >
      <Camera:Yaw>${yaw.toStringAsFixed(2)}</Camera:Yaw>
      <Camera:Pitch>${pitch.toStringAsFixed(2)}</Camera:Pitch>
      <Camera:Roll>${roll.toStringAsFixed(2)}</Camera:Roll>
    </rdf:Description>
  </rdf:RDF>
</x:xmpmeta>
<?xpacket end="w"?>
''';

    File imageFile = File(imagePath);
    List<int> imageBytes = imageFile.readAsBytesSync();

    // Add XMP official header
    String xmpHeader = 'http://ns.adobe.com/xap/1.0/\u0000';

    // Convert XMP into bytes
    List<int> xmpBytes = utf8.encode(xmpHeader + xmpData);

    List<int> app1Header = [0xFF, 0xE1];
    int length = xmpBytes.length + 2;
    app1Header.add((length >> 8) & 0xFF);
    app1Header.add(length & 0xFF);

    // Add XMP after JPEG Markor (FF D8)
    List<int> modifiedImage = [];
    modifiedImage.addAll(imageBytes.sublist(0, 2));
    modifiedImage.addAll(app1Header);
    modifiedImage.addAll(xmpBytes);
    modifiedImage.addAll(imageBytes.sublist(2));

    File(imagePath).writeAsBytesSync(modifiedImage);
  }

  static Future<void> debugPrintImageExifTags(XFile rawImage) async {
    final exif2 = await Exif.fromPath(rawImage.path);
    final attributes = await exif2.getAttributes();
    var GPSImgDirection = await exif2.getAttribute('GPSImgDirection');
    var GPSAltitude = await exif2.getAttribute('GPSAltitude');
    var xmp = await exif2.getAttribute('Xmp');
    Logger.getInstance().info('file: ${rawImage}, all known attributes: ${attributes.toString()}, GPSImgDirection: ${GPSImgDirection}, GPSAltitude: ${GPSAltitude}, xmp: ${xmp}');
  }

  static PhotoGpsInterpolationResult interpolatePhoto({
    required List<PhotoToBeRelocated> photos,
    required GpsCourse gpsCourse,
    required DateTime lastInterpolationDate,
  }) {
    final result = PhotoGpsInterpolationResult(items: []);
    final intervals = CaptureService.getInterpolationIntervals(
      gpsCourse,
      lastInterpolationDate
    );

    result.items.addAll(
      intervals.expand((interval) => getPhotoToInterpolateGPS(
        photos: photos,
        before: interval.$1,
        after: interval.$2
      ).map((photo) => PhotoGpsInterpolationResultItem (
          newLocation: interpolateBetweenGpsCaptures(
            photoTime: photo.captureTime,
            before: interval.$1,
            after: interval.$2
          ),
          photo: photo..hasBeenRelocated = true
        ))
      )
    );

    return result;
  }

  static List<(GpsCapture, GpsCapture)> getInterpolationIntervals(GpsCourse course, DateTime fromDate) {
    final captures = course.captures;

    captures.sort((a, b) => a.date.compareTo(b.date));

    final intervals = <(GpsCapture, GpsCapture)>[];

    for (int i = 0; i < captures.length - 1; i++) {
      final current = captures[i];
      final next = captures[i + 1];

      if (current.date.isAfter(fromDate) ||
          current.date.isAtSameMomentAs(fromDate) ||
          (current.date.isBefore(fromDate) && fromDate.isBefore(next.date))
      ) {
        intervals.add((current, next));
      }
    }

    return intervals;
  }

  static List<PhotoToBeRelocated> getPhotoToInterpolateGPS({
    required List<PhotoToBeRelocated> photos,
    required GpsCapture before,
    required GpsCapture after,
  }) => photos
        .where((photo) => !photo.hasBeenRelocated &&
          !photo.captureTime.isBefore(before.date) &&
          !photo.captureTime.isAfter(after.date) &&
          photo.mustInterpolatePosition)
        .toList();

  static List<PhotoToBeRelocated> getPhotoToLocateWithoutInterpolation({required List<PhotoToBeRelocated> photos}) =>
      photos.where(
        (photo) => !photo.hasBeenRelocated
          && !photo.mustInterpolatePosition
      )
      .toList();

  static GpsLocation interpolateBetweenGpsCaptures({
    required DateTime photoTime,
    required GpsCapture before,
    required GpsCapture after,
  }) {
    final t1 = before.date.millisecondsSinceEpoch;
    final t2 = after.date.millisecondsSinceEpoch;
    final tp = photoTime.millisecondsSinceEpoch;

    if (t1 == t2) return before.location;

    final ratio = (tp - t1) / (t2 - t1);

    final lat1 = before.location.latitude;
    final lon1 = before.location.longitude;
    final lat2 = after.location.latitude;
    final lon2 = after.location.longitude;
    final alt1 = after.location.altitude;
    final alt2 = after.location.altitude;

    return GpsLocation.fromMap({
      'latitude': lat1 + (lat2 - lat1) * ratio,
      'longitude': lon1 + (lon2 - lon1) * ratio,
      'altitude': alt1 + (alt2 - alt1) * ratio,
      'timestamp': photoTime.millisecondsSinceEpoch
    });
  }
}

class PictureFromCamera {
  final DateTime captureTime;
  final XFile file;

  PictureFromCamera({required this.captureTime, required this.file});
}

class PhotoGpsInterpolationResult {
  final List<PhotoGpsInterpolationResultItem> items;

  PhotoGpsInterpolationResult({required this.items});

  @override
  String toString() => 'PhotoGpsInterpolationResult{items: $items}';
}

class PhotoGpsInterpolationResultItem {
  final PhotoToBeRelocated photo;
  final GpsLocation newLocation;

  PhotoGpsInterpolationResultItem({required this.photo, required this.newLocation});

  @override
  String toString() => 'PhotoGpsInterpolationResultItem{photo: $photo, newLocation: $newLocation}';

  @override
  bool operator ==(Object other) =>
      identical(this, other) ||
      other is PhotoGpsInterpolationResultItem &&
          runtimeType == other.runtimeType &&
          photo == other.photo &&
          newLocation == other.newLocation;

  @override
  int get hashCode => photo.hashCode ^ newLocation.hashCode;
}

class PhotoToBeRelocated {
  final String filePath;
  final DateTime captureTime;
  final double cameraDirection;
  final CalibrationStatus calibrationStatus;
  bool hasBeenRelocated = false;
  bool mustInterpolatePosition;
  final GpsLocation initialPosition;

  PhotoToBeRelocated({
    required this.filePath,
    required this.captureTime,
    required this.cameraDirection,
    required this.mustInterpolatePosition,
    required this.initialPosition,
    required this.calibrationStatus
  });

  @override
  String toString() => 'PhotoToBeRelocated{filePath: $filePath, captureTime: $captureTime, cameraDirection: $cameraDirection, hasBeenRelocated: $hasBeenRelocated, mustInterpolatePosition: $mustInterpolatePosition, initialPosition: $initialPosition}, calibrationStatus: ${calibrationStatus}';
}

class GpsCourse {
  final List<GpsCapture> captures = [];

  @override
  String toString() => 'GpsCourse{captures: ${captures.toString()}}';
}

class GpsCapture {
  final GpsLocation location;
  final DateTime date;

  GpsCapture({required this.location, required this.date});

  @override
  String toString() => 'GpsCapture{location: $location, date: $date}';
}

class AvailableCaptorsData {
  final double direction;
  final GyroscopeEvent accelerometer;
  final GpsLocation location;
  final CalibrationStatus calibrationStatus;

  AvailableCaptorsData({
    required this.direction,
    required this.accelerometer,
    required this.location,
    required this.calibrationStatus
  });

  @override
  String toString() => 'AvailableCaptorsData{direction: ${direction.toString()}, accelerometer: ${accelerometer.toString()}, location: ${location.toString()}, calibrationStatus: ${calibrationStatus.toString()}}';
}
