part of panoramax;

DateFormat filenameDateFormat = DateFormat('yyyyMMdd_HHmmssSSS');

abstract class ICapturePage {
  Widget createSequenceButton(BuildContext context);

  Widget showSequenceButton(BuildContext context);

  Widget captureButton(AvailableCaptorsData? captorsData);

  List<Widget> actionButtons(AvailableCaptorsData? captorsData);

  List<Widget> captureModeButtons();
}

class CaptureButton {
  final Icon icon;
  final Color backgroundColor;
  final dynamic onPressedAction;

  CaptureButton({required this.icon,
    required this.backgroundColor,
    required this.onPressedAction});
}

class CapturePage extends StatefulWidget {
  const CapturePage({Key? key, required this.cameras, this.sequenceToEditId})
      : super(key: key);

  final List<CameraDescription>? cameras;
  final int? sequenceToEditId;

  @override
  State<CapturePage> createState() => _CapturePageState();

  @override
  void debugFillProperties(DiagnosticPropertiesBuilder properties) {
    super.debugFillProperties(properties);
    properties.add(IterableProperty<CameraDescription>('cameras', cameras));
    properties.add(IntProperty('sequenceToEditId', sequenceToEditId));
  }
}

class _CapturePageState extends State<CapturePage>
    with WidgetsBindingObserver
    implements ICapturePage {

  final List<AppLifecycleState> _stateHistoryList = <AppLifecycleState>[];
  late CameraController _cameraController;
  bool _isProcessing = false;
  // bool _isPermissionDialogOpen = false;
  SequenceDto? _sequenceToCreate;

  final Duration _relocationTimer = Duration(milliseconds: 500);
  bool _isBurstMode = false;
  bool _isBurstPlay = false;
  Timer? _timerImageLocator;
  CaptureMode captureMode = CaptureMode.SINGLE;
  AvailableCaptorsData? _availableCaptorsData;
  GpsCourse _gpsCourse = new GpsCourse();
  List<PhotoToBeRelocated> _photosToBeRelocated = [];
  DateTime? _lastInterpolatedTimestamp;
  late Stream<AvailableCaptorsData> _allPageStreamsCombined;

  var isPortraitOrientation = true;

  bool _isFlashTurnOn = false;
  bool _reduceBrightnessOnCapture = true;
  bool _isSuspended = false;

  _CapturePageState() {
    _allPageStreamsCombined = Rx.combineLatest4(
      FlutterCompass.events!,
      gyroscopeEventStream(),
      GpsLocator.locationStream.doOnData((newLocation) {
        _gpsCourse.captures.add(
          GpsCapture(
            location: newLocation,
            date: newLocation.timestamp!
          )
        );
        Logger.getInstance().debug('New location captured : ${newLocation}');
      }),
      MagnetometerCalibrationService.calibrationStatusStream.doOnData((calibrationStatus) {
        Logger.getInstance().debug('New calibrationStatus : ${calibrationStatus}');
      }),
      (compass, accelerometer, location, calibrationStatus) => AvailableCaptorsData(
        direction: (360 + compass.heading!) % 360,
        accelerometer: accelerometer,
        location: location,
        calibrationStatus: calibrationStatus
      )
    );
  }

  @override
  void dispose() {
    Logger.getInstance().debug('Disposing cameraController');
    stopBurstPictures();
    _timerImageLocator?.cancel();
    _cameraController.dispose();
    WidgetsBinding.instance.removeObserver(this);
    WakelockPlus.disable();
    super.dispose();
  }

  @override
  initState() {
    _initAsync();

    super.initState();

    WidgetsBinding.instance.addObserver(this);
    if (WidgetsBinding.instance.lifecycleState != null) {
      _stateHistoryList.add(WidgetsBinding.instance.lifecycleState!);
    }

    _isFlashTurnOn = context
        .read<SettingsProvider>()
        .isFlashTurnOn;

    _reduceBrightnessOnCapture =
        context
            .read<SettingsProvider>()
            .reduceBrightnessOnCapture;

    if (widget.cameras?.isNotEmpty ?? false) {
      initCamera(widget.cameras![0]);
    }
  }

  Future<void> _initAsync() async {
    if (widget.sequenceToEditId != null) {
      _sequenceToCreate =
      await SequenceRepository.getSequence(widget.sequenceToEditId!);
    }
  }

  @override
  void didChangeAppLifecycleState(AppLifecycleState state) {
    if (state == AppLifecycleState.inactive) {
      _isSuspended = true;
      _cameraController.dispose();
    } else if (state == AppLifecycleState.resumed) {
      initCamera(widget.cameras![0], shouldRefeshState: false);
      _isSuspended = false;
    }
  }

  void goToCollectionCreationPage() async {
    await _cameraController.setFlashMode(FlashMode.off);
    GetIt.instance<NavigationService>().pushTo(Routes.newSequenceUpload,
      arguments: {
        'sequenceToCreateId': _sequenceToCreate!.id,
        'sendOnStart': false
      }
    );
  }

  Future createSequence() async {
    _sequenceToCreate = await CaptureService.createSequenceDto(context);
  }

  Future createSequenceAndTakePicture(AvailableCaptorsData captorsData) async {
    if (_sequenceToCreate == null) {
      await createSequence();
    }
    await _cameraController.setFocusMode(FocusMode.auto);
    await takePicture();
    await relocatePhotosWithoutInterpolation();
  }

  Future takePicture() async {
    if (_isBurstMode && !_isBurstPlay) {
      Logger.getInstance().debug('Burst mode have been stopped');
      await locateRemainingSequencePhotos();
      return;
    }

    setState(() {
      _isProcessing = true;
    });

    if (!_cameraController.value.isInitialized) {
      Logger.getInstance().debug('Camera is not intialized');
      return;
    }

    if (_cameraController.value.isTakingPicture) {
      Logger.getInstance().debug('Is already taking picture');
      return;
    }

    if (_availableCaptorsData == null) {
      Logger.getInstance().debug('No capture data available');
      return;
    }

    try {
      Logger.getInstance().info('takePicture requested');

      final PictureFromCamera pictureFromCamera = await getPictureFromCamera();

      Logger.getInstance().info(
        'All captors before capture: ' + _availableCaptorsData.toString()
      );

      setState(() {
        _isProcessing = false;
      });

      _photosToBeRelocated.add(
        PhotoToBeRelocated(
          filePath: pictureFromCamera.file.path,
          captureTime: pictureFromCamera.captureTime,
          cameraDirection: _availableCaptorsData!.direction,
          initialPosition: _availableCaptorsData!.location,
          mustInterpolatePosition: _isBurstMode,
          calibrationStatus: _availableCaptorsData!.calibrationStatus
        )
      );

      if (_isBurstMode) {
        takePicture();
      }

    } on CameraException catch (e) {
      Logger.getInstance().error('Error occurred while taking picture', e);
      return null;
    }
  }

  void takeBurstPictures(AvailableCaptorsData captorsData) async {
    if (_isBurstPlay) {
      Logger.getInstance().info('End burst mode');
      stopBurstPictures();
      await ScreenBrightness.instance.resetApplicationScreenBrightness();
    } else {
      if (_reduceBrightnessOnCapture) {
        ScreenBrightness.instance.setApplicationScreenBrightness(0.05);
      }
      Logger.getInstance().info('Start burst mode');
      startBurstPictures(captorsData);
    }
  }

  void stopBurstPictures() {
    if (_isBurstPlay) {
      try {
        setState(() {
          _isBurstPlay = false;
        });
      } on Exception catch(_) {
        Logger.getInstance().debug('An error occured during stopBurstPictures');
      }
    }
  }

  Future startBurstPictures(AvailableCaptorsData captorsData) async {
    await _cameraController.setFocusPoint(Offset(0.5, 0.5));
    await _cameraController.setFocusMode(FocusMode.locked);

    if (_sequenceToCreate == null) {
      await createSequence();
    }

    setState(() {
      _isBurstPlay = !_isBurstPlay;
    });

    var isProcessing = false;

    _timerImageLocator = buildImageLocatorTimer(isProcessing);

    await takePicture();
  }

  Timer buildImageLocatorTimer(bool isProcessing) => Timer.periodic(_relocationTimer, (timer) async {
    if(isProcessing) return;

    try {
      isProcessing = true;
      var lastGpsCaptureDate = _gpsCourse.captures.last.date;
      if (_lastInterpolatedTimestamp != null) {
        Logger.getInstance().info(
            'lastInterpolatedTimestamp: ${_lastInterpolatedTimestamp}, lastGpsCaptureDate: ${lastGpsCaptureDate}');

        List<PhotoGpsInterpolationResultItem> photosToRelocate = await relocatePhotosWithInterpolation();

        Logger.getInstance().info('${photosToRelocate.length} files relocated with interpolation');
        if (photosToRelocate.length > 0) {
          _lastInterpolatedTimestamp = lastGpsCaptureDate;
        }
      } else {
        _lastInterpolatedTimestamp = _gpsCourse.captures.first.date;
        Logger.getInstance().info('No interpolation yet');
      }
    } on Exception catch (_) {
      Logger.getInstance().info('No Gps capture yet');
    }

    if(_isBurstMode && !_isBurstPlay) {
      // Sequence have been stopped by user
      timer.cancel();
    }
    isProcessing = false;
  });

  Future<void> locateRemainingSequencePhotos() async {
    Logger.getInstance().debug('locateRemainingSequencePhotos');
    _photosToBeRelocated.where((photo) => photo.mustInterpolatePosition && !photo.hasBeenRelocated)
      .forEach((photo) => photo.mustInterpolatePosition = false);
    await relocatePhotosWithoutInterpolation();
    Logger.getInstance().debug('End of locateRemainingSequencePhotos');
  }

  Future<List<PhotoGpsInterpolationResultItem>> relocatePhotosWithInterpolation() async {
    Logger.getInstance().debug('relocatePhotosWithInterpolation');
    var photosToRelocate = CaptureService.interpolatePhoto(
      photos: _photosToBeRelocated,
      gpsCourse: _gpsCourse,
      lastInterpolationDate: _lastInterpolatedTimestamp!.subtract(
        Duration(seconds: 2)
      )
    ).items;

    for (var photoToRelocate in photosToRelocate) {
      Logger.getInstance().info('Photo to relocate: ${photoToRelocate}');

      await CaptureService.processImage(
        photoToRelocate.photo.filePath,
        photoToRelocate.newLocation,
        photoToRelocate.photo.cameraDirection,
        photoToRelocate.photo.calibrationStatus,
        _availableCaptorsData!.accelerometer
      );

      Logger.getInstance().info(
          'Photo ${photoToRelocate.photo.filePath} relocated to location ${photoToRelocate
              .newLocation}');
      addImageToList(photoToRelocate.photo);
    }

    Logger.getInstance().debug('end of relocatePhotosWithInterpolation');
    return photosToRelocate;
  }

  Future<void> relocatePhotosWithoutInterpolation() async {
    Logger.getInstance().debug('relocatePhotosWithoutInterpolation');
    for (var photoToLocate in CaptureService.getPhotoToLocateWithoutInterpolation(photos: _photosToBeRelocated)) {
      Logger.getInstance().info('Photo to locate: ${photoToLocate}');

      await CaptureService.processImage(
        photoToLocate.filePath,
        photoToLocate.initialPosition,
        photoToLocate.cameraDirection,
        photoToLocate.calibrationStatus,
        _availableCaptorsData!.accelerometer
      );

      photoToLocate.hasBeenRelocated = true;

      Logger.getInstance().info(
          'Photo ${photoToLocate.filePath} located to location ${photoToLocate
              .initialPosition}');
      addImageToList(photoToLocate);
    }
    Logger.getInstance().debug('end of relocatePhotosWithoutInterpolation');
  }

  Future<void> addImageToList(PhotoToBeRelocated photo) async {
    final sequenceToCreate =  await CaptureService.addImageSequence(
        _sequenceToCreate!,
        photo.filePath
    );
    _sequenceToCreate = sequenceToCreate;
    Logger.getInstance().info('photosToBeRelocated: ${_photosToBeRelocated}');
    Logger.getInstance().info('Image ${photo} added local list');
  }

  Future<PictureFromCamera> getPictureFromCamera() async {
    Logger.getInstance().info('_cameraController.takePicture');
    final XFile rawImage = await _cameraController.takePicture();
    Logger.getInstance().info('StorageService.getImageFolder');
    final DateTime captureTime = DateTime.now();
    final panoramaxImageRootFolder = await StorageService.getImageFolder();
    final newPath = path.join(
      panoramaxImageRootFolder.path,
      CaptureService.getPictureFilename(captureTime)
    );
    Logger.getInstance().info('Saving image as ${newPath}');
    await rawImage.saveTo(newPath);
    Logger.getInstance().info('Image saved as ${newPath}');
    return PictureFromCamera(
      file: XFile(newPath),
      captureTime: captureTime
    );
  }

  Future initCamera(CameraDescription cameraDescription,
      {bool shouldRefeshState = true}) async {
    _cameraController = CameraController(
      cameraDescription,
      ResolutionPreset.max,
      enableAudio: false,
    );
    try {
      await _cameraController.initialize().then((_) async {
        if (!mounted) return;
        if (shouldRefeshState) {
          setState(() {});
        }
      });
      await setCameraFlashMode();
    } on CameraException catch (e) {
      Logger.getInstance().debug('camera error $e');
    }
  }

  @override
  Widget build(BuildContext context) {
    WakelockPlus.enable();
    if (widget.cameras?.isEmpty ?? true) {
      return Scaffold(
          key: GlobalKey<ScaffoldState>(),
          body: Center(
            child: Text(AppLocalizations.of(context)!.noCameraFoundError),
          ));
    }
    return StreamBuilder<AvailableCaptorsData>(
      stream: _allPageStreamsCombined,
      builder: (context, snapshot) {
        _availableCaptorsData = snapshot.data;
        return PopScope(
          onPopInvokedWithResult: (didPop, result) {
            OrientationService.allowOnlyPortrait();
          },
          child: OrientationBuilder(
            builder: (context, orientation) {
              isPortraitOrientation = orientation == Orientation.portrait;
              return Stack(children: [
                Positioned.fill(
                  child: Container(
                    color: Theme.of(context)
                      .colorScheme
                      .primaryContainer,
                  ),
                ),
                Padding(
                  padding: EdgeInsets.only(top: isPortraitOrientation ? 60 : 0, bottom: isPortraitOrientation ? 170 : 0),
                  child: cameraPreview()
                ),
                accuracyComponent(_availableCaptorsData),
                // FIXME SSE manage only one layout with icon, text and component rotate
                (!isPortraitOrientation)
                    ? landscapeLayout(context, snapshot.data)
                    : portraitLayout(context, snapshot.data),
                if (_isProcessing && !_isBurstMode) processingLoader(context),
                if (!CaptureService.isGpsAccuracyGood(_availableCaptorsData?.location.accuracy))
                  alertDialogGpsAccurency(),
                if (CaptureService.isGpsAccuracyGood(_availableCaptorsData?.location.accuracy) && !CaptureService.isPhoneCalibrationGood(_availableCaptorsData?.calibrationStatus))
                  alertDialogCalibration(),
              ]);
            },
          ),
        );
      }
    );
  }

  Widget accuracyComponent(AvailableCaptorsData? captorsData) => _availableCaptorsData?.location.accuracy == null
        ? Container()
        : Padding(
        padding: EdgeInsets.only(left: 32, top: 10),
        child: Container(
            decoration: BoxDecoration(
              color: Theme
                  .of(context)
                  .cardTheme.color,
              borderRadius: BorderRadius.all(Radius.circular(8)),
            ),
            padding: EdgeInsets.all(8.0),
            child: Row(mainAxisSize: MainAxisSize.min, children: [
              Icon(
                !CaptureService.isGpsAccuracyGood(_availableCaptorsData?.location.accuracy)
                    ? Icons.error
                    : Icons.gps_fixed,
                size: 24,
                color: !CaptureService.isGpsAccuracyGood(_availableCaptorsData?.location.accuracy)
                    ? Colors.red
                    : Theme
                    .of(context)
                    .colorScheme
                    .primary,
              ),
              SizedBox(width: 2),
              DefaultTextStyle(
                style: TextStyle(
                    color: !CaptureService.isGpsAccuracyGood(_availableCaptorsData?.location.accuracy)
                        ? Colors.red
                        : Theme
                        .of(context)
                        .colorScheme
                        .primary),
                child: Text('${AppLocalizations.of(context)!
                    .meters(_availableCaptorsData!.location.accuracy!.toInt() as Object)}'
                ),
              ),
              SizedBox(width: 10),
              Icon(
                !CaptureService.isPhoneCalibrationGood(_availableCaptorsData?.calibrationStatus)
                    ? Icons.error
                    : Icons.navigation,
                size: 24,
                color: !CaptureService.isPhoneCalibrationGood(_availableCaptorsData?.calibrationStatus)
                    ? Colors.red
                    : Theme
                    .of(context)
                    .colorScheme
                    .primary,
              ),
              SizedBox(width: 0),
              DefaultTextStyle(
                style: TextStyle(
                    color: !CaptureService.isPhoneCalibrationGood(_availableCaptorsData?.calibrationStatus)
                        ? Colors.red
                        : Theme
                        .of(context)
                        .colorScheme
                        .primary),
                child: Text('${_availableCaptorsData!.direction.toStringAsFixed(2)}°'
                ),
              ),
            ])));

  Widget portraitLayout(BuildContext context, AvailableCaptorsData? captorsData) {
    var actionBar = Row(
        mainAxisAlignment: MainAxisAlignment.spaceAround,
        children: actionButtons(captorsData));
    var captureModesBar = Padding(
      padding: EdgeInsets.only(left: 16, right: 16),
      child: Row(
          mainAxisAlignment: MainAxisAlignment.spaceBetween,
          children: captureModeButtons()),
    );
    return Stack(
      children: [
        _isBurstMode
            ? BackdropFilter(
          filter: ImageFilter.blur(sigmaX: 10, sigmaY: 10),
          child: Container(color: Colors.transparent),
        )
            : Container(),
        Center(
          child: _isBurstMode ? askTurnDevice() : Container(),
        ),
        Align(
          alignment: Alignment.bottomCenter,
          child: Column(
              mainAxisSize: MainAxisSize.min,
              spacing: 20.0,
              children: [actionBar, captureModesBar, subActionButtons()]),
        ),
      ],
    );
  }

  List<Widget> defaultBurstBar(AvailableCaptorsData? captorsData) => [
      Container(height: !isPortraitOrientation ? 80 : 0),
      (!_isBurstMode || !isPortraitOrientation) ? captureButton(captorsData) : Container(),
      Container(
        height: 70,
      ),
    ];

  List<Widget> fullBurstBar(AvailableCaptorsData? captorsData) => [
      badges.Badge(
          position: badges.BadgePosition.bottomEnd(),
          badgeStyle: badges.BadgeStyle(
              badgeColor: Theme
                  .of(context)
                  .colorScheme
                  .secondary),
          badgeContent: Text('${_sequenceToCreate!.pictures.length}'),
          child: showSequenceButton(context)),
      (!_isBurstMode || !isPortraitOrientation) ? captureButton(captorsData) : Container(),
      createSequenceButton(context)
    ];

  Widget askTurnDevice() => Column(
        mainAxisAlignment: MainAxisAlignment.center,
        crossAxisAlignment: CrossAxisAlignment.center,
        children: [
          Icon(Icons.screen_rotation, size: 120.0, color: Colors.white),
          Padding(
              padding: EdgeInsets.all(50),
              child: Card(
                  child: Padding(
                      padding: EdgeInsets.all(16),
                      child: Row(children: [
                        Icon(
                          Icons.info_outline,
                          color: Theme
                              .of(context)
                              .colorScheme
                              .secondary,
                        ),
                        Expanded(
                            child: Padding(
                                padding: EdgeInsets.only(left: 8),
                                child: Text(
                                    AppLocalizations.of(context)!
                                        .switchCameraRequired,
                                    style: TextStyle(
                                      fontWeight: FontWeight.bold,
                                      color: Theme
                                          .of(context)
                                          .colorScheme
                                          .secondary,
                                      fontSize: 16,
                                    )))),
                      ]))))
        ]);

  Widget landscapeLayout(BuildContext context, AvailableCaptorsData? captorsData) {
    var actionBar = Container(
        padding: EdgeInsets.only(bottom: 24, top: 24, left: 24, right: 58),
        width: MediaQuery
            .of(context)
            .size
            .width,
        child: Container(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.end,
              mainAxisAlignment: MainAxisAlignment.spaceAround,
              children: actionButtons(captorsData),
            )));
    Widget createBurstBar() => Container(
          padding: EdgeInsets.all(24),
          height: MediaQuery
              .of(context)
              .size
              .height,
          child: Row(
            crossAxisAlignment: CrossAxisAlignment.end,
            children: captureModeButtons(),
          ));

    return Container(
        margin: EdgeInsets.only(left: 30, right: 30),
        width: MediaQuery
            .of(context)
            .size
            .width,
        child: Stack(
            children: [createBurstBar(), actionBar, subActionButtons()]));
  }

  bool picturesListIsNotEmpty() =>
      _sequenceToCreate != null && _sequenceToCreate!.pictures.isNotEmpty;

  StatelessWidget cameraPreview() {
    if (!_isSuspended && _cameraController.value.isInitialized) {
      return Container(
          alignment: Alignment.center, child: CameraPreview(_cameraController));
    } else {
      return Container(
        alignment: Alignment.center,
        color: Colors.transparent,
        child: const Center(child: CircularProgressIndicator()),
      );
    }
  }

  Positioned processingLoader(BuildContext context) => Positioned(
      top: 0,
      bottom: 0,
      left: 0,
      right: 0,
      child: Loader(
        message: DefaultTextStyle(
          style: Theme
              .of(context)
              .textTheme
              .bodyLarge!,
          child: Container(),
        )
      ),
    );

  Widget alertDialogGpsAccurency() => AlertDialog(
      title: Text(AppLocalizations.of(context)!.lowGpsAccuracyTitle),
      content: Text(AppLocalizations.of(context)!.lowGpsAccuracyDesc),
      backgroundColor: Theme
          .of(context)
          .colorScheme
          .primary,
      titleTextStyle: TextStyle(color: Theme
          .of(context)
          .colorScheme
          .onPrimaryContainer),
      contentTextStyle: TextStyle(color: Theme
          .of(context)
          .colorScheme
          .onPrimaryContainer),
    );

  Widget alertDialogCalibration() => AlertDialog(
    content: Column(
      mainAxisSize: MainAxisSize.min,
      children: [
        Image.asset(
          'assets/animation/recalibration_needed.gif',
          width: 300,
          height: 160,
          fit: BoxFit.contain,
        ),
        Card(
          child: Padding(
            padding: EdgeInsets.all(16),
            child: Row(children: [
              Icon(
                Icons.info_outline,
                color: Theme.of(context).colorScheme.secondary,
              ),
              Expanded(
                  child: Padding(
                      padding: EdgeInsets.only(left: 8),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Text(
                              AppLocalizations.of(context)!
                                  .calibrationRequiredTitle,
                              style: TextStyle(
                                fontWeight: FontWeight.bold,
                                color: Theme
                                    .of(context)
                                    .colorScheme
                                    .secondary,
                                fontSize: 16,
                              )
                          ),
                          Text(
                              AppLocalizations.of(context)!
                                  .calibrationInstruction,
                              style: TextStyle(
                                fontWeight: FontWeight.normal,
                                color: Theme
                                    .of(context)
                                    .colorScheme
                                    .secondary,
                                fontSize: 16,
                              )
                          )
                        ],
                      )
                  )
              ),
            ])
          )
        ),
        // Text(AppLocalizations.of(context)!.calibrationRequiredDesc),
      ],
    ),
    backgroundColor: Colors.transparent,
    );

  Future<void> showGPSDialog() async => showDialog<void>(
      context: context,
      builder: (context) => AlertDialog(
          title: Text(AppLocalizations.of(context)!.gpsIsDisableTitle),
          content: SingleChildScrollView(
            child: ListBody(
              children: <Widget>[
                Text(AppLocalizations.of(context)!.gpsIsDisableContent),
              ],
            ),
          ),
          actions: <Widget>[
            TextButton(
              child: Text(AppLocalizations.of(context)!.common_close),
              onPressed: () {
                Navigator.of(context).pop();
              },
            ),
          ],
        ),
    );

  Future<void> showPermissionDialog() async => showDialog<void>(
      context: context,
      barrierDismissible: false, // user must tap button!
      builder: (context) => AlertDialog(
          title: Text(AppLocalizations.of(context)!.common_permissionDenied),
          content: SingleChildScrollView(
            child: ListBody(
              children: <Widget>[
                Text(AppLocalizations.of(context)!.permissionLocationRequired),
              ],
            ),
          ),
          actions: <Widget>[
            TextButton(
              child: Text(AppLocalizations.of(context)!.common_close),
              onPressed: () {
                Navigator.of(context).pop();
                // _isPermissionDialogOpen = false;
              },
            ),
            TextButton(
              child: Text(AppLocalizations.of(context)!.goToSettings),
              onPressed: () {},
            ),
          ],
        ),
    );

  @override
  Widget createSequenceButton(BuildContext context) => IconButton(
        padding: EdgeInsets.zero,
        iconSize: 30,
        icon: const Icon(Icons.send_outlined, color: Colors.white),
        onPressed: goToCollectionCreationPage,
        tooltip:
        AppLocalizations.of(context)!.createSequenceWithPicture_tooltip);

  @override
  Widget showSequenceButton(BuildContext context) => IconButton(
        padding: EdgeInsets.only(left: !isPortraitOrientation ? 10.0 : 0),
        onPressed: goToCollectionCreationPage,
        icon: Wrap(
          spacing: -55,
          children: _sequenceToCreate!.pictures
              .map((photo) => Container(
            height: 60,
            width: 60,
            decoration: BoxDecoration(
              border: Border.all(
                color: Theme
                    .of(context)
                    .colorScheme
                    .onPrimaryContainer,
                width: 1),
              borderRadius: BorderRadius.all(Radius.circular(10)),
            ),
            child: picturesListIsNotEmpty()
              ? ClipRRect(
                borderRadius: BorderRadius.circular(10),
                child: Image.file(
                  File(photo.localFilePath),
                  fit: BoxFit.cover,
                  cacheWidth: 60,
                  cacheHeight: 60,
                  )
                )
              : null,
            )
          )
              .skip(_sequenceToCreate!.pictures.length > 3
              ? _sequenceToCreate!.pictures.length - 3
              : 0)
              .toList(),
        ));

  CaptureButton gpsNotGrantedButton() => CaptureButton(
    icon: Icon(
      Icons.gps_off,
      color: Colors.red,
      size: 35,
    ),
    backgroundColor: Colors.grey,
    onPressedAction: (){}
  );

  CaptureButton gpsInAcquisitionButton() => CaptureButton(
    icon: Icon(
      Icons.not_listed_location,
      color: Colors.red,
      size: 35,
    ),
    backgroundColor: Colors.grey,
    onPressedAction: (){}
  );

  CaptureButton gpsNotPreciseButton(bool isBurstPlay) => CaptureButton(
    icon: Icon(
      Icons.gps_fixed,
      color: Colors.red,
      size: 35,
    ),
    backgroundColor: Colors.grey,
    onPressedAction: () {
      if (isBurstPlay) {
        stopBurstPictures();
      }
    }
  );

  CaptureButton sequenceInProgressButton(AvailableCaptorsData captorsData) => CaptureButton(
    icon: Icon(
      Icons.stop,
      color: Colors.white,
      size: 40,
    ),
    backgroundColor: Colors.red,
    onPressedAction: () => takeBurstPictures(captorsData)
  );

  CaptureButton sequenceAvailableButton(AvailableCaptorsData captorsData) => CaptureButton(
    icon: Icon(
      Icons.circle,
      // FIXME add #1b4d71 to theme and apply it
      color: Colors.white,
      size: 50,
    ),
    backgroundColor: Theme
        .of(context)
        .colorScheme
        .secondary,
    onPressedAction: () => takeBurstPictures(captorsData)
  );

  CaptureButton singlePhotoInProgressButton() => CaptureButton(
    icon: Icon(
      Icons.circle,
      color: Colors.grey,
      size: 50,
    ),
    backgroundColor: Colors.grey,
    onPressedAction: () => {}
  );

  CaptureButton singlePhotoAvailableButton(AvailableCaptorsData captorsData) => CaptureButton(
    icon: Icon(
      Icons.circle,
      color: Theme
          .of(context)
          .colorScheme
          .secondary,
      size: 50,
    ),
    backgroundColor: Colors.white,
    onPressedAction: () => createSequenceAndTakePicture(captorsData)
  );

  CaptureButton getCaptureButton(CaptureButtonState state, AvailableCaptorsData? captorsData) => switch (state) {
      CaptureButtonState.SINGLE_PHOTO_AVAILABLE =>
          this.singlePhotoAvailableButton(captorsData!),
      CaptureButtonState.SEQUENCE_PHOTO_AVAILABLE =>
          this.sequenceAvailableButton(captorsData!),
      CaptureButtonState.SINGLE_PHOTO_IN_PROGRESS =>
          this.singlePhotoInProgressButton(),
      CaptureButtonState.SEQUENCE_PHOTO_IN_PROGRESS =>
          this.sequenceInProgressButton(captorsData!),
      CaptureButtonState.GPS_NOT_GRANTED => this.gpsNotGrantedButton(),
      CaptureButtonState.GPS_IN_ACQUISITION => this.gpsInAcquisitionButton(),
      CaptureButtonState.GPS_NOT_PRECISE => this.gpsNotPreciseButton(_isBurstPlay),
    };

  @override
  Widget captureButton(AvailableCaptorsData? captorsData) {
    CaptureButtonState captureButtonState =
    CaptureService.getCaptureButtonState(
      captorsData,
      _isBurstMode,
      _isBurstPlay,
      _isProcessing
    );
    CaptureButton captureButton = getCaptureButton(captureButtonState, captorsData);
    return GestureDetector(
      child: IconButton(
          onPressed: captureButton.onPressedAction,
          iconSize: 100,
          padding: EdgeInsets.zero,
          constraints: const BoxConstraints(),
          icon: Container(
            height: 60,
            width: 60,
            decoration: BoxDecoration(
                shape: BoxShape.circle, color: captureButton.backgroundColor),
            child: captureButton.icon,
          ),
          tooltip: AppLocalizations.of(context)!.capture_tooltip),
    );
  }

  @override
  List<Widget> actionButtons(AvailableCaptorsData? captorsData) => picturesListIsNotEmpty() ? fullBurstBar(captorsData) : defaultBurstBar(captorsData);

  Widget subActionButtons() => Padding(padding: EdgeInsets.only(bottom: 8),
        child: Row(mainAxisAlignment: MainAxisAlignment.spaceEvenly, children: [
          AtomicIconButton.switchAtPressed(
            tooltip: AppLocalizations.of(context)!.settings_sensors_brightness,
            selectedIcon: Icons.flash_on,
            icon: Icons.flash_off,
            isSelected: _isFlashTurnOn,
            callback: () async {
              setState(() {
                _isFlashTurnOn = !_isFlashTurnOn;
              });
              await setCameraFlashMode();
            },
          ),

          // Second AtomicIconButton widget
          AtomicIconButton.switchAtPressed(
            tooltip: AppLocalizations.of(context)!.settings_sensors_flash,
            selectedIcon: Icons.brightness_low_sharp,
            icon: Icons.brightness_high,
            isSelected: _reduceBrightnessOnCapture,
            callback: () async {
              setState(() {
                _reduceBrightnessOnCapture = !_reduceBrightnessOnCapture;
              });
            },
          ),
        ]));

  Future<void> setCameraFlashMode() async {
    await _cameraController.setFlashMode(
      _isFlashTurnOn ? FlashMode.torch : FlashMode.off,
    );
  }

  @override
  List<Widget> captureModeButtons() => [
      Expanded(
          child: Row(mainAxisAlignment: MainAxisAlignment.center, children: [
            Container(
                width: 300,
                child: SegmentedButton<CaptureMode>(
                  segments: <ButtonSegment<CaptureMode>>[
                    ButtonSegment<CaptureMode>(
                        value: CaptureMode.SINGLE,
                        label:
                        Text(AppLocalizations.of(context)!.photo.toUpperCase()),
                        icon: Icon(Icons.insert_photo),
                        tooltip: AppLocalizations.of(context)!
                            .singleModeCapture_tooltip),
                    ButtonSegment<CaptureMode>(
                        value: CaptureMode.BURST,
                        label: Text(
                            AppLocalizations.of(context)!.sequence
                                .toUpperCase()),
                        icon: Icon(Icons.burst_mode),
                        tooltip:
                        AppLocalizations.of(context)!.burstModeCapture_tooltip)
              ],
              selected: <CaptureMode>{captureMode},
              onSelectionChanged: (newSelection) async {
                setState(() {
                  captureMode = newSelection.first;
                  _isBurstMode = captureMode == CaptureMode.BURST;
                });
                Logger.getInstance().info('Selected mode: ${captureMode}');
                if (captureMode == CaptureMode.SINGLE) {
                  stopBurstPictures();
                  await _cameraController.setFocusMode(FocusMode.auto);
                }
              },
            )),
      ])),
    ];

  @override
  void debugFillProperties(DiagnosticPropertiesBuilder properties) {
    super.debugFillProperties(properties);
    properties.add(EnumProperty<CaptureMode>('captureMode', captureMode));
    properties.add(DiagnosticsProperty<bool>('isPortraitOrientation', isPortraitOrientation));
  }
}