package com.panoramax.app

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.location.Location
import android.location.LocationListener
import android.location.LocationManager
import android.os.Bundle
import android.util.Log
import io.flutter.plugin.common.EventChannel

class GpsLocationHelper(
    private val context: Context
) : EventChannel.StreamHandler {

    private lateinit var locationManager: LocationManager
    private var eventSink: EventChannel.EventSink? = null

    private val locationListener = object : LocationListener {
        override fun onLocationChanged(location: Location) {
            sendLocation(location)
        }

        override fun onStatusChanged(provider: String?, status: Int, extras: Bundle?) {}
        override fun onProviderEnabled(provider: String) {}
        override fun onProviderDisabled(provider: String) {}
    }

    override fun onListen(arguments: Any?, events: EventChannel.EventSink?) {
        eventSink = events
        locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager

        if (context.checkSelfPermission(Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            eventSink?.error("PERMISSION_DENIED", "Permission denied", null)
            return
        }

        val lastGps = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER)
        val lastNetwork = locationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER)

        val bestLastLocation = when {
            lastGps != null && lastNetwork != null ->
                if (lastGps.accuracy <= lastNetwork.accuracy) lastGps else lastNetwork
            lastGps != null -> lastGps
            lastNetwork != null -> lastNetwork
            else -> null
        }

        bestLastLocation?.let {
            sendLocation(it)
        }

        locationManager.requestLocationUpdates(
            LocationManager.GPS_PROVIDER,
            100L, // 100 millisecond
            0f,
            locationListener
        )

        locationManager.requestLocationUpdates(
            LocationManager.NETWORK_PROVIDER,
            1000L,
            0f,
            locationListener
        )
    }

    private fun sendLocation(location: Location) {
        val currentTimeStamp = System.currentTimeMillis()
        val data = mapOf(
            "latitude" to location.latitude,
            "longitude" to location.longitude,
            "accuracy" to location.accuracy.toDouble(),
            "altitude" to location.altitude,
            "timestamp" to currentTimeStamp,
        )
        Log.d("GPS", "New location: $data")
        Log.d("GPS", "Delay from GPS fix: ${currentTimeStamp - location.time} ms")
        eventSink?.success(data)
    }

    override fun onCancel(arguments: Any?) {
        locationManager.removeUpdates(locationListener)
        eventSink = null
    }
}
