import type { PartialAtlasBundle, AtlasBundle, AtlasSource } from './types';
export type AtlasMetadata = {
    name: string;
    version: string;
};
export declare class AtlasFileSource implements AtlasSource {
    readonly filePath: string;
    constructor(filePath: string);
    listBundles(): Promise<PartialAtlasBundle[]>;
    getBundle(id: string): Promise<AtlasBundle>;
    hasHmrSupport(): boolean;
    getBundleHmr(): null;
}
/**
 * List all entries without parsing the data.
 * This only reads the bundle name, and adds a line number as ID.
 *
 * @note Ensure the (de)serialization is in sync with both {@link readAtlasEntry} and {@link writeAtlasEntry}.
 */
export declare function listAtlasEntries(filePath: string): Promise<PartialAtlasBundle[]>;
/**
 * Get the entry by id or line number, and parse the data.
 *
 * @note Ensure the (de)serialization is in sync with both {@link listAtlasEntries} and {@link writeAtlasEntry}.
 */
export declare function readAtlasEntry(filePath: string, id: number): Promise<AtlasBundle>;
/**
 * Wait until the Atlas file has all data written.
 * Note, this is a workaround whenever `process.exit` is required, avoid if possible.
 * @internal
 */
export declare function waitUntilAtlasFileReady(): Promise<any>;
/**
 * Add a new entry to the Atlas file.
 * This function also ensures the Atlas file is ready to be written to, due to complications with Expo CLI.
 * Eventually, the entry is appended on a new line, so we can load them selectively.
 *
 * @note Ensure the (de)serialization is in sync with both {@link listAtlasEntries} and {@link readAtlasEntry}.
 */
export declare function writeAtlasEntry(filePath: string, entry: AtlasBundle): Promise<any>;
/** The default location of the metro file */
export declare function getAtlasPath(projectRoot: string): string;
/** The information to validate if a file is compatible with this library version */
export declare function getAtlasMetdata(): AtlasMetadata;
/** Validate if the file is compatible with this library version */
export declare function validateAtlasFile(filePath: string, metadata?: AtlasMetadata): Promise<void>;
/**
 * Create or overwrite the file with basic metadata.
 * This metdata is used by the API to determine version compatibility.
 */
export declare function createAtlasFile(filePath: string): Promise<void>;
/**
 * Create the Atlas file if it doesn't exist, or recreate it if it's incompatible.
 */
export declare function ensureAtlasFileExist(filePath: string): Promise<boolean>;
//# sourceMappingURL=AtlasFileSource.d.ts.map