package app.flicky.data.remote

import android.content.Context
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import java.security.MessageDigest

private val Context.mirrorStore by preferencesDataStore("mirror_state")

/**
 * Saves last-good mirror per repo base so StickyLastGood persists across app restarts.
 */
class MirrorStateStore(private val context: Context) : MirrorRegistry.MirrorStateStore {

    private fun keyFor(base: String): Preferences.Key<String> {
        val norm = base.trim().lowercase()
        val digest = MessageDigest.getInstance("SHA-256").digest(norm.toByteArray())
        val hex = digest.joinToString("") { "%02x".format(it) }.take(16)
        return stringPreferencesKey("mirror_last_$hex")
    }

    override fun getLastGood(base: String): String? = runBlocking {
        val prefs = context.mirrorStore.data.first()
        prefs[keyFor(base)]
    }

    override fun putLastGood(base: String, url: String) {
        runBlocking {
            context.mirrorStore.edit { it[keyFor(base)] = url }
        }
    }

    override fun clear(base: String) {
        runBlocking {
            context.mirrorStore.edit { it.remove(keyFor(base)) }
        }
    }
}