package app.flicky.viewmodel

import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import app.flicky.data.external.UpdatesPreferences
import app.flicky.data.local.AppDao
import app.flicky.data.local.AppVariant
import app.flicky.data.model.FDroidApp
import app.flicky.data.repository.InstalledAppsRepository
import app.flicky.data.repository.SettingsRepository
import app.flicky.data.repository.PreferredRepo
import app.flicky.data.repository.VariantSelector
import app.flicky.install.Installer
import app.flicky.install.TaskStage
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.*
import kotlinx.coroutines.launch

data class DetailUiState(
    val app: FDroidApp? = null,
    val installedVersionCode: Long? = null,
    val isInstalling: Boolean = false,
    val progress: Float = 0f,
    val error: String? = null,
    val stage: TaskStage? = null,
    val variants: List<AppVariant> = emptyList(),
)

class AppDetailViewModel(
    private val dao: AppDao,
    private val installedRepo: InstalledAppsRepository,
    private val installer: Installer,
    private val settings: SettingsRepository,
    private val packageName: String
) : ViewModel() {

    private val _ui = MutableStateFlow(DetailUiState())
    val ui: StateFlow<DetailUiState> = _ui.asStateFlow()

    init {
        viewModelScope.launch {
            dao.observeOne(packageName).collect { app ->
                val installed = installedRepo.getVersionCode(packageName)
                val variants = runCatching { dao.variantsFor(packageName) }.getOrElse { emptyList() }
                _ui.value = _ui.value.copy(
                    app = app,
                    installedVersionCode = installed,
                    variants = variants.sortedByDescending { it.versionCode } // newest first
                )
            }
            viewModelScope.launch {
                installer.errors.collect { map ->
                    val msg = map[packageName]
                    if (!msg.isNullOrBlank()) {
                        _ui.update { it.copy(error = msg) }
                    }
                }
            }
        }
        viewModelScope.launch {
            installer.tasks
            .map { it[packageName] }
            .distinctUntilChanged()
            .onStart { emit(installer.tasks.value[packageName]) }
            .collect { stage -> when (stage) {
                    is TaskStage.Downloading -> _ui.update {
                        it.copy(
                            isInstalling = true,
                            stage = stage,
                            progress = (0.99f * stage.progress).coerceIn(0f, 0.99f),
                            error = null
                        )
                    }
                    is TaskStage.Verifying -> _ui.update {
                        it.copy(
                            isInstalling = true,
                            stage = stage,
                            progress = 0.995f,
                            error = null
                        )
                    }
                    is TaskStage.Installing -> _ui.update {
                        it.copy(
                            isInstalling = true,
                            stage = stage,
                            progress = (0.99f + 0.01f * stage.progress).coerceIn(0.99f, 1f),
                            error = null
                        )
                    }
                    is TaskStage.Cancelled -> _ui.update {
                        it.copy(
                            isInstalling = false,
                            stage = stage,
                            error = null
                        )
                    }
                    is TaskStage.Finished -> {
                        _ui.update {
                            it.copy(
                                isInstalling = false,
                                stage = stage,
                                progress = if (stage.success) 1f else it.progress,
                                error = if (stage.success) null else it.error
                            )
                        }
                        val newInstalled = installedRepo.getVersionCode(packageName)
                        _ui.update { it.copy(installedVersionCode = newInstalled) }
                    }
                    else -> { /* no-op */ }
                }
            }
        }
        viewModelScope.launch {
            installedRepo.packageNameChangesFlow().collect { changed ->
                if (changed == packageName) {
                    val newInstalled = installedRepo.getVersionCode(packageName)
                    _ui.update {
                        it.copy(
                            installedVersionCode = newInstalled,
                            isInstalling = false,
                            progress = if (newInstalled != null) 1f else it.progress,
                            error = null
                        )
                    }
                }
            }
        }
    }

    fun install() {
        val app = _ui.value.app ?: return

        viewModelScope.launch {
            _ui.value = _ui.value.copy(isInstalling = true, progress = 0f, error = null, stage = TaskStage.Downloading(0f))

            try {
                Log.d("AppDetailViewModel", "Starting install for ${app.packageName}")

                val prefIdx = settings.settingsFlow.first().preferredRepo
                val globalPref = PreferredRepo.fromIndex(prefIdx)
                val perAppPref = UpdatesPreferences[packageName]
                val variants = dao.variantsFor(app.packageName)

                val chosen = VariantSelector.pick(
                    variants = variants,
                    preferred = globalPref,
                    preferredRepoUrl = perAppPref.preferredRepoUrl,
                    strict = perAppPref.lockToRepo
                )

                val success = if (chosen != null) {
                    Log.d("AppDetailViewModel", "Installing via variant from ${chosen.repositoryName} (${chosen.repositoryUrl}) with vercode: ${chosen.versionCode}")
                    installer.install(chosen)
                } else {
                    Log.d("AppDetailViewModel", "No variant match; installing via app metadata URL")
                    installer.install(app)
                }

                if (success) {
                    _ui.value = _ui.value.copy(isInstalling = false, progress = 1f, stage = TaskStage.Finished(true))
                } else {
                    _ui.value = _ui.value.copy(isInstalling = false, error = "Installation failed", stage = TaskStage.Finished(false))
                }

                delay(1000)
                val newInstalled = installedRepo.getVersionCode(packageName)
                _ui.value = _ui.value.copy(installedVersionCode = newInstalled)

            } catch (e: Exception) {
                _ui.value = _ui.value.copy(isInstalling = false, error = "Install failed: ${e.message}", stage = TaskStage.Finished(false))
            }
        }
    }

    fun openApp() = installer.open(packageName)
    fun cancel() = installer.cancel(packageName)

    fun uninstall() {
        installer.uninstall(packageName)
        viewModelScope.launch {
            delay(1000)
            val newInstalled = installedRepo.getVersionCode(packageName)
            _ui.value = _ui.value.copy(installedVersionCode = newInstalled)
        }
    }

    fun installVariant(variant: AppVariant) {
        viewModelScope.launch {
            _ui.value = _ui.value.copy(isInstalling = true, progress = 0f, error = null, stage = TaskStage.Downloading(0f))
            try {
                val cur = UpdatesPreferences[packageName]
                UpdatesPreferences[packageName] = cur.copy(
                    preferredRepoUrl = variant.repositoryUrl.trim().trimEnd('/'),
                    lockToRepo = true
                )
                val ok = installer.install(variant)
                _ui.update {
                    it.copy(
                        isInstalling = false,
                        stage = TaskStage.Finished(ok),
                        progress = if (ok) 1f else it.progress,
                        error = if (ok) null else "Installation failed"
                    )
                }
                delay(1000)
                val newInstalled = installedRepo.getVersionCode(packageName)
                _ui.update { it.copy(installedVersionCode = newInstalled) }
            } catch (e: Exception) {
                _ui.update { it.copy(isInstalling = false, error = "Install failed: ${e.message}", stage = TaskStage.Finished(false)) }
            }
        }
    }
}