"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createMonacoLanguagesAPI = exports.registerInlayHintsProvider = exports.registerInlineCompletionsProvider = exports.registerDocumentRangeSemanticTokensProvider = exports.registerDocumentSemanticTokensProvider = exports.registerSelectionRangeProvider = exports.registerDeclarationProvider = exports.registerFoldingRangeProvider = exports.registerColorProvider = exports.registerCompletionItemProvider = exports.registerLinkProvider = exports.registerOnTypeFormattingEditProvider = exports.registerDocumentRangeFormattingEditProvider = exports.registerDocumentFormattingEditProvider = exports.registerCodeActionProvider = exports.registerCodeLensProvider = exports.registerTypeDefinitionProvider = exports.registerImplementationProvider = exports.registerDefinitionProvider = exports.registerLinkedEditingRangeProvider = exports.registerDocumentHighlightProvider = exports.registerDocumentSymbolProvider = exports.registerHoverProvider = exports.registerSignatureHelpProvider = exports.registerRenameProvider = exports.registerReferenceProvider = exports.setMonarchTokensProvider = exports.setTokensProvider = exports.registerTokensProviderFactory = exports.setColorMap = exports.TokenizationSupportAdapter = exports.EncodedTokenizationSupportAdapter = exports.setLanguageConfiguration = exports.onLanguage = exports.getEncodedLanguageId = exports.getLanguages = exports.register = void 0;
const color_1 = require("../../../base/common/color");
const range_1 = require("../../common/core/range");
const languages = require("../../common/languages");
const languageConfigurationRegistry_1 = require("../../common/languages/languageConfigurationRegistry");
const modesRegistry_1 = require("../../common/languages/modesRegistry");
const language_1 = require("../../common/languages/language");
const standaloneEnums = require("../../common/standalone/standaloneEnums");
const standaloneServices_1 = require("./standaloneServices");
const monarchCompile_1 = require("../common/monarch/monarchCompile");
const monarchLexer_1 = require("../common/monarch/monarchLexer");
const standaloneTheme_1 = require("../common/standaloneTheme");
const markers_1 = require("../../../platform/markers/common/markers");
const languageFeatures_1 = require("../../common/services/languageFeatures");
/**
 * Register information about a new language.
 */
function register(language) {
    modesRegistry_1.ModesRegistry.registerLanguage(language);
}
exports.register = register;
/**
 * Get the information of all the registered languages.
 */
function getLanguages() {
    let result = [];
    result = result.concat(modesRegistry_1.ModesRegistry.getLanguages());
    return result;
}
exports.getLanguages = getLanguages;
function getEncodedLanguageId(languageId) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    return languageService.languageIdCodec.encodeLanguageId(languageId);
}
exports.getEncodedLanguageId = getEncodedLanguageId;
/**
 * An event emitted when a language is needed for the first time (e.g. a model has it set).
 * @event
 */
function onLanguage(languageId, callback) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    const disposable = languageService.onDidEncounterLanguage((encounteredLanguageId) => {
        if (encounteredLanguageId === languageId) {
            // stop listening
            disposable.dispose();
            // invoke actual listener
            callback();
        }
    });
    return disposable;
}
exports.onLanguage = onLanguage;
/**
 * Set the editing configuration for a language.
 */
function setLanguageConfiguration(languageId, configuration) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    if (!languageService.isRegisteredLanguageId(languageId)) {
        throw new Error(`Cannot set configuration for unknown language ${languageId}`);
    }
    return languageConfigurationRegistry_1.LanguageConfigurationRegistry.register(languageId, configuration, 100);
}
exports.setLanguageConfiguration = setLanguageConfiguration;
/**
 * @internal
 */
class EncodedTokenizationSupportAdapter {
    constructor(languageId, actual) {
        this._languageId = languageId;
        this._actual = actual;
    }
    getInitialState() {
        return this._actual.getInitialState();
    }
    tokenize(line, hasEOL, state) {
        if (typeof this._actual.tokenize === 'function') {
            return TokenizationSupportAdapter.adaptTokenize(this._languageId, this._actual, line, state);
        }
        throw new Error('Not supported!');
    }
    tokenizeEncoded(line, hasEOL, state) {
        const result = this._actual.tokenizeEncoded(line, state);
        return new languages.EncodedTokenizationResult(result.tokens, result.endState);
    }
}
exports.EncodedTokenizationSupportAdapter = EncodedTokenizationSupportAdapter;
/**
 * @internal
 */
class TokenizationSupportAdapter {
    constructor(_languageId, _actual, _languageService, _standaloneThemeService) {
        this._languageId = _languageId;
        this._actual = _actual;
        this._languageService = _languageService;
        this._standaloneThemeService = _standaloneThemeService;
    }
    getInitialState() {
        return this._actual.getInitialState();
    }
    static _toClassicTokens(tokens, language) {
        const result = [];
        let previousStartIndex = 0;
        for (let i = 0, len = tokens.length; i < len; i++) {
            const t = tokens[i];
            let startIndex = t.startIndex;
            // Prevent issues stemming from a buggy external tokenizer.
            if (i === 0) {
                // Force first token to start at first index!
                startIndex = 0;
            }
            else if (startIndex < previousStartIndex) {
                // Force tokens to be after one another!
                startIndex = previousStartIndex;
            }
            result[i] = new languages.Token(startIndex, t.scopes, language);
            previousStartIndex = startIndex;
        }
        return result;
    }
    static adaptTokenize(language, actual, line, state) {
        const actualResult = actual.tokenize(line, state);
        const tokens = TokenizationSupportAdapter._toClassicTokens(actualResult.tokens, language);
        let endState;
        // try to save an object if possible
        if (actualResult.endState.equals(state)) {
            endState = state;
        }
        else {
            endState = actualResult.endState;
        }
        return new languages.TokenizationResult(tokens, endState);
    }
    tokenize(line, hasEOL, state) {
        return TokenizationSupportAdapter.adaptTokenize(this._languageId, this._actual, line, state);
    }
    _toBinaryTokens(languageIdCodec, tokens) {
        const languageId = languageIdCodec.encodeLanguageId(this._languageId);
        const tokenTheme = this._standaloneThemeService.getColorTheme().tokenTheme;
        const result = [];
        let resultLen = 0;
        let previousStartIndex = 0;
        for (let i = 0, len = tokens.length; i < len; i++) {
            const t = tokens[i];
            const metadata = tokenTheme.match(languageId, t.scopes);
            if (resultLen > 0 && result[resultLen - 1] === metadata) {
                // same metadata
                continue;
            }
            let startIndex = t.startIndex;
            // Prevent issues stemming from a buggy external tokenizer.
            if (i === 0) {
                // Force first token to start at first index!
                startIndex = 0;
            }
            else if (startIndex < previousStartIndex) {
                // Force tokens to be after one another!
                startIndex = previousStartIndex;
            }
            result[resultLen++] = startIndex;
            result[resultLen++] = metadata;
            previousStartIndex = startIndex;
        }
        const actualResult = new Uint32Array(resultLen);
        for (let i = 0; i < resultLen; i++) {
            actualResult[i] = result[i];
        }
        return actualResult;
    }
    tokenizeEncoded(line, hasEOL, state) {
        const actualResult = this._actual.tokenize(line, state);
        const tokens = this._toBinaryTokens(this._languageService.languageIdCodec, actualResult.tokens);
        let endState;
        // try to save an object if possible
        if (actualResult.endState.equals(state)) {
            endState = state;
        }
        else {
            endState = actualResult.endState;
        }
        return new languages.EncodedTokenizationResult(tokens, endState);
    }
}
exports.TokenizationSupportAdapter = TokenizationSupportAdapter;
function isATokensProvider(provider) {
    return (typeof provider.getInitialState === 'function');
}
function isEncodedTokensProvider(provider) {
    return 'tokenizeEncoded' in provider;
}
function isThenable(obj) {
    return obj && typeof obj.then === 'function';
}
/**
 * Change the color map that is used for token colors.
 * Supported formats (hex): #RRGGBB, $RRGGBBAA, #RGB, #RGBA
 */
function setColorMap(colorMap) {
    const standaloneThemeService = standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService);
    if (colorMap) {
        const result = [null];
        for (let i = 1, len = colorMap.length; i < len; i++) {
            result[i] = color_1.Color.fromHex(colorMap[i]);
        }
        standaloneThemeService.setColorMapOverride(result);
    }
    else {
        standaloneThemeService.setColorMapOverride(null);
    }
}
exports.setColorMap = setColorMap;
/**
 * @internal
 */
function createTokenizationSupportAdapter(languageId, provider) {
    if (isEncodedTokensProvider(provider)) {
        return new EncodedTokenizationSupportAdapter(languageId, provider);
    }
    else {
        return new TokenizationSupportAdapter(languageId, provider, standaloneServices_1.StandaloneServices.get(language_1.ILanguageService), standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService));
    }
}
/**
 * Register a tokens provider factory for a language. This tokenizer will be exclusive with a tokenizer
 * set using `setTokensProvider` or one created using `setMonarchTokensProvider`, but will work together
 * with a tokens provider set using `registerDocumentSemanticTokensProvider` or `registerDocumentRangeSemanticTokensProvider`.
 */
function registerTokensProviderFactory(languageId, factory) {
    const adaptedFactory = {
        createTokenizationSupport: () => __awaiter(this, void 0, void 0, function* () {
            const result = yield Promise.resolve(factory.create());
            if (!result) {
                return null;
            }
            if (isATokensProvider(result)) {
                return createTokenizationSupportAdapter(languageId, result);
            }
            return new monarchLexer_1.MonarchTokenizer(standaloneServices_1.StandaloneServices.get(language_1.ILanguageService), standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService), languageId, (0, monarchCompile_1.compile)(languageId, result));
        })
    };
    return languages.TokenizationRegistry.registerFactory(languageId, adaptedFactory);
}
exports.registerTokensProviderFactory = registerTokensProviderFactory;
/**
 * Set the tokens provider for a language (manual implementation). This tokenizer will be exclusive
 * with a tokenizer created using `setMonarchTokensProvider`, or with `registerTokensProviderFactory`,
 * but will work together with a tokens provider set using `registerDocumentSemanticTokensProvider`
 * or `registerDocumentRangeSemanticTokensProvider`.
 */
function setTokensProvider(languageId, provider) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    if (!languageService.isRegisteredLanguageId(languageId)) {
        throw new Error(`Cannot set tokens provider for unknown language ${languageId}`);
    }
    if (isThenable(provider)) {
        return registerTokensProviderFactory(languageId, { create: () => provider });
    }
    return languages.TokenizationRegistry.register(languageId, createTokenizationSupportAdapter(languageId, provider));
}
exports.setTokensProvider = setTokensProvider;
/**
 * Set the tokens provider for a language (monarch implementation). This tokenizer will be exclusive
 * with a tokenizer set using `setTokensProvider`, or with `registerTokensProviderFactory`, but will
 * work together with a tokens provider set using `registerDocumentSemanticTokensProvider` or
 * `registerDocumentRangeSemanticTokensProvider`.
 */
function setMonarchTokensProvider(languageId, languageDef) {
    const create = (languageDef) => {
        return new monarchLexer_1.MonarchTokenizer(standaloneServices_1.StandaloneServices.get(language_1.ILanguageService), standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService), languageId, (0, monarchCompile_1.compile)(languageId, languageDef));
    };
    if (isThenable(languageDef)) {
        return registerTokensProviderFactory(languageId, { create: () => languageDef });
    }
    return languages.TokenizationRegistry.register(languageId, create(languageDef));
}
exports.setMonarchTokensProvider = setMonarchTokensProvider;
/**
 * Register a reference provider (used by e.g. reference search).
 */
function registerReferenceProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.referenceProvider.register(languageId, provider);
}
exports.registerReferenceProvider = registerReferenceProvider;
/**
 * Register a rename provider (used by e.g. rename symbol).
 */
function registerRenameProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.renameProvider.register(languageId, provider);
}
exports.registerRenameProvider = registerRenameProvider;
/**
 * Register a signature help provider (used by e.g. parameter hints).
 */
function registerSignatureHelpProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.signatureHelpProvider.register(languageId, provider);
}
exports.registerSignatureHelpProvider = registerSignatureHelpProvider;
/**
 * Register a hover provider (used by e.g. editor hover).
 */
function registerHoverProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.hoverProvider.register(languageId, {
        provideHover: (model, position, token) => {
            const word = model.getWordAtPosition(position);
            return Promise.resolve(provider.provideHover(model, position, token)).then((value) => {
                if (!value) {
                    return undefined;
                }
                if (!value.range && word) {
                    value.range = new range_1.Range(position.lineNumber, word.startColumn, position.lineNumber, word.endColumn);
                }
                if (!value.range) {
                    value.range = new range_1.Range(position.lineNumber, position.column, position.lineNumber, position.column);
                }
                return value;
            });
        }
    });
}
exports.registerHoverProvider = registerHoverProvider;
/**
 * Register a document symbol provider (used by e.g. outline).
 */
function registerDocumentSymbolProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.documentSymbolProvider.register(languageId, provider);
}
exports.registerDocumentSymbolProvider = registerDocumentSymbolProvider;
/**
 * Register a document highlight provider (used by e.g. highlight occurrences).
 */
function registerDocumentHighlightProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.documentHighlightProvider.register(languageId, provider);
}
exports.registerDocumentHighlightProvider = registerDocumentHighlightProvider;
/**
 * Register an linked editing range provider.
 */
function registerLinkedEditingRangeProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.linkedEditingRangeProvider.register(languageId, provider);
}
exports.registerLinkedEditingRangeProvider = registerLinkedEditingRangeProvider;
/**
 * Register a definition provider (used by e.g. go to definition).
 */
function registerDefinitionProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.definitionProvider.register(languageId, provider);
}
exports.registerDefinitionProvider = registerDefinitionProvider;
/**
 * Register a implementation provider (used by e.g. go to implementation).
 */
function registerImplementationProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.implementationProvider.register(languageId, provider);
}
exports.registerImplementationProvider = registerImplementationProvider;
/**
 * Register a type definition provider (used by e.g. go to type definition).
 */
function registerTypeDefinitionProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.typeDefinitionProvider.register(languageId, provider);
}
exports.registerTypeDefinitionProvider = registerTypeDefinitionProvider;
/**
 * Register a code lens provider (used by e.g. inline code lenses).
 */
function registerCodeLensProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.codeLensProvider.register(languageId, provider);
}
exports.registerCodeLensProvider = registerCodeLensProvider;
/**
 * Register a code action provider (used by e.g. quick fix).
 */
function registerCodeActionProvider(languageId, provider, metadata) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.codeActionProvider.register(languageId, {
        providedCodeActionKinds: metadata === null || metadata === void 0 ? void 0 : metadata.providedCodeActionKinds,
        provideCodeActions: (model, range, context, token) => {
            const markerService = standaloneServices_1.StandaloneServices.get(markers_1.IMarkerService);
            const markers = markerService.read({ resource: model.uri }).filter(m => {
                return range_1.Range.areIntersectingOrTouching(m, range);
            });
            return provider.provideCodeActions(model, range, { markers, only: context.only }, token);
        },
        resolveCodeAction: provider.resolveCodeAction
    });
}
exports.registerCodeActionProvider = registerCodeActionProvider;
/**
 * Register a formatter that can handle only entire models.
 */
function registerDocumentFormattingEditProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.documentFormattingEditProvider.register(languageId, provider);
}
exports.registerDocumentFormattingEditProvider = registerDocumentFormattingEditProvider;
/**
 * Register a formatter that can handle a range inside a model.
 */
function registerDocumentRangeFormattingEditProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.documentRangeFormattingEditProvider.register(languageId, provider);
}
exports.registerDocumentRangeFormattingEditProvider = registerDocumentRangeFormattingEditProvider;
/**
 * Register a formatter than can do formatting as the user types.
 */
function registerOnTypeFormattingEditProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.onTypeFormattingEditProvider.register(languageId, provider);
}
exports.registerOnTypeFormattingEditProvider = registerOnTypeFormattingEditProvider;
/**
 * Register a link provider that can find links in text.
 */
function registerLinkProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.linkProvider.register(languageId, provider);
}
exports.registerLinkProvider = registerLinkProvider;
/**
 * Register a completion item provider (use by e.g. suggestions).
 */
function registerCompletionItemProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.completionProvider.register(languageId, provider);
}
exports.registerCompletionItemProvider = registerCompletionItemProvider;
/**
 * Register a document color provider (used by Color Picker, Color Decorator).
 */
function registerColorProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.colorProvider.register(languageId, provider);
}
exports.registerColorProvider = registerColorProvider;
/**
 * Register a folding range provider
 */
function registerFoldingRangeProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.foldingRangeProvider.register(languageId, provider);
}
exports.registerFoldingRangeProvider = registerFoldingRangeProvider;
/**
 * Register a declaration provider
 */
function registerDeclarationProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.declarationProvider.register(languageId, provider);
}
exports.registerDeclarationProvider = registerDeclarationProvider;
/**
 * Register a selection range provider
 */
function registerSelectionRangeProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.selectionRangeProvider.register(languageId, provider);
}
exports.registerSelectionRangeProvider = registerSelectionRangeProvider;
/**
 * Register a document semantic tokens provider. A semantic tokens provider will complement and enhance a
 * simple top-down tokenizer. Simple top-down tokenizers can be set either via `setMonarchTokensProvider`
 * or `setTokensProvider`.
 *
 * For the best user experience, register both a semantic tokens provider and a top-down tokenizer.
 */
function registerDocumentSemanticTokensProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.documentSemanticTokensProvider.register(languageId, provider);
}
exports.registerDocumentSemanticTokensProvider = registerDocumentSemanticTokensProvider;
/**
 * Register a document range semantic tokens provider. A semantic tokens provider will complement and enhance a
 * simple top-down tokenizer. Simple top-down tokenizers can be set either via `setMonarchTokensProvider`
 * or `setTokensProvider`.
 *
 * For the best user experience, register both a semantic tokens provider and a top-down tokenizer.
 */
function registerDocumentRangeSemanticTokensProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.documentRangeSemanticTokensProvider.register(languageId, provider);
}
exports.registerDocumentRangeSemanticTokensProvider = registerDocumentRangeSemanticTokensProvider;
/**
 * Register an inline completions provider.
 */
function registerInlineCompletionsProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.inlineCompletionsProvider.register(languageId, provider);
}
exports.registerInlineCompletionsProvider = registerInlineCompletionsProvider;
/**
 * Register an inlay hints provider.
 */
function registerInlayHintsProvider(languageId, provider) {
    const languageFeaturesService = standaloneServices_1.StandaloneServices.get(languageFeatures_1.ILanguageFeaturesService);
    return languageFeaturesService.inlayHintsProvider.register(languageId, provider);
}
exports.registerInlayHintsProvider = registerInlayHintsProvider;
/**
 * @internal
 */
function createMonacoLanguagesAPI() {
    return {
        register: register,
        getLanguages: getLanguages,
        onLanguage: onLanguage,
        getEncodedLanguageId: getEncodedLanguageId,
        // provider methods
        setLanguageConfiguration: setLanguageConfiguration,
        setColorMap: setColorMap,
        registerTokensProviderFactory: registerTokensProviderFactory,
        setTokensProvider: setTokensProvider,
        setMonarchTokensProvider: setMonarchTokensProvider,
        registerReferenceProvider: registerReferenceProvider,
        registerRenameProvider: registerRenameProvider,
        registerCompletionItemProvider: registerCompletionItemProvider,
        registerSignatureHelpProvider: registerSignatureHelpProvider,
        registerHoverProvider: registerHoverProvider,
        registerDocumentSymbolProvider: registerDocumentSymbolProvider,
        registerDocumentHighlightProvider: registerDocumentHighlightProvider,
        registerLinkedEditingRangeProvider: registerLinkedEditingRangeProvider,
        registerDefinitionProvider: registerDefinitionProvider,
        registerImplementationProvider: registerImplementationProvider,
        registerTypeDefinitionProvider: registerTypeDefinitionProvider,
        registerCodeLensProvider: registerCodeLensProvider,
        registerCodeActionProvider: registerCodeActionProvider,
        registerDocumentFormattingEditProvider: registerDocumentFormattingEditProvider,
        registerDocumentRangeFormattingEditProvider: registerDocumentRangeFormattingEditProvider,
        registerOnTypeFormattingEditProvider: registerOnTypeFormattingEditProvider,
        registerLinkProvider: registerLinkProvider,
        registerColorProvider: registerColorProvider,
        registerFoldingRangeProvider: registerFoldingRangeProvider,
        registerDeclarationProvider: registerDeclarationProvider,
        registerSelectionRangeProvider: registerSelectionRangeProvider,
        registerDocumentSemanticTokensProvider: registerDocumentSemanticTokensProvider,
        registerDocumentRangeSemanticTokensProvider: registerDocumentRangeSemanticTokensProvider,
        registerInlineCompletionsProvider: registerInlineCompletionsProvider,
        registerInlayHintsProvider: registerInlayHintsProvider,
        // enums
        DocumentHighlightKind: standaloneEnums.DocumentHighlightKind,
        CompletionItemKind: standaloneEnums.CompletionItemKind,
        CompletionItemTag: standaloneEnums.CompletionItemTag,
        CompletionItemInsertTextRule: standaloneEnums.CompletionItemInsertTextRule,
        SymbolKind: standaloneEnums.SymbolKind,
        SymbolTag: standaloneEnums.SymbolTag,
        IndentAction: standaloneEnums.IndentAction,
        CompletionTriggerKind: standaloneEnums.CompletionTriggerKind,
        SignatureHelpTriggerKind: standaloneEnums.SignatureHelpTriggerKind,
        InlayHintKind: standaloneEnums.InlayHintKind,
        InlineCompletionTriggerKind: standaloneEnums.InlineCompletionTriggerKind,
        // classes
        FoldingRangeKind: languages.FoldingRangeKind,
    };
}
exports.createMonacoLanguagesAPI = createMonacoLanguagesAPI;
//# sourceMappingURL=standaloneLanguages.js.map