"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMimeTypes = exports.clearLanguageAssociations = exports.registerLanguageAssociation = void 0;
const glob_1 = require("../../../base/common/glob");
const mime_1 = require("../../../base/common/mime");
const network_1 = require("../../../base/common/network");
const path_1 = require("../../../base/common/path");
const resources_1 = require("../../../base/common/resources");
const strings_1 = require("../../../base/common/strings");
let registeredAssociations = [];
let nonUserRegisteredAssociations = [];
let userRegisteredAssociations = [];
/**
 * Associate a language to the registry.
 */
function registerLanguageAssociation(association, warnOnOverwrite = false) {
    // Register
    const associationItem = toLanguageAssociationItem(association);
    registeredAssociations.push(associationItem);
    if (!associationItem.userConfigured) {
        nonUserRegisteredAssociations.push(associationItem);
    }
    else {
        userRegisteredAssociations.push(associationItem);
    }
    // Check for conflicts unless this is a user configured association
    if (warnOnOverwrite && !associationItem.userConfigured) {
        registeredAssociations.forEach(a => {
            if (a.mime === associationItem.mime || a.userConfigured) {
                return; // same mime or userConfigured is ok
            }
            if (associationItem.extension && a.extension === associationItem.extension) {
                console.warn(`Overwriting extension <<${associationItem.extension}>> to now point to mime <<${associationItem.mime}>>`);
            }
            if (associationItem.filename && a.filename === associationItem.filename) {
                console.warn(`Overwriting filename <<${associationItem.filename}>> to now point to mime <<${associationItem.mime}>>`);
            }
            if (associationItem.filepattern && a.filepattern === associationItem.filepattern) {
                console.warn(`Overwriting filepattern <<${associationItem.filepattern}>> to now point to mime <<${associationItem.mime}>>`);
            }
            if (associationItem.firstline && a.firstline === associationItem.firstline) {
                console.warn(`Overwriting firstline <<${associationItem.firstline}>> to now point to mime <<${associationItem.mime}>>`);
            }
        });
    }
}
exports.registerLanguageAssociation = registerLanguageAssociation;
function toLanguageAssociationItem(association) {
    return {
        id: association.id,
        mime: association.mime,
        filename: association.filename,
        extension: association.extension,
        filepattern: association.filepattern,
        firstline: association.firstline,
        userConfigured: association.userConfigured,
        filenameLowercase: association.filename ? association.filename.toLowerCase() : undefined,
        extensionLowercase: association.extension ? association.extension.toLowerCase() : undefined,
        filepatternLowercase: association.filepattern ? (0, glob_1.parse)(association.filepattern.toLowerCase()) : undefined,
        filepatternOnPath: association.filepattern ? association.filepattern.indexOf(path_1.posix.sep) >= 0 : false
    };
}
/**
 * Clear language associations from the registry.
 */
function clearLanguageAssociations(onlyUserConfigured) {
    if (!onlyUserConfigured) {
        registeredAssociations = [];
        nonUserRegisteredAssociations = [];
        userRegisteredAssociations = [];
    }
    else {
        registeredAssociations = registeredAssociations.filter(a => !a.userConfigured);
        userRegisteredAssociations = [];
    }
}
exports.clearLanguageAssociations = clearLanguageAssociations;
/**
 * Given a file, return the best matching mime types for it
 * based on the registered language associations.
 */
function getMimeTypes(resource, firstLine) {
    let path;
    if (resource) {
        switch (resource.scheme) {
            case network_1.Schemas.file:
                path = resource.fsPath;
                break;
            case network_1.Schemas.data: {
                const metadata = resources_1.DataUri.parseMetaData(resource);
                path = metadata.get(resources_1.DataUri.META_DATA_LABEL);
                break;
            }
            default:
                path = resource.path;
        }
    }
    if (!path) {
        return [mime_1.Mimes.unknown];
    }
    path = path.toLowerCase();
    const filename = (0, path_1.basename)(path);
    // 1.) User configured mappings have highest priority
    const configuredLanguage = getMimeByPath(path, filename, userRegisteredAssociations);
    if (configuredLanguage) {
        return [configuredLanguage, mime_1.Mimes.text];
    }
    // 2.) Registered mappings have middle priority
    const registeredLanguage = getMimeByPath(path, filename, nonUserRegisteredAssociations);
    if (registeredLanguage) {
        return [registeredLanguage, mime_1.Mimes.text];
    }
    // 3.) Firstline has lowest priority
    if (firstLine) {
        const firstlineLanguage = getMimeByFirstline(firstLine);
        if (firstlineLanguage) {
            return [firstlineLanguage, mime_1.Mimes.text];
        }
    }
    return [mime_1.Mimes.unknown];
}
exports.getMimeTypes = getMimeTypes;
function getMimeByPath(path, filename, associations) {
    var _a;
    let filenameMatch = undefined;
    let patternMatch = undefined;
    let extensionMatch = undefined;
    // We want to prioritize associations based on the order they are registered so that the last registered
    // association wins over all other. This is for https://github.com/microsoft/vscode/issues/20074
    for (let i = associations.length - 1; i >= 0; i--) {
        const association = associations[i];
        // First exact name match
        if (filename === association.filenameLowercase) {
            filenameMatch = association;
            break; // take it!
        }
        // Longest pattern match
        if (association.filepattern) {
            if (!patternMatch || association.filepattern.length > patternMatch.filepattern.length) {
                const target = association.filepatternOnPath ? path : filename; // match on full path if pattern contains path separator
                if ((_a = association.filepatternLowercase) === null || _a === void 0 ? void 0 : _a.call(association, target)) {
                    patternMatch = association;
                }
            }
        }
        // Longest extension match
        if (association.extension) {
            if (!extensionMatch || association.extension.length > extensionMatch.extension.length) {
                if (filename.endsWith(association.extensionLowercase)) {
                    extensionMatch = association;
                }
            }
        }
    }
    // 1.) Exact name match has second highest priority
    if (filenameMatch) {
        return filenameMatch.mime;
    }
    // 2.) Match on pattern
    if (patternMatch) {
        return patternMatch.mime;
    }
    // 3.) Match on extension comes next
    if (extensionMatch) {
        return extensionMatch.mime;
    }
    return undefined;
}
function getMimeByFirstline(firstLine) {
    if ((0, strings_1.startsWithUTF8BOM)(firstLine)) {
        firstLine = firstLine.substr(1);
    }
    if (firstLine.length > 0) {
        // We want to prioritize associations based on the order they are registered so that the last registered
        // association wins over all other. This is for https://github.com/microsoft/vscode/issues/20074
        for (let i = registeredAssociations.length - 1; i >= 0; i--) {
            const association = registeredAssociations[i];
            if (!association.firstline) {
                continue;
            }
            const matches = firstLine.match(association.firstline);
            if (matches && matches.length > 0) {
                return association.mime;
            }
        }
    }
    return undefined;
}
//# sourceMappingURL=languagesAssociations.js.map