/*********************************************************************
* Copyright (c) 2005, 2019 SAP SE
*
* This program and the accompanying materials are made
* available under the terms of the Eclipse Public License 2.0
* which is available at https://www.eclipse.org/legal/epl-2.0/
*
* Contributors:
*    SAP SE - initial API, implementation and documentation
*    mwenz - Bug 415884 - Cannot query size of a multi-line text
*    Aurelien Pupier - Bug 499720 - DefaultConnectionEditPolicy doesn't support scroll
*
* SPDX-License-Identifier: EPL-2.0
**********************************************************************/
package org.eclipse.graphiti.ui.internal.services;

import java.util.List;

import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPartViewer;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.graphiti.datatypes.IDimension;
import org.eclipse.graphiti.mm.algorithms.styles.Font;
import org.eclipse.graphiti.mm.pictograms.Connection;
import org.eclipse.graphiti.ui.internal.util.draw2d.ChopboxAnchorFixed;

/**
 * @noimplement This interface is not intended to be implemented by clients.
 * @noextend This class is not intended to be subclassed by clients.
 */
public interface IGefService {

	/**
	 * Selects the EditPart for the given model Object.
	 * 
	 * @param viewer
	 *            The viewer, which contains the EditParts.
	 * @param modelObject
	 *            The model Object, which EditPart to select.
	 */
	void selectEditPart(EditPartViewer viewer, Object modelObject);

	/**
	 * Returns the translation between the coordinate-systems of EditParts. It
	 * will translate from the coordinate-system of the source EditPart to the
	 * coordinate-system of the target EditPart. Both EditParts must be an
	 * instance of GraphicalEditPart. Otherwise it throws an Exception.
	 * 
	 * @param source
	 *            The EditPart with the source coordinate-system.
	 * @param target
	 *            The EditPart with the target coordinate-system.
	 * @return Returns the translation between the coordinate-systems of
	 *         EditParts.
	 */
	Point calculateTranslation(EditPart source, EditPart target);

	/**
	 * Returns the layout constraint for the given EditPart. It returns null, if
	 * the constraint can not be determined, for example because the child has
	 * no parent or the parent has no LayoutManager.
	 * 
	 * @param editPart
	 *            The EditPart for which to return the layout constraint.
	 * @return Returns the layout constraint for the given EditPart.
	 */
	Object getLayoutConstraint(EditPart editPart);

	/**
	 * @param viewer
	 * @param location
	 *            : the absolute location for the whole RootEditpart (if any,
	 *            the scroll should be considered)
	 * @param includeConnections
	 * @return
	 */
	EditPart findEditPartAt(EditPartViewer viewer, Point location, boolean includeConnections);

	/**
	 * Computes EditParts of the connections and related ConnectionDecorators
	 * which live inside the containment hierarchy of the given edit part.
	 * 
	 * @param ep
	 *            the EditPart whose inner connection and ConnectionDecorators
	 *            should be computed
	 * @returns List of EditParts (connection and Decorator) which live in
	 *          EditPart
	 */
	List<EditPart> getConnectionsContainedInEditPart(EditPart ep);

	/**
	 * @param c
	 * @param d
	 * @return
	 */
	Point getConnectionPointAt(Connection c, double d);

	/**
	 * This method has to be kept in sync with
	 * {@link ChopboxAnchorFixed#getLocation(Point)}.
	 */
	Point getChopboxLocationOnBox(Point reference, Rectangle box);

	/**
	 * @param c
	 * @param absDistance
	 *            absDistance > 0 means distance from beginning. absDistance < 0
	 *            means distance from connection end.
	 * @return
	 */
	Point getAbsolutePointOnConnection(Connection c, double distance);

	Point getDistantPoint(Point start, Point end, double distance);

	Point getDistantPoint(int startX, int startY, int endX, int endY, double distance);

	/**
	 * @param startX
	 * @param startY
	 * @param endX
	 * @param endY
	 * @param d
	 * @return
	 */
	Point getPointAt(int startX, int startY, int endX, int endY, double d);

	Point getPointAt(Point start, Point end, double d);

	double getDistance(Point[] points);

	/**
	 * Calculates and returns the size of the text ignoring any new line
	 * characters in the string.
	 * 
	 * @param text
	 * @param font
	 * 
	 * @return the size of the text
	 * 
	 * @see #calculateTextSize(String, Font, boolean)
	 */
	IDimension calculateTextSize(String text, Font font);

	/**
	 * Calculates and returns the size of the text.
	 * 
	 * @param text
	 * @param font
	 * @param handleMultiline
	 *            Defines if line breaks in the string should be used in the
	 *            calculation of the size or not. In case <code>true</code>, a
	 *            new line character in the string will increase the size of the
	 *            returned dimensions by one line, in case <code>false</code> a
	 *            new line character will be ignored.
	 * 
	 * @return the size of the text
	 */
	IDimension calculateTextSize(String text, Font font, boolean handleMultiline);

	/**
	 * @param draw2dPoints
	 */
	void mirrorArray(Point[] draw2dPoints);

	/**
	 * Provides the direct children of the edit part.
	 * 
	 * @param editPart
	 * @return the direct child edit parts
	 */
	public List<EditPart> getEditPartChildren(EditPart editPart);

	/**
	 * Provides the source connections of the graphical edit part.
	 * 
	 * @param graphicalEditPart
	 * @return the source connections of the graphical edit part
	 */
	public List<GraphicalEditPart> getSourceConnections(GraphicalEditPart graphicalEditPart);

	/**
	 * Provides the target connections of the graphical edit part.
	 * 
	 * @param graphicalEditPart
	 * @return the target connections of the graphical edit part
	 */
	public List<GraphicalEditPart> getTargetConnections(GraphicalEditPart graphicalEditPart);

	/**
	 * Provides the selected edit parts of the viewer.
	 * 
	 * @param editPartViewer
	 * @return the selected edit parts of the viewer
	 */
	public List<EditPart> getSelectedEditParts(EditPartViewer editPartViewer);
}