# Commands covered:  exec
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1991-1994 The Regents of the University of California.
# Copyright (c) 1994-1997 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: exec.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

package require tcltest 2
namespace import -force ::tcltest::*

# All tests require the "exec" command.
# Skip them if exec is not defined.
testConstraint exec [llength [info commands exec]]
unset -nocomplain path
set path(echo) [makeFile {
    puts -nonewline [lindex $argv 0]
    foreach str [lrange $argv 1 end] {
	puts -nonewline " $str"
    }
    puts {}
    exit
} echo]

set path(cat) [makeFile {
    if {$argv == {}} {
	set argv -
    }
    foreach name $argv {
	if {$name == "-"} {
	    set f stdin
	} elseif {[catch {open $name r} f] != 0} {
	    puts stderr $f
	    continue
	}
	while {[eof $f] == 0} {
	    puts -nonewline [read $f]
	}
	if {$f != "stdin"} {
	    close $f
	}
    }
    exit
} cat]

set path(wc) [makeFile {
    set data [read stdin]
    set lines [regsub -all "\n" $data {} dummy]
    set words [regsub -all "\[^ \t\n]+" $data {} dummy]
    set chars [string length $data]
    puts [format "%8.d%8.d%8.d" $lines $words $chars]
    exit
} wc]

set path(sh) [makeFile {
    if {[lindex $argv 0] != "-c"} {
	error "sh: unexpected arguments $argv"
    }
    set cmd [lindex $argv 1]
    lappend cmd ";"

    set newcmd {}
    
    foreach arg $cmd {
	if {$arg == ";"} {
	    eval exec >@stdout 2>@stderr [list [info nameofexecutable]] $newcmd
	    set newcmd {}
	    continue
	}
	if {$arg == "1>&2"} {
	    set arg >@stderr
	}
	lappend newcmd $arg
    }
    exit
} sh]

set path(sleep) [makeFile {
    after [expr $argv*1000]
    exit
} sleep]

set path(exit) [makeFile {
    exit $argv
} exit]

# Basic operations.

test exec-1.1 {basic exec operation} {exec} {
    exec [interpreter] $path(echo) a b c
} "a b c"
test exec-1.2 {pipelining} {exec stdio} {
    exec [interpreter] $path(echo) a b c d | [interpreter] $path(cat) | [interpreter] $path(cat)
} "a b c d"
test exec-1.3 {pipelining} {exec stdio} {
    set a [exec [interpreter] $path(echo) a b c d | [interpreter] $path(cat) | [interpreter] $path(wc)]
    list [scan $a "%d %d %d" b c d] $b $c
} {3 1 4}
set arg {12345678901234567890123456789012345678901234567890}
set arg "$arg$arg$arg$arg$arg$arg"
test exec-1.4 {long command lines} {exec} {
    exec [interpreter] $path(echo) $arg
} $arg
set arg {}

# I/O redirection: input from Tcl command.

test exec-2.1 {redirecting input from immediate source} {exec stdio} {
    exec [interpreter] $path(cat) << "Sample text"
} {Sample text}
test exec-2.2 {redirecting input from immediate source} {exec stdio} {
    exec << "Sample text" [interpreter] $path(cat) | [interpreter] $path(cat)
} {Sample text}
test exec-2.3 {redirecting input from immediate source} {exec stdio} {
    exec [interpreter] $path(cat) << "Sample text" | [interpreter] $path(cat)
} {Sample text}
test exec-2.4 {redirecting input from immediate source} {exec stdio} {
    exec [interpreter] $path(cat) | [interpreter] $path(cat) << "Sample text"
} {Sample text}
test exec-2.5 {redirecting input from immediate source} {exec} {
    exec [interpreter] $path(cat) "<<Joined to arrows"
} {Joined to arrows}
test exec-2.6 {redirecting input from immediate source, with UTF} {exec} {
    # If this fails, it may give back:
    # "\uC3\uA9\uC3\uA0\uC3\uBC\uC3\uB1"
    # If it does, this means that the UTF -> external conversion did not 
    # occur before writing out the temp file.
    exec [interpreter] $path(cat) << "\uE9\uE0\uFC\uF1"
} "\uE9\uE0\uFC\uF1"

# I/O redirection: output to file.

set path(gorp.file) [makeFile {} gorp.file]
file delete $path(gorp.file)

test exec-3.1 {redirecting output to file} {exec} {
    exec [interpreter] $path(echo) "Some simple words" > $path(gorp.file)
    exec [interpreter] $path(cat) $path(gorp.file)
} "Some simple words"
test exec-3.2 {redirecting output to file} {exec stdio} {
    exec [interpreter] $path(echo) "More simple words" | >$path(gorp.file) [interpreter] $path(cat) | [interpreter] $path(cat)
    exec [interpreter] $path(cat) $path(gorp.file)
} "More simple words"
test exec-3.3 {redirecting output to file} {exec stdio} {
    exec > $path(gorp.file) [interpreter] $path(echo) "Different simple words" | [interpreter] $path(cat) | [interpreter] $path(cat)
    exec [interpreter] $path(cat) $path(gorp.file)
} "Different simple words"
test exec-3.4 {redirecting output to file} {exec} {
    exec [interpreter] $path(echo) "Some simple words" >$path(gorp.file)
    exec [interpreter] $path(cat) $path(gorp.file)
} "Some simple words"
test exec-3.5 {redirecting output to file} {exec} {
    exec [interpreter] $path(echo) "First line" >$path(gorp.file)
    exec [interpreter] $path(echo) "Second line" >> $path(gorp.file)
    exec [interpreter] $path(cat) $path(gorp.file)
} "First line\nSecond line"
test exec-3.6 {redirecting output to file} {exec} {
    exec [interpreter] $path(echo) "First line" >$path(gorp.file)
    exec [interpreter] $path(echo) "Second line" >>$path(gorp.file)
    exec [interpreter] $path(cat) $path(gorp.file)
} "First line\nSecond line"
test exec-3.7 {redirecting output to file} {exec} {
    set f [open $path(gorp.file) w]
    puts $f "Line 1"
    flush $f
    exec [interpreter] $path(echo) "More text" >@ $f
    exec [interpreter] $path(echo) >@$f "Even more"
    puts $f "Line 3"
    close $f
    exec [interpreter] $path(cat) $path(gorp.file)
} "Line 1\nMore text\nEven more\nLine 3"

# I/O redirection: output and stderr to file.

file delete $path(gorp.file)

test exec-4.1 {redirecting output and stderr to file} {exec} {
    exec [interpreter] "$path(echo)" "test output" >& $path(gorp.file)
    exec [interpreter] "$path(cat)" "$path(gorp.file)"
} "test output"
test exec-4.2 {redirecting output and stderr to file} {exec} {
    list [exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" >&$path(gorp.file)] \
	    [exec [interpreter] "$path(cat)" "$path(gorp.file)"]
} {{} {foo bar}}
test exec-4.3 {redirecting output and stderr to file} {exec} {
    exec [interpreter] $path(echo) "first line" > $path(gorp.file)
    list [exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" >>&$path(gorp.file)] \
	    [exec [interpreter] "$path(cat)" "$path(gorp.file)"]
} "{} {first line\nfoo bar}"
test exec-4.4 {redirecting output and stderr to file} {exec} {
    set f [open "$path(gorp.file)" w]
    puts $f "Line 1"
    flush $f
    exec [interpreter] "$path(echo)" "More text" >&@ $f
    exec [interpreter] "$path(echo)" >&@$f "Even more"
    puts $f "Line 3"
    close $f
    exec [interpreter] "$path(cat)" "$path(gorp.file)"
} "Line 1\nMore text\nEven more\nLine 3"
test exec-4.5 {redirecting output and stderr to file} {exec} {
    set f [open "$path(gorp.file)" w]
    puts $f "Line 1"
    flush $f
    exec >&@ $f [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2"
    exec >&@$f [interpreter] "$path(sh)" -c "\"$path(echo)\" xyzzy 1>&2"
    puts $f "Line 3"
    close $f
    exec [interpreter] "$path(cat)" "$path(gorp.file)"
} "Line 1\nfoo bar\nxyzzy\nLine 3"

# I/O redirection: input from file.

if { [set ::tcltest::testConstraints(exec)] } {
exec [interpreter] $path(echo) "Just a few thoughts" > $path(gorp.file)
}
test exec-5.1 {redirecting input from file} {exec} {
    exec [interpreter] $path(cat) < $path(gorp.file)
} {Just a few thoughts}
test exec-5.2 {redirecting input from file} {exec stdio} {
    exec [interpreter] $path(cat) | [interpreter] $path(cat) < $path(gorp.file)
} {Just a few thoughts}
test exec-5.3 {redirecting input from file} {exec stdio} {
    exec [interpreter] $path(cat) < $path(gorp.file) | [interpreter] $path(cat)
} {Just a few thoughts}
test exec-5.4 {redirecting input from file} {exec stdio} {
    exec < $path(gorp.file) [interpreter] $path(cat) | [interpreter] $path(cat)
} {Just a few thoughts}
test exec-5.5 {redirecting input from file} {exec} {
    exec [interpreter] $path(cat) <$path(gorp.file)
} {Just a few thoughts}
test exec-5.6 {redirecting input from file} {exec} {
    set f [open $path(gorp.file) r]
    set result [exec [interpreter] $path(cat) <@ $f]
    close $f
    set result
} {Just a few thoughts}
test exec-5.7 {redirecting input from file} {exec} {
    set f [open $path(gorp.file) r]
    set result [exec <@$f [interpreter] $path(cat)]
    close $f
    set result
} {Just a few thoughts}

# I/O redirection: standard error through a pipeline.

test exec-6.1 {redirecting stderr through a pipeline} {exec stdio} {
    exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar" |& [interpreter] "$path(cat)"
} "foo bar"
test exec-6.2 {redirecting stderr through a pipeline} {exec stdio} {
    exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" |& [interpreter] "$path(cat)"
} "foo bar"
test exec-6.3 {redirecting stderr through a pipeline} {exec stdio} {
    exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" \
	|& [interpreter] "$path(sh)" -c "\"$path(echo)\" second msg 1>&2 ; \"$path(cat)\"" |& [interpreter] "$path(cat)"
} "second msg\nfoo bar"

# I/O redirection: combinations.

set path(gorp.file2) [makeFile {} gorp.file2]
file delete $path(gorp.file2)

test exec-7.1 {multiple I/O redirections} {exec} {
    exec << "command input" > $path(gorp.file2) [interpreter] $path(cat) < $path(gorp.file)
    exec [interpreter] $path(cat) $path(gorp.file2)
} {Just a few thoughts}
test exec-7.2 {multiple I/O redirections} {exec} {
    exec < $path(gorp.file) << "command input" [interpreter] $path(cat)
} {command input}

# Long input to command and output from command.

set a "0123456789 xxxxxxxxx abcdefghi ABCDEFGHIJK\n"
set a [concat $a $a $a $a]
set a [concat $a $a $a $a]
set a [concat $a $a $a $a]
set a [concat $a $a $a $a]
test exec-8.1 {long input and output} {exec} {
    exec [interpreter] $path(cat) << $a
} $a

# More than 20 arguments to exec.

test exec-8.2 {long input and output} {exec} {
    exec [interpreter] $path(echo) 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23
} {1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23}

# Commands that return errors.

test exec-9.1 {commands returning errors} {exec} {
    set x [catch {exec gorp456} msg]
    list $x [string tolower $msg] [string tolower $errorCode]
} {1 {couldn't execute "gorp456": no such file or directory} {posix enoent {no such file or directory}}}
test exec-9.2 {commands returning errors} {exec} {
    string tolower [list [catch {exec [interpreter] echo foo | foo123} msg] $msg $errorCode]
} {1 {couldn't execute "foo123": no such file or directory} {posix enoent {no such file or directory}}}
test exec-9.3 {commands returning errors} {exec stdio} {
    list [catch {exec [interpreter] $path(sleep) 1 | [interpreter] $path(exit) 43 | [interpreter] $path(sleep) 1} msg] $msg
} {1 {child process exited abnormally}}
test exec-9.4 {commands returning errors} {exec stdio} {
    list [catch {exec [interpreter] $path(exit) 43 | [interpreter] $path(echo) "foo bar"} msg] $msg
} {1 {foo bar
child process exited abnormally}}
test exec-9.5 {commands returning errors} {exec stdio} {
    list [catch {exec gorp456 | [interpreter] echo a b c} msg] [string tolower $msg]
} {1 {couldn't execute "gorp456": no such file or directory}}
test exec-9.6 {commands returning errors} {exec} {
    list [catch {exec [interpreter] "$path(sh)" -c "\"$path(echo)\" error msg 1>&2"} msg] $msg
} {1 {error msg}}
test exec-9.7 {commands returning errors} {exec stdio} {
    list [catch {exec [interpreter] "$path(sh)" -c "\"$path(echo)\" error msg 1>&2" \
		     | [interpreter] "$path(sh)" -c "\"$path(echo)\" error msg 1>&2"} msg] $msg
} {1 {error msg
error msg}}

set path(err) [makeFile {} err]

test exec-9.8 {commands returning errors} {exec} {
    set f [open $path(err) w]
    puts $f {
	puts stdout out
	puts stderr err
    }
    close $f
    list [catch {exec [interpreter] $path(err)} msg] $msg
} {1 {out
err}}

# Errors in executing the Tcl command, as opposed to errors in the
# processes that are invoked.

test exec-10.1 {errors in exec invocation} {exec} {
    list [catch {exec} msg] $msg
} {1 {wrong # args: should be "exec ?switches? arg ?arg ...?"}}
test exec-10.2 {errors in exec invocation} {exec} {
    list [catch {exec | cat} msg] $msg
} {1 {illegal use of | or |& in command}}
test exec-10.3 {errors in exec invocation} {exec} {
    list [catch {exec cat |} msg] $msg
} {1 {illegal use of | or |& in command}}
test exec-10.4 {errors in exec invocation} {exec} {
    list [catch {exec cat | | cat} msg] $msg
} {1 {illegal use of | or |& in command}}
test exec-10.5 {errors in exec invocation} {exec} {
    list [catch {exec cat | |& cat} msg] $msg
} {1 {illegal use of | or |& in command}}
test exec-10.6 {errors in exec invocation} {exec} {
    list [catch {exec cat |&} msg] $msg
} {1 {illegal use of | or |& in command}}
test exec-10.7 {errors in exec invocation} {exec} {
    list [catch {exec cat <} msg] $msg
} {1 {can't specify "<" as last word in command}}
test exec-10.8 {errors in exec invocation} {exec} {
    list [catch {exec cat >} msg] $msg
} {1 {can't specify ">" as last word in command}}
test exec-10.9 {errors in exec invocation} {exec} {
    list [catch {exec cat <<} msg] $msg
} {1 {can't specify "<<" as last word in command}}
test exec-10.10 {errors in exec invocation} {exec} {
    list [catch {exec cat >>} msg] $msg
} {1 {can't specify ">>" as last word in command}}
test exec-10.11 {errors in exec invocation} {exec} {
    list [catch {exec cat >&} msg] $msg
} {1 {can't specify ">&" as last word in command}}
test exec-10.12 {errors in exec invocation} {exec} {
    list [catch {exec cat >>&} msg] $msg
} {1 {can't specify ">>&" as last word in command}}
test exec-10.13 {errors in exec invocation} {exec} {
    list [catch {exec cat >@} msg] $msg
} {1 {can't specify ">@" as last word in command}}
test exec-10.14 {errors in exec invocation} {exec} {
    list [catch {exec cat <@} msg] $msg
} {1 {can't specify "<@" as last word in command}}
test exec-10.15 {errors in exec invocation} {exec} {
    list [catch {exec cat < a/b/c} msg] [string tolower $msg]
} {1 {couldn't read file "a/b/c": no such file or directory}}
test exec-10.16 {errors in exec invocation} {exec} {
    list [catch {exec cat << foo > a/b/c} msg] [string tolower $msg]
} {1 {couldn't write file "a/b/c": no such file or directory}}
test exec-10.17 {errors in exec invocation} {exec} {
    list [catch {exec cat << foo > a/b/c} msg] [string tolower $msg]
} {1 {couldn't write file "a/b/c": no such file or directory}}
set f [open $path(gorp.file) w]
test exec-10.18 {errors in exec invocation} {exec} {
    list [catch {exec cat <@ $f} msg] $msg
} "1 {channel \"$f\" wasn't opened for reading}"
close $f
set f [open $path(gorp.file) r]
test exec-10.19 {errors in exec invocation} {exec} {
    list [catch {exec cat >@ $f} msg] $msg
} "1 {channel \"$f\" wasn't opened for writing}"
close $f
test exec-10.20 {errors in exec invocation} {exec} {
    list [catch {exec ~non_existent_user/foo/bar} msg] $msg
} {1 {user "non_existent_user" doesn't exist}}
test exec-10.21 {errors in exec invocation} {exec} {
    list [catch {exec [interpreter] true | ~xyzzy_bad_user/x | false} msg] $msg
} {1 {user "xyzzy_bad_user" doesn't exist}}

# Commands in background.

test exec-11.1 {commands in background} {exec} {
    set x [lindex [time {exec [interpreter] $path(sleep) 2 &}] 0]
    expr $x<1000000
} 1
test exec-11.2 {commands in background} {exec} {
    list [catch {exec [interpreter] $path(echo) a &b} msg] $msg
} {0 {a &b}}
test exec-11.3 {commands in background} {exec} {
    llength [exec [interpreter] $path(sleep) 1 &]
} 1
test exec-11.4 {commands in background} {exec stdio} {
    llength [exec [interpreter] $path(sleep) 1 | [interpreter] $path(sleep) 1 | [interpreter] $path(sleep) 1 &]
} 3
test exec-11.5 {commands in background} {exec} {
    set f [open $path(gorp.file) w]
    puts $f [list catch [list exec [info nameofexecutable] $path(echo) foo &]]
    close $f
    string compare "foo" [exec [interpreter] $path(gorp.file)]
} 0

# Make sure that background commands are properly reaped when
# they eventually die.

if { [set ::tcltest::testConstraints(exec)] } {
exec [interpreter] $path(sleep) 3
}
test exec-12.1 {reaping background processes} \
	{exec unixOnly nonPortable} {
    for {set i 0} {$i < 20} {incr i} {
	exec echo foo > /dev/null &
    }
    exec sleep 1
    catch {exec ps | fgrep "echo foo" | fgrep -v fgrep | wc} msg
    lindex $msg 0
} 0
test exec-12.2 {reaping background processes} \
	{exec unixOnly nonPortable} {
    exec sleep 2 | sleep 2 | sleep 2 &
    catch {exec ps | fgrep -i "sleep" | fgrep -i -v fgrep | wc} msg
    set x [lindex $msg 0]
    exec sleep 3
    catch {exec ps | fgrep -i "sleep" | fgrep -i -v fgrep | wc} msg
    list $x [lindex $msg 0]
} {3 0}
test exec-12.3 {reaping background processes} \
	{exec unixOnly nonPortable} {
    exec sleep 1000 &
    exec sleep 1000 &
    set x [exec ps | fgrep "sleep" | fgrep -v fgrep]
    set pids {}
    foreach i [split $x \n] {
	lappend pids [lindex $i 0]
    }
    foreach i $pids {
	catch {exec kill -STOP $i}
    }
    catch {exec ps | fgrep "sleep" | fgrep -v fgrep | wc} msg
    set x [lindex $msg 0]

    foreach i $pids {
	catch {exec kill -KILL $i}
    }
    catch {exec ps | fgrep "sleep" | fgrep -v fgrep | wc} msg
    list $x [lindex $msg 0]
} {2 0}

# Make sure "errorCode" is set correctly.

test exec-13.1 {setting errorCode variable} {exec} {
    list [catch {exec [interpreter] $path(cat) < a/b/c} msg] [string tolower $errorCode]
} {1 {posix enoent {no such file or directory}}}
test exec-13.2 {setting errorCode variable} {exec} {
    list [catch {exec [interpreter] $path(cat) > a/b/c} msg] [string tolower $errorCode]
} {1 {posix enoent {no such file or directory}}}
test exec-13.3 {setting errorCode variable} {exec} {
    set x [catch {exec _weird_cmd_} msg]
    list $x [string tolower $msg] [lindex $errorCode 0] \
	    [string tolower [lrange $errorCode 2 end]]
} {1 {couldn't execute "_weird_cmd_": no such file or directory} POSIX {{no such file or directory}}}

# Switches before the first argument

test exec-14.1 {-keepnewline switch} {exec} {
    exec -keepnewline [interpreter] $path(echo) foo
} "foo\n"
test exec-14.2 {-keepnewline switch} {exec} {
    list [catch {exec -keepnewline} msg] $msg
} {1 {wrong # args: should be "exec ?switches? arg ?arg ...?"}}
test exec-14.3 {unknown switch} {exec} {
    list [catch {exec -gorp} msg] $msg
} {1 {bad switch "-gorp": must be -keepnewline or --}}
test exec-14.4 {-- switch} {exec} {
    list [catch {exec -- -gorp} msg] [string tolower $msg]
} {1 {couldn't execute "-gorp": no such file or directory}}

# Redirecting standard error separately from standard output

test exec-15.1 {standard error redirection} {exec} {
    exec [interpreter] "$path(echo)" "First line" > "$path(gorp.file)"
    list [exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" 2> "$path(gorp.file)"] \
	    [exec [interpreter] "$path(cat)" "$path(gorp.file)"]
} {{} {foo bar}}
test exec-15.2 {standard error redirection} {exec stdio} {
    list [exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" \
		| [interpreter] "$path(echo)" biz baz >$path(gorp.file) 2> "$path(gorp.file2)"] \
	    [exec [interpreter] "$path(cat)" "$path(gorp.file)"] \
	    [exec [interpreter] "$path(cat)" "$path(gorp.file2)"]
} {{} {biz baz} {foo bar}}
test exec-15.3 {standard error redirection} {exec stdio} {
    list [exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" \
	        | [interpreter] "$path(echo)" biz baz 2>$path(gorp.file) > "$path(gorp.file2)"] \
	    [exec [interpreter] "$path(cat)" "$path(gorp.file)"] \
	    [exec [interpreter] "$path(cat)" "$path(gorp.file2)"]
} {{} {foo bar} {biz baz}}
test exec-15.4 {standard error redirection} {exec} {
    set f [open "$path(gorp.file)" w]
    puts $f "Line 1"
    flush $f
    exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" 2>@ $f
    puts $f "Line 3"
    close $f
    exec [interpreter] "$path(cat)" "$path(gorp.file)"
} {Line 1
foo bar
Line 3}
test exec-15.5 {standard error redirection} {exec} {
    exec [interpreter] "$path(echo)" "First line" > "$path(gorp.file)"
    exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" 2>> "$path(gorp.file)"
    exec [interpreter] "$path(cat)" "$path(gorp.file)"
} {First line
foo bar}
test exec-15.6 {standard error redirection} {exec stdio} {
    exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" > "$path(gorp.file2)" 2> "$path(gorp.file)" \
	    >& "$path(gorp.file)" 2> "$path(gorp.file2)" | [interpreter] "$path(echo)" biz baz
    list [exec [interpreter] "$path(cat)" "$path(gorp.file)"] [exec [interpreter] "$path(cat)" "$path(gorp.file2)"]
} {{biz baz} {foo bar}}
test exec-15.7 {standard error redirection 2>@1} {exec stdio} {
    # This redirects stderr output into normal result output from exec
    exec [interpreter] "$path(sh)" -c "\"$path(echo)\" foo bar 1>&2" 2>@1
} {foo bar}

test exec-16.1 {flush output before exec} {exec} {
    set f [open $path(gorp.file) w]
    puts $f "First line"
    exec [interpreter] $path(echo) "Second line" >@ $f
    puts $f "Third line"
    close $f
    exec [interpreter] $path(cat) $path(gorp.file)
} {First line
Second line
Third line}
test exec-16.2 {flush output before exec} {exec} {
    set f [open $path(gorp.file) w]
    puts $f "First line"
    exec [interpreter] << {puts stderr {Second line}} >&@ $f > $path(gorp.file2)
    puts $f "Third line"
    close $f
    exec [interpreter] $path(cat) $path(gorp.file)
} {First line
Second line
Third line}

set path(script) [makeFile {} script]

test exec-17.1 { inheriting standard I/O } {exec} {
    set f [open $path(script) w]
    puts -nonewline $f {close stdout
	set f [}
    puts $f [list open $path(gorp.file) w]]
    puts $f [list catch \
	    [list exec [info nameofexecutable] $path(echo) foobar &]]
    puts $f [list exec [info nameofexecutable] $path(sleep) 2]
    puts $f {close $f}
    close $f
    catch {exec [interpreter] $path(script)} result
    set f [open $path(gorp.file) r]
    lappend result [read $f]
    close $f
    set result
} {{foobar
}}

test exec-18.1 { exec cat deals with weird file names} {exec tempNotWin} {
    # This is cross-platform, but the cat isn't predictably correct on
    # Windows.
    set f "foo\[\{blah"
    set path(fooblah) [makeFile {} $f]
    set fout [open $path(fooblah) w]
    puts $fout "contents"
    close $fout
    set res [list [catch {exec cat $path(fooblah)} msg] $msg]
    removeFile $f
    set res
} {0 contents}

# cleanup

foreach file {script gorp.file gorp.file2 echo cat wc sh sleep exit err} {
    removeFile $file
}
unset -nocomplain path

::tcltest::cleanupTests
return
