/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.tooling.ui.queryregistry.index;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Map;
import java.util.TreeMap;
import java.util.WeakHashMap;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.viatra.query.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.Pattern;
import org.eclipse.viatra.query.patternlanguage.patternLanguage.PatternLanguagePackage;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.extensibility.IQuerySpecificationProvider;
import org.eclipse.viatra.query.runtime.registry.IConnectorListener;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistry;
import org.eclipse.viatra.query.runtime.registry.connector.AbstractRegistrySourceConnector;
import org.eclipse.viatra.query.runtime.util.ViatraQueryLoggingUtil;
import org.eclipse.viatra.query.tooling.ui.queryregistry.index.IPatternBasedSpecificationProvider;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.ui.notification.IStateChangeEventBroker;
import org.eclipse.xtext.ui.resource.IResourceSetProvider;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class XtextIndexBasedRegistryUpdater {
  @FinalFieldsConstructor
  private static final class QueryRegistryUpdaterListener implements IResourceDescription.Event.Listener {
    private final XtextIndexBasedRegistryUpdater updater;
    
    @Override
    public void descriptionsChanged(final IResourceDescription.Event event) {
      ImmutableList<IResourceDescription.Delta> _deltas = event.getDeltas();
      final Procedure1<IResourceDescription.Delta> _function = new Procedure1<IResourceDescription.Delta>() {
        @Override
        public void apply(final IResourceDescription.Delta delta) {
          final IResourceDescription oldDesc = delta.getOld();
          final IResourceDescription newDesc = delta.getNew();
          URI _uri = delta.getUri();
          final String uri = _uri.toString();
          URI _uri_1 = delta.getUri();
          boolean _isPlatformResource = _uri_1.isPlatformResource();
          boolean _not = (!_isPlatformResource);
          if (_not) {
            return;
          }
          URI _uri_2 = delta.getUri();
          final String projectName = _uri_2.segment(1);
          IWorkspace _workspace = ResourcesPlugin.getWorkspace();
          IWorkspaceRoot _root = _workspace.getRoot();
          final IProject project = _root.getProject(projectName);
          final boolean projectExists = project.exists();
          if ((!projectExists)) {
            return;
          }
          final String connectorId = (XtextIndexBasedRegistryUpdater.DYNAMIC_CONNECTOR_ID_PREFIX + projectName);
          try {
            boolean _notEquals = (!Objects.equal(oldDesc, null));
            if (_notEquals) {
              boolean _or = false;
              boolean _equals = Objects.equal(newDesc, null);
              if (_equals) {
                _or = true;
              } else {
                boolean _isOpen = project.isOpen();
                boolean _not_1 = (!_isOpen);
                _or = _not_1;
              }
              if (_or) {
                final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = QueryRegistryUpdaterListener.this.updater.connectorMap.get(connectorId);
                boolean _notEquals_1 = (!Objects.equal(connector, null));
                if (_notEquals_1) {
                  connector.clearProviders(uri);
                  boolean _isEmpty = connector.descriptionToProvider.isEmpty();
                  if (_isEmpty) {
                    QueryRegistryUpdaterListener.this.updater.connectedRegistry.removeSource(connector);
                    QueryRegistryUpdaterListener.this.updater.connectorMap.remove(connectorId);
                  }
                }
              } else {
                QueryRegistryUpdaterListener.this.processResourceDescription(delta, newDesc, connectorId, projectName);
              }
            } else {
              boolean _and = false;
              boolean _notEquals_2 = (!Objects.equal(newDesc, null));
              if (!_notEquals_2) {
                _and = false;
              } else {
                Iterable<IEObjectDescription> _exportedObjectsByType = newDesc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN);
                boolean _isEmpty_1 = IterableExtensions.isEmpty(_exportedObjectsByType);
                boolean _not_2 = (!_isEmpty_1);
                _and = _not_2;
              }
              if (_and) {
                QueryRegistryUpdaterListener.this.processResourceDescription(delta, newDesc, connectorId, projectName);
              }
            }
          } catch (final Throwable _t) {
            if (_t instanceof Exception) {
              final Exception ex = (Exception)_t;
              final Logger logger = ViatraQueryLoggingUtil.getLogger(XtextIndexBasedRegistryUpdater.class);
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("Could not update registry based on Xtext index for ");
              _builder.append(uri, "");
              logger.error(_builder, ex);
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        }
      };
      IterableExtensions.<IResourceDescription.Delta>forEach(_deltas, _function);
    }
    
    public Object processResourceDescription(final IResourceDescription.Delta delta, final IResourceDescription desc, final String connectorId, final String projectName) {
      Object _xifexpression = null;
      boolean _containsKey = this.updater.connectorMap.containsKey(connectorId);
      if (_containsKey) {
        URI _uRI = desc.getURI();
        _xifexpression = this.updater.workspaceListener.connectorsToUpdate.put(_uRI, desc);
      } else {
        boolean _xifexpression_1 = false;
        Iterable<IEObjectDescription> _exportedObjectsByType = desc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN);
        boolean _isEmpty = IterableExtensions.isEmpty(_exportedObjectsByType);
        boolean _not = (!_isEmpty);
        if (_not) {
          boolean _xblockexpression = false;
          {
            final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector(connectorId);
            this.updater.connectorMap.put(connectorId, connector);
            boolean _haveEObjectDescriptionsChanged = delta.haveEObjectDescriptionsChanged();
            if (_haveEObjectDescriptionsChanged) {
              final ResourceSet resourceSet = this.updater.createResourceSet(projectName);
              Iterable<IEObjectDescription> _exportedObjectsByType_1 = desc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN);
              final Procedure1<IEObjectDescription> _function = new Procedure1<IEObjectDescription>() {
                @Override
                public void apply(final IEObjectDescription it) {
                  final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider(QueryRegistryUpdaterListener.this.updater, desc, it, resourceSet);
                  URI _uRI = desc.getURI();
                  String _string = _uRI.toString();
                  connector.addProvider(_string, provider);
                }
              };
              IterableExtensions.<IEObjectDescription>forEach(_exportedObjectsByType_1, _function);
            }
            _xblockexpression = this.updater.connectedRegistry.addSource(connector);
          }
          _xifexpression_1 = _xblockexpression;
        }
        _xifexpression = Boolean.valueOf(_xifexpression_1);
      }
      return _xifexpression;
    }
    
    public QueryRegistryUpdaterListener(final XtextIndexBasedRegistryUpdater updater) {
      super();
      this.updater = updater;
    }
  }
  
  @FinalFieldsConstructor
  private static final class PatternDescriptionBasedSpecificationProvider implements IPatternBasedSpecificationProvider {
    private final XtextIndexBasedRegistryUpdater updater;
    
    private final IResourceDescription resourceDesc;
    
    private final IEObjectDescription description;
    
    private final ResourceSet resourceSet;
    
    private IQuerySpecification<?> specification;
    
    @Override
    public String getFullyQualifiedName() {
      QualifiedName _qualifiedName = this.description.getQualifiedName();
      return _qualifiedName.toString();
    }
    
    @Override
    public IQuerySpecification<?> get() {
      try {
        boolean _equals = Objects.equal(this.specification, null);
        if (_equals) {
          final Pattern pattern = this.findPatternForDescription();
          final SpecificationBuilder builder = new SpecificationBuilder();
          IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> _orCreateSpecification = builder.getOrCreateSpecification(pattern);
          this.specification = _orCreateSpecification;
        }
        return this.specification;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    @Override
    public IQuerySpecification<?> getSpecification(final SpecificationBuilder builder) {
      try {
        final Pattern pattern = this.findPatternForDescription();
        final IQuerySpecification<? extends ViatraQueryMatcher<? extends IPatternMatch>> spec = builder.getOrCreateSpecification(pattern);
        return spec;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    public Pattern findPatternForDescription() {
      EObject pattern = this.description.getEObjectOrProxy();
      boolean _eIsProxy = pattern.eIsProxy();
      if (_eIsProxy) {
        EObject _resolve = EcoreUtil.resolve(pattern, this.resourceSet);
        pattern = _resolve;
      }
      boolean _eIsProxy_1 = pattern.eIsProxy();
      if (_eIsProxy_1) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Cannot load specification ");
        String _fullyQualifiedName = this.getFullyQualifiedName();
        _builder.append(_fullyQualifiedName, "");
        _builder.append(" from Xtext index");
        throw new IllegalStateException(_builder.toString());
      }
      return ((Pattern) pattern);
    }
    
    @Override
    public String getSourceProjectName() {
      URI _uRI = this.resourceDesc.getURI();
      return _uRI.segment(1);
    }
    
    @Override
    public URI getSpecificationSourceURI() {
      return this.description.getEObjectURI();
    }
    
    public PatternDescriptionBasedSpecificationProvider(final XtextIndexBasedRegistryUpdater updater, final IResourceDescription resourceDesc, final IEObjectDescription description, final ResourceSet resourceSet) {
      super();
      this.updater = updater;
      this.resourceDesc = resourceDesc;
      this.description = description;
      this.resourceSet = resourceSet;
    }
  }
  
  private static final class PatternDescriptionBasedSourceConnector extends AbstractRegistrySourceConnector {
    private final Multimap<String, IQuerySpecificationProvider> descriptionToProvider;
    
    public PatternDescriptionBasedSourceConnector(final String identifier) {
      super(identifier, false);
      HashMultimap<String, IQuerySpecificationProvider> _create = HashMultimap.<String, IQuerySpecificationProvider>create();
      this.descriptionToProvider = _create;
    }
    
    public void addProvider(final String resourceUri, final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider) {
      this.descriptionToProvider.put(resourceUri, provider);
      final Procedure1<IConnectorListener> _function = new Procedure1<IConnectorListener>() {
        @Override
        public void apply(final IConnectorListener it) {
          it.querySpecificationAdded(PatternDescriptionBasedSourceConnector.this, provider);
        }
      };
      IterableExtensions.<IConnectorListener>forEach(this.listeners, _function);
    }
    
    public Collection<IQuerySpecificationProvider> clearProviders(final String resourceUri) {
      Collection<IQuerySpecificationProvider> _xblockexpression = null;
      {
        Collection<IQuerySpecificationProvider> _get = this.descriptionToProvider.get(resourceUri);
        final Procedure1<IQuerySpecificationProvider> _function = new Procedure1<IQuerySpecificationProvider>() {
          @Override
          public void apply(final IQuerySpecificationProvider provider) {
            final Procedure1<IConnectorListener> _function = new Procedure1<IConnectorListener>() {
              @Override
              public void apply(final IConnectorListener it) {
                it.querySpecificationRemoved(PatternDescriptionBasedSourceConnector.this, provider);
              }
            };
            IterableExtensions.<IConnectorListener>forEach(PatternDescriptionBasedSourceConnector.this.listeners, _function);
          }
        };
        IterableExtensions.<IQuerySpecificationProvider>forEach(_get, _function);
        _xblockexpression = this.descriptionToProvider.removeAll(resourceUri);
      }
      return _xblockexpression;
    }
    
    @Override
    protected void sendQuerySpecificationsToListener(final IConnectorListener listener) {
      Collection<IQuerySpecificationProvider> _values = this.descriptionToProvider.values();
      final Procedure1<IQuerySpecificationProvider> _function = new Procedure1<IQuerySpecificationProvider>() {
        @Override
        public void apply(final IQuerySpecificationProvider it) {
          listener.querySpecificationAdded(PatternDescriptionBasedSourceConnector.this, it);
        }
      };
      IterableExtensions.<IQuerySpecificationProvider>forEach(_values, _function);
    }
  }
  
  @FinalFieldsConstructor
  private static final class WorkspaceBuildCompletedListener implements IResourceChangeListener {
    private final Map<URI, IResourceDescription> connectorsToUpdate = CollectionLiterals.<URI, IResourceDescription>newHashMap();
    
    private final XtextIndexBasedRegistryUpdater updater;
    
    @Override
    public void resourceChanged(final IResourceChangeEvent event) {
      final int type = event.getType();
      if ((type == IResourceChangeEvent.POST_CHANGE)) {
        boolean _isEmpty = this.connectorsToUpdate.isEmpty();
        if (_isEmpty) {
          return;
        }
        final ImmutableMap<URI, IResourceDescription> update = ImmutableMap.<URI, IResourceDescription>copyOf(this.connectorsToUpdate);
        final Procedure2<URI, IResourceDescription> _function = new Procedure2<URI, IResourceDescription>() {
          @Override
          public void apply(final URI uri, final IResourceDescription descr) {
            try {
              WorkspaceBuildCompletedListener.this.connectorsToUpdate.remove(uri);
              final String projectName = uri.segment(1);
              final String connectorId = (XtextIndexBasedRegistryUpdater.DYNAMIC_CONNECTOR_ID_PREFIX + projectName);
              final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = WorkspaceBuildCompletedListener.this.updater.connectorMap.get(connectorId);
              String _string = uri.toString();
              connector.clearProviders(_string);
              final ResourceSet resourceSet = WorkspaceBuildCompletedListener.this.updater.createResourceSet(projectName);
              final Iterable<IEObjectDescription> patternObjects = descr.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN);
              final Procedure1<IEObjectDescription> _function = new Procedure1<IEObjectDescription>() {
                @Override
                public void apply(final IEObjectDescription it) {
                  final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider(WorkspaceBuildCompletedListener.this.updater, descr, it, resourceSet);
                  String _string = uri.toString();
                  connector.addProvider(_string, provider);
                }
              };
              IterableExtensions.<IEObjectDescription>forEach(patternObjects, _function);
            } catch (final Throwable _t) {
              if (_t instanceof Exception) {
                final Exception ex = (Exception)_t;
                final Logger logger = ViatraQueryLoggingUtil.getLogger(XtextIndexBasedRegistryUpdater.class);
                StringConcatenation _builder = new StringConcatenation();
                _builder.append("Could not update registry based on Xtext index for ");
                _builder.append(uri, "");
                logger.error(_builder, ex);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
          }
        };
        MapExtensions.<URI, IResourceDescription>forEach(update, _function);
      }
    }
    
    public WorkspaceBuildCompletedListener(final XtextIndexBasedRegistryUpdater updater) {
      super();
      this.updater = updater;
    }
  }
  
  public final static String DYNAMIC_CONNECTOR_ID_PREFIX = "dynamic:";
  
  private final IStateChangeEventBroker source;
  
  private final IResourceDescriptions descriptions;
  
  private final IResourceSetProvider resourceSetProvider;
  
  private final XtextIndexBasedRegistryUpdater.QueryRegistryUpdaterListener listener;
  
  private final Map<String, XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> connectorMap;
  
  private final XtextIndexBasedRegistryUpdater.WorkspaceBuildCompletedListener workspaceListener;
  
  @Accessors(AccessorType.PROTECTED_GETTER)
  private IQuerySpecificationRegistry connectedRegistry;
  
  private Map<IProject, ResourceSet> resourceSetMap = new WeakHashMap<IProject, ResourceSet>();
  
  @Inject
  public XtextIndexBasedRegistryUpdater(final IStateChangeEventBroker source, final IResourceDescriptions descriptions, final IResourceSetProvider resSetProvider) {
    super();
    this.source = source;
    this.descriptions = descriptions;
    this.resourceSetProvider = resSetProvider;
    XtextIndexBasedRegistryUpdater.WorkspaceBuildCompletedListener _workspaceBuildCompletedListener = new XtextIndexBasedRegistryUpdater.WorkspaceBuildCompletedListener(this);
    this.workspaceListener = _workspaceBuildCompletedListener;
    XtextIndexBasedRegistryUpdater.QueryRegistryUpdaterListener _queryRegistryUpdaterListener = new XtextIndexBasedRegistryUpdater.QueryRegistryUpdaterListener(this);
    this.listener = _queryRegistryUpdaterListener;
    TreeMap<String, XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> _newTreeMap = Maps.<String, XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector>newTreeMap();
    this.connectorMap = _newTreeMap;
  }
  
  public void connectIndexToRegistry(final IQuerySpecificationRegistry registry) {
    boolean _equals = Objects.equal(this.connectedRegistry, null);
    if (_equals) {
      this.connectedRegistry = registry;
      Iterable<IResourceDescription> _allResourceDescriptions = this.descriptions.getAllResourceDescriptions();
      final Procedure1<IResourceDescription> _function = new Procedure1<IResourceDescription>() {
        @Override
        public void apply(final IResourceDescription resourceDesc) {
          URI _uRI = resourceDesc.getURI();
          boolean _isPlatformResource = _uRI.isPlatformResource();
          boolean _not = (!_isPlatformResource);
          if (_not) {
            return;
          }
          final Iterable<IEObjectDescription> patternObjects = resourceDesc.getExportedObjectsByType(PatternLanguagePackage.Literals.PATTERN);
          boolean _isEmpty = IterableExtensions.isEmpty(patternObjects);
          if (_isEmpty) {
            return;
          }
          URI _uRI_1 = resourceDesc.getURI();
          final String uri = _uRI_1.toString();
          URI _uRI_2 = resourceDesc.getURI();
          final String projectName = _uRI_2.segment(1);
          IWorkspace _workspace = ResourcesPlugin.getWorkspace();
          IWorkspaceRoot _root = _workspace.getRoot();
          final IProject project = _root.getProject(projectName);
          final boolean projectExists = project.exists();
          boolean _or = false;
          if ((!projectExists)) {
            _or = true;
          } else {
            boolean _isOpen = project.isOpen();
            boolean _not_1 = (!_isOpen);
            _or = _not_1;
          }
          if (_or) {
            return;
          }
          final String connectorId = (XtextIndexBasedRegistryUpdater.DYNAMIC_CONNECTOR_ID_PREFIX + projectName);
          XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector = XtextIndexBasedRegistryUpdater.this.connectorMap.get(connectorId);
          boolean _equals = Objects.equal(connector, null);
          if (_equals) {
            XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector _patternDescriptionBasedSourceConnector = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector(connectorId);
            connector = _patternDescriptionBasedSourceConnector;
            XtextIndexBasedRegistryUpdater.this.connectorMap.put(connectorId, connector);
          }
          final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector conn = connector;
          final ResourceSet resourceSet = XtextIndexBasedRegistryUpdater.this.createResourceSet(projectName);
          final Procedure1<IEObjectDescription> _function = new Procedure1<IEObjectDescription>() {
            @Override
            public void apply(final IEObjectDescription it) {
              final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider provider = new XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSpecificationProvider(XtextIndexBasedRegistryUpdater.this, resourceDesc, it, resourceSet);
              conn.addProvider(uri, provider);
            }
          };
          IterableExtensions.<IEObjectDescription>forEach(patternObjects, _function);
        }
      };
      IterableExtensions.<IResourceDescription>forEach(_allResourceDescriptions, _function);
      Collection<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> _values = this.connectorMap.values();
      final Procedure1<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> _function_1 = new Procedure1<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector>() {
        @Override
        public void apply(final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector connector) {
          registry.addSource(connector);
        }
      };
      IterableExtensions.<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector>forEach(_values, _function_1);
      this.source.addListener(this.listener);
      IWorkspace _workspace = ResourcesPlugin.getWorkspace();
      _workspace.addResourceChangeListener(this.workspaceListener);
    }
  }
  
  public IQuerySpecificationRegistry disconnectIndexFromRegistry() {
    IQuerySpecificationRegistry _xifexpression = null;
    boolean _notEquals = (!Objects.equal(this.connectedRegistry, null));
    if (_notEquals) {
      IQuerySpecificationRegistry _xblockexpression = null;
      {
        Collection<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> _values = this.connectorMap.values();
        final Procedure1<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector> _function = new Procedure1<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector>() {
          @Override
          public void apply(final XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector it) {
            XtextIndexBasedRegistryUpdater.this.connectedRegistry.removeSource(it);
          }
        };
        IterableExtensions.<XtextIndexBasedRegistryUpdater.PatternDescriptionBasedSourceConnector>forEach(_values, _function);
        this.source.removeListener(this.listener);
        this.connectorMap.clear();
        _xblockexpression = this.connectedRegistry = null;
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  public ResourceSet createResourceSet(final String projectName) {
    IWorkspace _workspace = ResourcesPlugin.getWorkspace();
    final IWorkspaceRoot root = _workspace.getRoot();
    final IProject project = root.getProject(projectName);
    final ResourceSet resourceSet = this.resourceSetProvider.get(project);
    return resourceSet;
  }
  
  @Pure
  protected IQuerySpecificationRegistry getConnectedRegistry() {
    return this.connectedRegistry;
  }
}
