/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.query.tooling.ui.queryresult;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import com.google.common.collect.TreeMultimap;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.viatra.query.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.viatra.query.runtime.api.AdvancedViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.IMatchProcessor;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngineLifecycleListener;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.api.scope.QueryScope;
import org.eclipse.viatra.query.runtime.base.api.BaseIndexOptions;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.runtime.extensibility.IQuerySpecificationProvider;
import org.eclipse.viatra.query.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistry;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryChangeListener;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryEntry;
import org.eclipse.viatra.query.runtime.registry.IRegistryView;
import org.eclipse.viatra.query.runtime.registry.IRegistryViewFactory;
import org.eclipse.viatra.query.runtime.registry.view.AbstractRegistryView;
import org.eclipse.viatra.query.tooling.ui.ViatraQueryGUIPlugin;
import org.eclipse.viatra.query.tooling.ui.browser.ViatraQueryToolingBrowserPlugin;
import org.eclipse.viatra.query.tooling.ui.queryexplorer.IModelConnector;
import org.eclipse.viatra.query.tooling.ui.queryregistry.index.IPatternBasedSpecificationProvider;
import org.eclipse.viatra.query.tooling.ui.queryresult.IQueryResultViewModelListener;
import org.eclipse.viatra.query.tooling.ui.queryresult.QueryResultTreeMatcher;
import org.eclipse.viatra.query.tooling.ui.util.IFilteredMatcherCollection;
import org.eclipse.viatra.query.tooling.ui.util.IFilteredMatcherContent;
import org.eclipse.viatra.transformation.evm.api.ActivationLifeCycle;
import org.eclipse.viatra.transformation.evm.api.ExecutionSchema;
import org.eclipse.viatra.transformation.evm.api.Job;
import org.eclipse.viatra.transformation.evm.api.RuleSpecification;
import org.eclipse.viatra.transformation.evm.specific.ExecutionSchemas;
import org.eclipse.viatra.transformation.evm.specific.Jobs;
import org.eclipse.viatra.transformation.evm.specific.Lifecycles;
import org.eclipse.viatra.transformation.evm.specific.Rules;
import org.eclipse.viatra.transformation.evm.specific.Schedulers;
import org.eclipse.viatra.transformation.evm.specific.crud.CRUDActivationStateEnum;
import org.eclipse.viatra.transformation.evm.specific.resolver.InvertedDisappearancePriorityConflictResolver;
import org.eclipse.viatra.transformation.evm.specific.scheduler.UpdateCompleteBasedScheduler;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;
import org.osgi.framework.Bundle;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class QueryResultTreeInput implements IFilteredMatcherCollection {
  @FinalFieldsConstructor
  public static class RegistryChangeListener implements IQuerySpecificationRegistryChangeListener {
    private final QueryResultTreeInput input;
    
    @Override
    public void entryAdded(final IQuerySpecificationRegistryEntry entry) {
      this.input.addMatcherIfLoaded(entry);
    }
    
    @Override
    public void entryRemoved(final IQuerySpecificationRegistryEntry entry) {
      this.input.removeMatcherIfLoaded(entry);
    }
    
    public RegistryChangeListener(final QueryResultTreeInput input) {
      super();
      this.input = input;
    }
  }
  
  @FinalFieldsConstructor
  public static class EngineLifecycleListener implements ViatraQueryEngineLifecycleListener {
    private final QueryResultTreeInput input;
    
    @Override
    public void engineBecameTainted(final String message, final Throwable t) {
      this.input.dispose();
    }
    
    @Override
    public void engineDisposed() {
      this.input.dispose();
    }
    
    @Override
    public void engineWiped() {
      this.input.resetInput();
    }
    
    @Override
    public void matcherInstantiated(final ViatraQueryMatcher<? extends IPatternMatch> matcher) {
      this.input.createMatcher(matcher);
    }
    
    public EngineLifecycleListener(final QueryResultTreeInput input) {
      super();
      this.input = input;
    }
  }
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private AdvancedViatraQueryEngine engine;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private boolean readOnlyEngine;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private Map<String, QueryResultTreeMatcher> matchers;
  
  /**
   * @since 1.4
   */
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  private IModelConnector modelConnector;
  
  private Table<String, String, IQuerySpecificationRegistryEntry> loadedEntries;
  
  private Multimap<String, String> knownErrorEntries;
  
  private SpecificationBuilder builder;
  
  private ExecutionSchema schema;
  
  private QueryResultTreeInput.EngineLifecycleListener lifecycleListener;
  
  private QueryResultTreeInput.RegistryChangeListener registryListener;
  
  private IRegistryView view;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private QueryEvaluationHint hint;
  
  private Set<IQueryResultViewModelListener> listeners;
  
  public QueryResultTreeInput(final AdvancedViatraQueryEngine engine, final IQuerySpecificationRegistry registry, final boolean readOnlyEngine, final QueryEvaluationHint hint) {
    this.engine = engine;
    this.hint = hint;
    this.readOnlyEngine = readOnlyEngine;
    TreeMap<String, QueryResultTreeMatcher> _newTreeMap = Maps.<String, QueryResultTreeMatcher>newTreeMap();
    this.matchers = _newTreeMap;
    HashBasedTable<String, String, IQuerySpecificationRegistryEntry> _create = HashBasedTable.<String, String, IQuerySpecificationRegistryEntry>create();
    this.loadedEntries = _create;
    TreeMultimap<String, String> _create_1 = TreeMultimap.<String, String>create();
    this.knownErrorEntries = _create_1;
    SpecificationBuilder _specificationBuilder = new SpecificationBuilder();
    this.builder = _specificationBuilder;
    HashSet<IQueryResultViewModelListener> _newHashSet = Sets.<IQueryResultViewModelListener>newHashSet();
    this.listeners = _newHashSet;
    UpdateCompleteBasedScheduler.UpdateCompleteBasedSchedulerFactory _queryEngineSchedulerFactory = Schedulers.getQueryEngineSchedulerFactory(engine);
    InvertedDisappearancePriorityConflictResolver _invertedDisappearancePriorityConflictResolver = new InvertedDisappearancePriorityConflictResolver();
    ExecutionSchema _createViatraQueryExecutionSchema = ExecutionSchemas.createViatraQueryExecutionSchema(engine, _queryEngineSchedulerFactory, _invertedDisappearancePriorityConflictResolver);
    this.schema = _createViatraQueryExecutionSchema;
    Set<? extends ViatraQueryMatcher<? extends IPatternMatch>> _currentMatchers = engine.getCurrentMatchers();
    final Procedure1<ViatraQueryMatcher<? extends IPatternMatch>> _function = new Procedure1<ViatraQueryMatcher<? extends IPatternMatch>>() {
      @Override
      public void apply(final ViatraQueryMatcher<? extends IPatternMatch> it) {
        final QueryResultTreeMatcher treeMatcher = QueryResultTreeInput.this.createMatcher(it);
      }
    };
    IterableExtensions.forEach(_currentMatchers, _function);
    this.schema.startUnscheduledExecution();
    QueryResultTreeInput.EngineLifecycleListener _engineLifecycleListener = new QueryResultTreeInput.EngineLifecycleListener(this);
    this.lifecycleListener = _engineLifecycleListener;
    engine.addLifecycleListener(this.lifecycleListener);
    if ((!readOnlyEngine)) {
      QueryResultTreeInput.RegistryChangeListener _registryChangeListener = new QueryResultTreeInput.RegistryChangeListener(this);
      this.registryListener = _registryChangeListener;
      final IRegistryViewFactory _function_1 = new IRegistryViewFactory() {
        @Override
        public IRegistryView createView(final IQuerySpecificationRegistry it) {
          return new AbstractRegistryView(registry, true) {
            @Override
            protected boolean isEntryRelevant(final IQuerySpecificationRegistryEntry entry) {
              return true;
            }
          };
        }
      };
      IRegistryView _createView = registry.createView(_function_1);
      this.view = _createView;
      this.view.addViewListener(this.registryListener);
    }
  }
  
  public QueryEvaluationHint setHint(final QueryEvaluationHint hint) {
    QueryEvaluationHint _xblockexpression = null;
    {
      Preconditions.<QueryEvaluationHint>checkNotNull(hint);
      _xblockexpression = this.hint = hint;
    }
    return _xblockexpression;
  }
  
  public QueryResultTreeMatcher createMatcher(final ViatraQueryMatcher matcher) {
    final QueryResultTreeMatcher treeMatcher = new QueryResultTreeMatcher(this, matcher);
    final IMatchProcessor<IPatternMatch> _function = new IMatchProcessor<IPatternMatch>() {
      @Override
      public void process(final IPatternMatch match) {
        final Procedure1<IQueryResultViewModelListener> _function = new Procedure1<IQueryResultViewModelListener>() {
          @Override
          public void apply(final IQueryResultViewModelListener it) {
            it.matchAdded(treeMatcher, match);
          }
        };
        IterableExtensions.<IQueryResultViewModelListener>forEach(QueryResultTreeInput.this.listeners, _function);
      }
    };
    final Job<IPatternMatch> matchCreatedJob = Jobs.<IPatternMatch>newStatelessJob(CRUDActivationStateEnum.CREATED, _function);
    final IMatchProcessor<IPatternMatch> _function_1 = new IMatchProcessor<IPatternMatch>() {
      @Override
      public void process(final IPatternMatch match) {
        final Procedure1<IQueryResultViewModelListener> _function = new Procedure1<IQueryResultViewModelListener>() {
          @Override
          public void apply(final IQueryResultViewModelListener it) {
            it.matchUpdated(treeMatcher, match);
          }
        };
        IterableExtensions.<IQueryResultViewModelListener>forEach(QueryResultTreeInput.this.listeners, _function);
      }
    };
    final Job<IPatternMatch> matchUpdatedJob = Jobs.<IPatternMatch>newStatelessJob(CRUDActivationStateEnum.UPDATED, _function_1);
    final IMatchProcessor<IPatternMatch> _function_2 = new IMatchProcessor<IPatternMatch>() {
      @Override
      public void process(final IPatternMatch match) {
        final Procedure1<IQueryResultViewModelListener> _function = new Procedure1<IQueryResultViewModelListener>() {
          @Override
          public void apply(final IQueryResultViewModelListener it) {
            it.matchRemoved(treeMatcher, match);
          }
        };
        IterableExtensions.<IQueryResultViewModelListener>forEach(QueryResultTreeInput.this.listeners, _function);
      }
    };
    final Job<IPatternMatch> matchDeletedJob = Jobs.<IPatternMatch>newStatelessJob(CRUDActivationStateEnum.DELETED, _function_2);
    ActivationLifeCycle _default = Lifecycles.getDefault(true, true);
    final RuleSpecification<IPatternMatch> ruleSpec = Rules.<IPatternMatch>newMatcherRuleSpecification(
      ((ViatraQueryMatcher<IPatternMatch>) matcher), _default, 
      Collections.<Job<IPatternMatch>>unmodifiableSet(CollectionLiterals.<Job<IPatternMatch>>newHashSet(matchCreatedJob, matchUpdatedJob, matchDeletedJob)));
    treeMatcher.setRuleSpec(ruleSpec);
    IQuerySpecification _specification = matcher.getSpecification();
    final String fullyQualifiedName = _specification.getFullyQualifiedName();
    this.matchers.put(fullyQualifiedName, treeMatcher);
    final Procedure1<IQueryResultViewModelListener> _function_3 = new Procedure1<IQueryResultViewModelListener>() {
      @Override
      public void apply(final IQueryResultViewModelListener it) {
        it.matcherAdded(treeMatcher);
      }
    };
    IterableExtensions.<IQueryResultViewModelListener>forEach(this.listeners, _function_3);
    this.schema.<IPatternMatch>addRule(ruleSpec);
    return treeMatcher;
  }
  
  public Object addMatcherIfLoaded(final IQuerySpecificationRegistryEntry entry) {
    Object _xifexpression = null;
    String _sourceIdentifier = entry.getSourceIdentifier();
    String _fullyQualifiedName = entry.getFullyQualifiedName();
    boolean _contains = this.loadedEntries.contains(_sourceIdentifier, _fullyQualifiedName);
    if (_contains) {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  public Object removeMatcherIfLoaded(final IQuerySpecificationRegistryEntry entry) {
    Object _xifexpression = null;
    String _sourceIdentifier = entry.getSourceIdentifier();
    String _fullyQualifiedName = entry.getFullyQualifiedName();
    boolean _contains = this.loadedEntries.contains(_sourceIdentifier, _fullyQualifiedName);
    if (_contains) {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  public QueryResultTreeMatcher removeMatcher(final IQuerySpecificationRegistryEntry entry) {
    String _fullyQualifiedName = entry.getFullyQualifiedName();
    final QueryResultTreeMatcher treeMatcher = this.matchers.get(_fullyQualifiedName);
    boolean _and = false;
    boolean _notEquals = (!Objects.equal(treeMatcher, null));
    if (!_notEquals) {
      _and = false;
    } else {
      IQuerySpecificationRegistryEntry _entry = treeMatcher.getEntry();
      String _sourceIdentifier = _entry.getSourceIdentifier();
      String _sourceIdentifier_1 = entry.getSourceIdentifier();
      boolean _equals = Objects.equal(_sourceIdentifier, _sourceIdentifier_1);
      _and = _equals;
    }
    if (_and) {
      return this.removeMatcher(treeMatcher);
    }
    return null;
  }
  
  public QueryResultTreeMatcher removeMatcher(final QueryResultTreeMatcher matcher) {
    IQuerySpecificationRegistryEntry _entry = matcher.getEntry();
    String _fullyQualifiedName = _entry.getFullyQualifiedName();
    this.matchers.remove(_fullyQualifiedName);
    final Procedure1<IQueryResultViewModelListener> _function = new Procedure1<IQueryResultViewModelListener>() {
      @Override
      public void apply(final IQueryResultViewModelListener it) {
        it.matcherRemoved(matcher);
      }
    };
    IterableExtensions.<IQueryResultViewModelListener>forEach(this.listeners, _function);
    RuleSpecification _ruleSpec = matcher.getRuleSpec();
    boolean _notEquals = (!Objects.equal(_ruleSpec, null));
    if (_notEquals) {
      RuleSpecification _ruleSpec_1 = matcher.getRuleSpec();
      this.schema.<Object>removeRule(_ruleSpec_1);
    }
    ViatraQueryMatcher _matcher = matcher.getMatcher();
    boolean _notEquals_1 = (!Objects.equal(_matcher, null));
    if (_notEquals_1) {
      ViatraQueryMatcher _matcher_1 = matcher.getMatcher();
      IQuerySpecification _specification = _matcher_1.getSpecification();
      this.builder.forgetSpecificationTransitively(_specification);
    } else {
      IQuerySpecificationRegistryEntry _entry_1 = matcher.getEntry();
      String _fullyQualifiedName_1 = _entry_1.getFullyQualifiedName();
      final IQuerySpecification<?> spec = this.builder.getSpecification(_fullyQualifiedName_1);
      boolean _notEquals_2 = (!Objects.equal(spec, null));
      if (_notEquals_2) {
        this.builder.forgetSpecificationTransitively(spec);
      }
    }
    return matcher;
  }
  
  public void loadQueries(final Iterable<IQuerySpecificationRegistryEntry> entries) {
    if (this.readOnlyEngine) {
      throw new UnsupportedOperationException("Cannot load queries to read-only engine");
    }
    SpecificationBuilder _specificationBuilder = new SpecificationBuilder();
    this.builder = _specificationBuilder;
    final Procedure1<IQuerySpecificationRegistryEntry> _function = new Procedure1<IQuerySpecificationRegistryEntry>() {
      @Override
      public void apply(final IQuerySpecificationRegistryEntry entry) {
        String _fullyQualifiedName = entry.getFullyQualifiedName();
        boolean _containsKey = QueryResultTreeInput.this.matchers.containsKey(_fullyQualifiedName);
        if (_containsKey) {
          final QueryResultTreeMatcher removedTreeMatcher = QueryResultTreeInput.this.removeMatcher(entry);
        }
        QueryResultTreeInput.this.loadQuery(entry);
        String _sourceIdentifier = entry.getSourceIdentifier();
        String _fullyQualifiedName_1 = entry.getFullyQualifiedName();
        QueryResultTreeInput.this.loadedEntries.put(_sourceIdentifier, _fullyQualifiedName_1, entry);
      }
    };
    IterableExtensions.<IQuerySpecificationRegistryEntry>forEach(entries, _function);
    this.schema.startUnscheduledExecution();
  }
  
  private QueryResultTreeMatcher loadQuery(final IQuerySpecificationRegistryEntry entry) {
    QueryResultTreeMatcher _xblockexpression = null;
    {
      final String entryFQN = entry.getFullyQualifiedName();
      boolean _containsKey = this.matchers.containsKey(entryFQN);
      if (_containsKey) {
        this.removeMatcher(entry);
      }
      QueryResultTreeMatcher _xtrycatchfinallyexpression = null;
      try {
        QueryResultTreeMatcher _xblockexpression_1 = null;
        {
          IQuerySpecificationProvider _provider = entry.getProvider();
          IQuerySpecification<?> _specificationOfProvider = this.getSpecificationOfProvider(_provider);
          final IQuerySpecification specification = ((IQuerySpecification) _specificationOfProvider);
          QueryResultTreeMatcher _xifexpression = null;
          PQuery _internalQueryRepresentation = specification.getInternalQueryRepresentation();
          PQuery.PQueryStatus _status = _internalQueryRepresentation.getStatus();
          boolean _equals = Objects.equal(_status, PQuery.PQueryStatus.ERROR);
          if (_equals) {
            IllegalArgumentException _illegalArgumentException = new IllegalArgumentException("Query definition contains errors");
            _xifexpression = this.addErroneousMatcher(entry, _illegalArgumentException);
          } else {
            final ViatraQueryMatcher<? extends IPatternMatch> matcher = this.engine.<ViatraQueryMatcher<? extends IPatternMatch>>getMatcher(specification, this.hint);
            final String specificationFQN = specification.getFullyQualifiedName();
            final QueryResultTreeMatcher treeMatcher = this.matchers.get(specificationFQN);
            boolean _notEquals = (!Objects.equal(specificationFQN, entryFQN));
            if (_notEquals) {
              this.matchers.remove(specificationFQN);
              this.matchers.put(entryFQN, treeMatcher);
            }
            treeMatcher.setEntry(entry);
            treeMatcher.setHint(this.hint);
            String _sourceIdentifier = entry.getSourceIdentifier();
            this.knownErrorEntries.remove(_sourceIdentifier, entryFQN);
            return treeMatcher;
          }
          _xblockexpression_1 = _xifexpression;
        }
        _xtrycatchfinallyexpression = _xblockexpression_1;
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          return this.addErroneousMatcher(entry, ex);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      _xblockexpression = _xtrycatchfinallyexpression;
    }
    return _xblockexpression;
  }
  
  private QueryResultTreeMatcher addErroneousMatcher(final IQuerySpecificationRegistryEntry entry, final Exception ex) {
    final String entryFQN = entry.getFullyQualifiedName();
    final QueryResultTreeMatcher treeMatcher = new QueryResultTreeMatcher(this, null);
    treeMatcher.setException(ex);
    treeMatcher.setEntry(entry);
    this.matchers.put(entryFQN, treeMatcher);
    final Procedure1<IQueryResultViewModelListener> _function = new Procedure1<IQueryResultViewModelListener>() {
      @Override
      public void apply(final IQueryResultViewModelListener it) {
        it.matcherAdded(treeMatcher);
      }
    };
    IterableExtensions.<IQueryResultViewModelListener>forEach(this.listeners, _function);
    String _sourceIdentifier = entry.getSourceIdentifier();
    boolean _put = this.knownErrorEntries.put(_sourceIdentifier, entryFQN);
    if (_put) {
      String _message = ex.getMessage();
      final String logMessage = String.format("Query Explorer has encountered an error during evaluation of query %s: %s", entryFQN, _message);
      ViatraQueryToolingBrowserPlugin _default = ViatraQueryToolingBrowserPlugin.getDefault();
      ILog _log = _default.getLog();
      ViatraQueryGUIPlugin _default_1 = ViatraQueryGUIPlugin.getDefault();
      Bundle _bundle = _default_1.getBundle();
      String _symbolicName = _bundle.getSymbolicName();
      Status _status = new Status(
        IStatus.ERROR, _symbolicName, logMessage, ex);
      _log.log(_status);
    }
    return treeMatcher;
  }
  
  private IQuerySpecification<?> getSpecificationOfProvider(final IQuerySpecificationProvider provider) {
    if ((provider instanceof IPatternBasedSpecificationProvider)) {
      final IQuerySpecification<?> specification = ((IPatternBasedSpecificationProvider)provider).getSpecification(this.builder);
      return specification;
    } else {
      final IQuerySpecification<?> specification_1 = provider.get();
      return specification_1;
    }
  }
  
  public boolean addListener(final IQueryResultViewModelListener listener) {
    return this.listeners.add(listener);
  }
  
  public boolean removeListener(final IQueryResultViewModelListener listener) {
    return this.listeners.remove(listener);
  }
  
  public IRegistryView dispose() {
    IRegistryView _xblockexpression = null;
    {
      if (this.schema!=null) {
        this.schema.dispose();
      }
      this.schema = null;
      if (this.engine!=null) {
        this.engine.removeLifecycleListener(this.lifecycleListener);
      }
      this.engine = null;
      this.resetMatchers();
      this.listeners.clear();
      if (this.view!=null) {
        this.view.removeViewListener(this.registryListener);
      }
      _xblockexpression = this.view = null;
    }
    return _xblockexpression;
  }
  
  protected void resetInput() {
    Collection<QueryResultTreeMatcher> _values = this.matchers.values();
    final Procedure1<QueryResultTreeMatcher> _function = new Procedure1<QueryResultTreeMatcher>() {
      @Override
      public void apply(final QueryResultTreeMatcher it) {
        RuleSpecification _ruleSpec = it.getRuleSpec();
        boolean _notEquals = (!Objects.equal(_ruleSpec, null));
        if (_notEquals) {
          RuleSpecification _ruleSpec_1 = it.getRuleSpec();
          QueryResultTreeInput.this.schema.<Object>removeRule(_ruleSpec_1);
        }
      }
    };
    IterableExtensions.<QueryResultTreeMatcher>forEach(_values, _function);
    this.resetMatchers();
    if ((!this.readOnlyEngine)) {
      this.engine.removeLifecycleListener(this.lifecycleListener);
      this.engine.wipe();
      this.engine.addLifecycleListener(this.lifecycleListener);
    }
  }
  
  protected void resetMatchers() {
    Collection<QueryResultTreeMatcher> _values = this.matchers.values();
    final Procedure1<QueryResultTreeMatcher> _function = new Procedure1<QueryResultTreeMatcher>() {
      @Override
      public void apply(final QueryResultTreeMatcher matcher) {
        final Procedure1<IQueryResultViewModelListener> _function = new Procedure1<IQueryResultViewModelListener>() {
          @Override
          public void apply(final IQueryResultViewModelListener it) {
            it.matcherRemoved(matcher);
          }
        };
        IterableExtensions.<IQueryResultViewModelListener>forEach(QueryResultTreeInput.this.listeners, _function);
      }
    };
    IterableExtensions.<QueryResultTreeMatcher>forEach(_values, _function);
    SpecificationBuilder _specificationBuilder = new SpecificationBuilder();
    this.builder = _specificationBuilder;
    this.matchers.clear();
    this.loadedEntries.clear();
  }
  
  public BaseIndexOptions getBaseIndexOptions() {
    QueryScope _scope = this.engine.getScope();
    final EMFScope emfScope = ((EMFScope) _scope);
    return emfScope.getOptions();
  }
  
  @Override
  public Iterable<IFilteredMatcherContent> getFilteredMatchers() {
    Collection<QueryResultTreeMatcher> _values = this.matchers.values();
    return Iterables.<IFilteredMatcherContent>filter(_values, IFilteredMatcherContent.class);
  }
  
  public void matcherFilterUpdated(final QueryResultTreeMatcher matcher) {
    final Procedure1<IQueryResultViewModelListener> _function = new Procedure1<IQueryResultViewModelListener>() {
      @Override
      public void apply(final IQueryResultViewModelListener it) {
        it.matcherFilterUpdated(matcher);
      }
    };
    IterableExtensions.<IQueryResultViewModelListener>forEach(this.listeners, _function);
  }
  
  @Pure
  public AdvancedViatraQueryEngine getEngine() {
    return this.engine;
  }
  
  @Pure
  public boolean isReadOnlyEngine() {
    return this.readOnlyEngine;
  }
  
  @Pure
  public Map<String, QueryResultTreeMatcher> getMatchers() {
    return this.matchers;
  }
  
  @Pure
  public IModelConnector getModelConnector() {
    return this.modelConnector;
  }
  
  protected void setModelConnector(final IModelConnector modelConnector) {
    this.modelConnector = modelConnector;
  }
  
  @Pure
  public QueryEvaluationHint getHint() {
    return this.hint;
  }
}
