/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd, node*/
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) { //$NON-NLS-0$
        define('orion/Deferred',factory);
    } else if (typeof exports === "object") { //$NON-NLS-0$
        module.exports = factory();
    } else {
        root.orion = root.orion || {};
        root.orion.Deferred = factory();
    }
}(this, function() {
    var queue = [],
        running = false;

    function run() {
        var fn;
        while ((fn = queue.shift())) {
            fn();
        }
        running = false;
    }

	var runAsync = (function() {
		if (typeof process !== "undefined" && typeof process.nextTick === "function") {
			var nextTick = process.nextTick;
    		return function() {
    			nextTick(run);
    		};
		} else if (typeof MutationObserver === "function") {
			var div = document.createElement("div");
			var observer = new MutationObserver(run);
			observer.observe(div, {
            	attributes: true
        	});
        	return function() {
        		div.setAttribute("class", "_tick");
        	};
		}
		return function() {
			setTimeout(run, 0);
		};
	})();

    function enqueue(fn) {
        queue.push(fn);
        if (!running) {
            running = true;
            runAsync();
        }
    }

    function noReturn(fn) {
        return function(result) {
            fn(result);
        };
    }
    
    function settleDeferred(fn, result, deferred) {
    	try {
    		var listenerResult = fn(result);
    		var listenerThen = listenerResult && (typeof listenerResult === "object" || typeof listenerResult === "function") && listenerResult.then;
    		if (typeof listenerThen === "function") {
    			if (listenerResult === deferred.promise) {
    				deferred.reject(new TypeError());
    			} else {
    				var listenerResultCancel = listenerResult.cancel;
    				if (typeof listenerResultCancel === "function") {
    					deferred._parentCancel = listenerResultCancel.bind(listenerResult);
    				} else {
    					delete deferred._parentCancel;
    				}
    				listenerThen.call(listenerResult, noReturn(deferred.resolve), noReturn(deferred.reject), noReturn(deferred.progress));
    			}
    		} else {
    			deferred.resolve(listenerResult);
    		}
    	} catch (e) {
    		deferred.reject(e);
    	}
    }


    /**
     * @name orion.Promise
     * @class Interface representing an eventual value.
     * @description Promise is an interface that represents an eventual value returned from the single completion of an operation.
     *
     * <p>For a concrete class that implements Promise and provides additional API, see {@link orion.Deferred}.</p>
     * @see orion.Deferred
     * @see orion.Deferred#promise
     */
    /**
     * @name then
     * @function
     * @memberOf orion.Promise.prototype
     * @description Adds handlers to be called on fulfillment or progress of this promise.
     * @param {Function} [onResolve] Called when this promise is resolved.
     * @param {Function} [onReject] Called when this promise is rejected.
     * @param {Function} [onProgress] May be called to report progress events on this promise.
     * @returns {orion.Promise} A new promise that is fulfilled when the given <code>onResolve</code> or <code>onReject</code>
     * callback is finished. The callback's return value gives the fulfillment value of the returned promise.
     */
    /**
     * Cancels this promise.
     * @name cancel
     * @function
     * @memberOf orion.Promise.prototype
     * @param {Object} reason The reason for canceling this promise.
     * @param {Boolean} [strict]
     */

    /**
     * @name orion.Deferred
     * @borrows orion.Promise#then as #then
     * @borrows orion.Promise#cancel as #cancel
     * @class Provides abstraction over asynchronous operations.
     * @description Deferred provides abstraction over asynchronous operations.
     *
     * <p>Because Deferred implements the {@link orion.Promise} interface, a Deferred may be used anywhere a Promise is called for.
     * However, in most such cases it is recommended to use the Deferred's {@link #promise} field instead, which exposes a 
     * simplified, minimally <a href="https://github.com/promises-aplus/promises-spec">Promises/A+</a>-compliant interface to callers.</p>
     */
    function Deferred() {
        var result, state, listeners = [],
            _this = this;

        function notify() {
            var listener;
            while ((listener = listeners.shift())) {
                var deferred = listener.deferred;
                var methodName = state === "fulfilled" ? "resolve" : "reject"; //$NON-NLS-0$ //$NON-NLS-1$ //$NON-NLS-2$
                var fn = listener[methodName];
                if (typeof fn === "function") { //$NON-NLS-0$
                	settleDeferred(fn, result, deferred);
                } else {
                    deferred[methodName](result);
                }
            }
        }

        function _reject(error) {
            delete _this._parentCancel;
            state = "rejected";
            result = error;
            if (listeners.length) {
                enqueue(notify);
            }
        }

        function _resolve(value) {
            function once(fn) {
                return function(result) {
                    if (!state || state === "assumed") {
                          fn(result);
                    }
                };
            }
            delete _this._parentCancel;
            try {
                var valueThen = value && (typeof value === "object" || typeof value === "function") && value.then;
                if (typeof valueThen === "function") {
                    if (value === _this) {
                        _reject(new TypeError());
                    } else {
                        state = "assumed";
                        var valueCancel = value && value.cancel;
                        if (typeof valueCancel !== "function") {
                            var deferred = new Deferred();
                            value = deferred.promise;
                            try {
                                valueThen(deferred.resolve, deferred.reject, deferred.progress);
                            } catch (thenError) {
                                deferred.reject(thenError);
                            }
                            valueCancel = value.cancel;
                            valueThen = value.then;
                        }
                        result = value;
                        valueThen.call(value, once(_resolve), once(_reject));
                        _this._parentCancel = valueCancel.bind(value);
                    }
                } else {
                    state = "fulfilled";
                    result = value;
                    if (listeners.length) {
                        enqueue(notify);
                    }
                }
            } catch (error) {
                once(_reject)(error);
            }
        }

        function cancel() {
            var parentCancel = _this._parentCancel;
            if (parentCancel) {
                delete _this._parentCancel;
                parentCancel();
            } else if (!state) {
                var cancelError = new Error("Cancel");
                cancelError.name = "Cancel";
                _reject(cancelError);
            }
        }


        /**
         * Resolves this Deferred.
         * @name resolve
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} value
         * @returns {orion.Promise}
         */
        this.resolve = function(value) {
            if (!state) {
                _resolve(value);
            }
            return _this;
        };

        /**
         * Rejects this Deferred.
         * @name reject
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} error
         * @param {Boolean} [strict]
         * @returns {orion.Promise}
         */
        this.reject = function(error) {
            if (!state) {
                _reject(error);
            }
            return _this;
        };

        /**
         * Notifies listeners of progress on this Deferred.
         * @name progress
         * @function
         * @memberOf orion.Deferred.prototype
         * @param {Object} update The progress update.
         * @returns {orion.Promise}
         */
        this.progress = function(update) {
            if (!state) {
                listeners.forEach(function(listener) {
                    if (listener.progress) {
                        try {
                            listener.progress(update);
                        } catch (ignore) {
                            // ignore
                        }
                    }
                });
            }
            return _this.promise;
        };

        this.cancel = function() {
            if (_this._parentCancel) {
                setTimeout(cancel, 0);
            } else {
                cancel();
            }
            return _this;
        };

        // Note: "then" ALWAYS returns before having onResolve or onReject called as per http://promises-aplus.github.com/promises-spec/
        this.then = function(onFulfill, onReject, onProgress) {
        	var deferred = new Deferred();
            deferred._parentCancel = _this.promise.cancel;
            listeners.push({
                resolve: onFulfill,
                reject: onReject,
                progress: onProgress,
                deferred: deferred
            });
            if (state === "fulfilled" || state === "rejected") {
                enqueue(notify);
            }
            return deferred.promise;
        };

        /**
         * The promise exposed by this Deferred.
         * @name promise
         * @field
         * @memberOf orion.Deferred.prototype
         * @type orion.Promise
         */
        this.promise = {
            then: _this.then,
            cancel: _this.cancel
        };
    }

    /**
     * Returns a promise that represents the outcome of all the input promises.
     * <p>When <code>all</code> is called with a single parameter, the returned promise has <dfn>eager</dfn> semantics,
     * meaning that if any input promise rejects, the returned promise immediately rejects, without waiting for the rest of the
     * input promises to fulfill.</p>
     *
     * To obtain <dfn>lazy</dfn> semantics (meaning the returned promise waits for every input promise to fulfill), pass the
     * optional parameter <code>optOnError</code>.
     * @name all
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {orion.Promise[]} promises The input promises.
     * @param {Function} [optOnError] Handles a rejected input promise. <code>optOnError</code> is invoked for every rejected
     * input promise, and is passed the reason the input promise was rejected. <p><code>optOnError</code> can return a value, which
     * allows it to act as a transformer: the return value serves as the final fulfillment value of the rejected promise in the 
     * results array generated by <code>all</code>.
     * @returns {orion.Promise} A new promise. The returned promise is generally fulfilled to an <code>Array</code> whose elements
     * give the fulfillment values of the input promises. <p>However, if an input promise rejects and eager semantics is used, the 
     * returned promise will instead be fulfilled to a single error value.</p>
     */
    Deferred.all = function(promises, optOnError) {
        var count = promises.length,
            result = [],
            rejected = false,
            deferred = new Deferred();

        deferred.then(undefined, function() {
            rejected = true;
            promises.forEach(function(promise) {
                if (promise.cancel) {
                    promise.cancel();
                }
            });
        });

        function onResolve(i, value) {
            if (!rejected) {
                result[i] = value;
                if (--count === 0) {
                    deferred.resolve(result);
                }
            }
        }

        function onReject(i, error) {
            if (!rejected) {
                if (optOnError) {
                    try {
                        onResolve(i, optOnError(error));
                        return;
                    } catch (e) {
                        error = e;
                    }
                }
                deferred.reject(error);
            }
        }

        if (count === 0) {
            deferred.resolve(result);
        } else {
            promises.forEach(function(promise, i) {
                promise.then(onResolve.bind(undefined, i), onReject.bind(undefined, i));
            });
        }
        return deferred.promise;
    };

    /**
     * Applies callbacks to a promise or to a regular object.
     * @name when
     * @function
     * @memberOf orion.Deferred
     * @static
     * @param {Object|orion.Promise} value Either a {@link orion.Promise}, or a normal value.
     * @param {Function} onResolve Called when the <code>value</code> promise is resolved. If <code>value</code> is not a promise,
     * this function is called immediately.
     * @param {Function} onReject Called when the <code>value</code> promise is rejected. If <code>value</code> is not a promise, 
     * this function is never called.
     * @param {Function} onProgress Called when the <code>value</code> promise provides a progress update. If <code>value</code> is
     * not a promise, this function is never called.
     * @returns {orion.Promise} A new promise.
     */
    Deferred.when = function(value, onResolve, onReject, onProgress) {
        var promise, deferred;
        if (value && typeof value.then === "function") { //$NON-NLS-0$
            promise = value;
        } else {
            deferred = new Deferred();
            deferred.resolve(value);
            promise = deferred.promise;
        }
        return promise.then(onResolve, onReject, onProgress);
    };

    return Deferred;
}));
/*******************************************************************************
 * @license
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
define('orion/EventTarget',[],function() {
	/**
	 * Creates an Event Target
	 *
	 * @name orion.EventTarget
	 * @class Base for creating an Orion event target
	 */
	function EventTarget() {
		this._namedListeners = {};
	}

	EventTarget.prototype = /** @lends orion.EventTarget.prototype */
	{
		/**
		 * Dispatches a named event along with an arbitrary set of arguments. Any arguments after <code>eventName</code>
		 * will be passed to the event listener(s).
		 * @param {Object} event The event to dispatch. The event object MUST have a type field
		 * @returns {boolean} false if the event has been canceled and any associated default action should not be performed
		 * listeners (if any) have resolved.
		 */
		dispatchEvent: function(event) {
			if (!event.type) {
				throw new Error("unspecified type");
			}
			var listeners = this._namedListeners[event.type];
			if (listeners) {
				listeners.forEach(function(listener) {
					try {
						if (typeof listener === "function") {
							listener(event);
						} else {
							listener.handleEvent(event);
						}
					} catch (e) {
						if (typeof console !== 'undefined') {
							console.log(e); // for now, probably should dispatch an ("error", e)
						}
					}			
				});
			}
			return !event.defaultPrevented;
		},

		/**
		 * Adds an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		addEventListener: function(eventName, listener) {
			if (typeof listener === "function" || listener.handleEvent) {
				this._namedListeners[eventName] = this._namedListeners[eventName] || [];
				this._namedListeners[eventName].push(listener);
			}
		},

		/**
		 * Removes an event listener for a named event
		 * @param {String} eventName The event name
		 * @param {Function} listener The function called when an event occurs
		 */
		removeEventListener: function(eventName, listener) {
			var listeners = this._namedListeners[eventName];
			if (listeners) {
				for (var i = 0; i < listeners.length; i++) {
					if (listeners[i] === listener) {
						if (listeners.length === 1) {
							delete this._namedListeners[eventName];
						} else {
							listeners.splice(i, 1);
						}
						break;
					}
				}
			}
		}
	};
	EventTarget.prototype.constructor = EventTarget;
	
	EventTarget.attach = function(obj) {
		var eventTarget = new EventTarget();
		obj.dispatchEvent = eventTarget.dispatchEvent.bind(eventTarget);
		obj.addEventListener = eventTarget.addEventListener.bind(eventTarget);
		obj.removeEventListener = eventTarget.removeEventListener.bind(eventTarget);
	};
	
	return EventTarget;
});
/*******************************************************************************
 * @license
 * Copyright (c) 2011, 2016 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd, node*/
/* eslint-disable missing-nls */
(function(root, factory) { // UMD
    if (typeof define === "function" && define.amd) {
        define('orion/plugin',["orion/Deferred", "orion/EventTarget"], factory);
    } else if (typeof exports === "object") {
        module.exports = factory(require("orion/Deferred"), require("orion/EventTarget"));
    } else {
        root.orion = root.orion || {};
        root.orion.PluginProvider = factory(root.orion.Deferred, root.orion.EventTarget);
    }
}(this, function(Deferred, EventTarget) {

    function _equal(obj1, obj2) {
        var keys1 = Object.keys(obj1);
        var keys2 = Object.keys(obj2);
        if (keys1.length !== keys2.length) {
            return false;
        }
        keys1.sort();
        keys2.sort();
        for (var i = 0, len = keys1.length; i < len; i++) {
            var key = keys1[i];
            if (key !== keys2[i]) {
                return false;
            }
            var value1 = obj1[key],
                value2 = obj2[key];
            if (value1 === value2) {
                continue;
            }
            if (JSON.stringify(value1) !== JSON.stringify(value2)) {
                return false;
            }
        }
        return true;
    }

    function ObjectReference(objectId, methods) {
        this.__objectId = objectId;
        this.__methods = methods;
    }
    
    function PluginProvider(headers, serviceRegistry) {
        var _headers = headers;
        var _connected = false;

        var _currentMessageId = 0;
        var _currentObjectId = 0;
        var _currentServiceId = 0;

        var _requestReferences = {};
        var _responseReferences = {};
        var _objectReferences = {};
        var _serviceReferences = {};
        
        var _services;
        var _remoteServices = {};
        var _registry = serviceRegistry;
        var _connectCallback;
        
        var _ports = [];
        var _shared = false;
        
        var _target = null;
        if (typeof(window) === "undefined") {
            if (self.postMessage) {
                _target = self;
            } else {
                _shared = true;
            }
        } else if (window !== window.parent) {
            _target = window.parent;
        } else if (window.opener !== null) {
            _target = window.opener;
        }        

        function _publish(message, target) {
            target = target || _target;
            if (target) {
                if (typeof(ArrayBuffer) === "undefined") {
                    message = JSON.stringify(message);
                }
                if (target === self || _shared) {
                    target.postMessage(message);
                } else {
                    target.postMessage(message, "*");
                }
            }
        }
        var _notify = _publish;
        var _errHandler = function(evt){
        	_publish({method: "error", error: _serializeError(evt.error)});
        };
        addEventListener("error", _errHandler);
        
        var lastHeartbeat;
        var startTime = Date.now();
        function log(state) {
            if (typeof(localStorage) !== "undefined" && localStorage.pluginLogging) {
            	console.log(state + "(" + (Date.now() - startTime) + "ms)=" + self.location);
        	}
        }
        function heartbeat() {
            var time = Date.now();
            // This timeout depends on the handshake timeout of the plugin registry. Update both accordingly.
            if (lastHeartbeat  && time - lastHeartbeat < 4000) return;
            lastHeartbeat = time;
            _publish({
                method: "loading"
            });
            log("heartbeat");
        }
        heartbeat();

        if (_shared) {
            self.addEventListener("connect", function(evt) {
                var port = evt.ports[0];
                _ports.push(port);
                if (_connected) {
                    var message = {
                        method: "plugin",
                        params: [_getPluginData()]
                    };
                    _publish(message, port);
                } else {
                    heartbeat();
                }
                port.addEventListener("message",  function(evt) {
                	_handleMessage(evt, port);
                });
                port.start();
            });
        }

        function _getPluginData() {
            var services = [];
            // we filter out the service implementation from the data
            Object.keys(_serviceReferences).forEach(function(serviceId) {
                var serviceReference = _serviceReferences[serviceId];
                services.push({
                    serviceId: serviceId,
                    names: serviceReference.names,
                    methods: serviceReference.methods,
                    properties: serviceReference.properties
                });
            });
            return {
            	updateRegistry: !!_registry,
                headers: _headers || {},
                services: services
            };
        }

        function _jsonXMLHttpRequestReplacer(name, value) {
            if (value && value instanceof XMLHttpRequest) {
                var status, statusText;
                try {
                    status = value.status;
                    statusText = value.statusText;
                } catch (e) {
                    // https://bugs.webkit.org/show_bug.cgi?id=45994
                    status = 0;
                    statusText = ""; //$NON-NLS-0
                }
                return {
                    status: status || 0,
                    statusText: statusText
                };
            }
            return value;
        }

        function _serializeError(error) {
            var result = error ? JSON.parse(JSON.stringify(error, _jsonXMLHttpRequestReplacer)) : error; // sanitizing Error object
            if (error instanceof Error) {
                result.__isError = true;
                result.message = result.message || error.message;
                result.name = result.name || error.name;
            }
            return result;
        }

        function _request(message, target) {
            target = target || _target;
            if (!target) {
                return new Deferred().reject(new Error("plugin not connected"));
            }

            message.id = String(_currentMessageId++);
            var d = new Deferred();
            _responseReferences[message.id] = d;
            d.then(null, function(error) {
                if (_connected && error instanceof Error && error.name === "Cancel") {
                    _notify({
                        requestId: message.id,
                        method: "cancel",
                        params: error.message ? [error.message] : []
                    }, target);
                }
            });

            var toStr = Object.prototype.toString;
            message.params.forEach(function(param, i) {
                if (toStr.call(param) === "[object Object]" && !(param instanceof ObjectReference)) {
                    var candidate, methods;
                    for (candidate in param) {
                        if (toStr.call(param[candidate]) === "[object Function]") {
                            methods = methods || [];
                            methods.push(candidate);
                        }
                    }
                    if (methods) {
                        var objectId = _currentObjectId++;
                        _objectReferences[objectId] = param;
                        var removeReference = function() {
                            delete _objectReferences[objectId];
                        };
                        d.then(removeReference, removeReference);
                        message.params[i] = new ObjectReference(objectId, methods);
                    }
                }
            });
            _notify(message, target);
            return d.promise;
        }

        function _throwError(messageId, error, target) {
            if (messageId || messageId === 0) {
                _notify({
                    id: messageId,
                    result: null,
                    error: error
                }, target);
            } else {
                console.log(error);
            }
        }

        function _callMethod(messageId, implementation, method, params, target) {
            params.forEach(function(param, i) {
                if (param && typeof param.__objectId !== "undefined") {
                    var obj = {};
                    param.__methods.forEach(function(method) {
                        obj[method] = function() {
                            return _request({
                                objectId: param.__objectId,
                                method: method,
                                params: Array.prototype.slice.call(arguments)
                            }, target);
                        };
                    });
                    params[i] = obj;
                }
            });
            var response = typeof messageId === "undefined" ? null : {
                id: messageId,
                result: null,
                error: null
            };
            try {
                var promiseOrResult = method.apply(implementation, params);
                if (!response) {
                    return;
                }

                if (promiseOrResult && typeof promiseOrResult.then === "function") {
                    _requestReferences[messageId] = promiseOrResult;
                    promiseOrResult.then(function(result) {
                        delete _requestReferences[messageId];
                        response.result = result;
                        _notify(response, target);
                    }, function(error) {
                        if (_requestReferences[messageId]) {
                            delete _requestReferences[messageId];
                            response.error = _serializeError(error);
                            _notify(response, target);
                        }
                    }, function() {
                        _notify({
                            responseId: messageId,
                            method: "progress",
                            params: Array.prototype.slice.call(arguments)
                        }, target);
                    });
                } else {
                    response.result = promiseOrResult;
                    _notify(response, target);
                }
            } catch (error) {
                if (response) {
                    response.error = _serializeError(error);
                    _notify(response, target);
                }
            }
        }

        function _handleMessage(evnt, target) {
            if (!_shared && evnt.source !== _target && typeof window !== "undefined") {
                return;
            }
            var data = evnt.data;
            var message = (typeof data !== "string" ? data : JSON.parse(data));
            try {
                if (message.method) { // request
                    var method = message.method,
                        params = message.params || [];
                    if ("serviceId" in message) {
                        var service = _serviceReferences[message.serviceId];
                        if (!service) {
                            _throwError(message.id, "service not found", target);
                        } else {
	                        service = service.implementation;
	                        if (method in service) {
	                            _callMethod(message.id, service, service[method], params, target);
	                        } else {
	                            _throwError(message.id, "method not found", target);
	                        }
                    	}
                    } else if ("objectId" in message) {
                        var object = _objectReferences[message.objectId];
                        if (!object) {
                            _throwError(message.id, "object not found", target);
                        } else if (method in object) {
                            _callMethod(message.id, object, object[method], params, target);
                        } else {
                            _throwError(message.id, "method not found", target);
                        }
                    } else if ("requestId" in message) {
                        var request = _requestReferences[message.requestId];
                        if (request && method === "cancel" && request.cancel) {
                            request.cancel.apply(request, params);
                        }
                    } else if ("responseId" in message) {
                        var response = _responseReferences[message.responseId];
                        if (response && method === "progress" && response.progress) {
                            response.progress.apply(response, params);
                        }
                    } else {
                        if ("plugin" === message.method) { //$NON-NLS-0$
                            var manifest = message.params[0];
                            _update({
                                services: manifest.services
                            });
                        } else {
                            throw new Error("Bad method: " + message.method);
                        }
                    }
                } else if (message.id) {
                    var deferred = _responseReferences[String(message.id)];
                    if (deferred) {
	                    delete _responseReferences[String(message.id)];
	                    if (message.error) {
	                        deferred.reject(message.error);
	                    } else {
	                        deferred.resolve(message.result);
	                    }
                    }
                }
            } catch (e) {
                console.log("Plugin._messageHandler " + e);
            }
        }        
        
        function _createServiceProxy(service) {
            var serviceProxy = {};
            if (service.methods) {
                service.methods.forEach(function(method) {
                    serviceProxy[method] = function() {
                        var message = {
                            serviceId: service.serviceId,
                            method: method,
                            params: Array.prototype.slice.call(arguments)
                        };
                        return _request(message);
                    };
                });

                if (serviceProxy.addEventListener && serviceProxy.removeEventListener && EventTarget) {
                    var eventTarget = new EventTarget();
                    var objectId = _currentObjectId++;
                    _objectReferences[objectId] = {
                        handleEvent: eventTarget.dispatchEvent.bind(eventTarget)
                    };
                    var listenerReference = new ObjectReference(objectId, ["handleEvent"]);

                    var _addEventListener = serviceProxy.addEventListener;
                    serviceProxy.addEventListener = function(type, listener) {
                        if (!eventTarget._namedListeners[type]) {
                            _addEventListener(type, listenerReference);
                        }
                        eventTarget.addEventListener(type, listener);
                    };
                    var _removeEventListener = serviceProxy.removeEventListener;
                    serviceProxy.removeEventListener = function(type, listener) {
                        eventTarget.removeEventListener(type, listener);
                        if (!eventTarget._namedListeners[type]) {
                            _removeEventListener(type, listenerReference);
                        }
                    };
                }
            }
            return serviceProxy;
        }

        function _createServiceProperties(service) {
            var properties = JSON.parse(JSON.stringify(service.properties));
            var objectClass = service.names || service.type || [];
            if (!Array.isArray(objectClass)) {
                objectClass = [objectClass];
            }
            properties.objectClass = objectClass;
            return properties;
        }

        function _registerService(service) {
        	if (!_registry) return;
            var serviceProxy = _createServiceProxy(service);
            var properties = _createServiceProperties(service);
            var registration = _registry.registerService(service.names || service.type, serviceProxy, properties);
            _remoteServices[service.serviceId] = {
                registration: registration,
                proxy: serviceProxy
            };
        }

        function _update(input) {
            var oldServices = _services || [];
            _services = input.services || [];

            if (!_equal(_services, oldServices)) {
	            var serviceIds = [];
				_services.forEach(function(service) {
					var serviceId = service.serviceId;
	                serviceIds.push(serviceId);
	                var remoteService = _remoteServices[serviceId];
	                if (remoteService) {
	                    if (_equal(service.methods, Object.keys(remoteService.proxy))) {
	                        var properties = _createServiceProperties(service);
	                        var reference = remoteService.registration.getReference();
	                        var currentProperties = {};
	                        reference.getPropertyKeys().forEach(function(_name) {
	                            currentProperties[_name] = reference.getProperty(_name);
	                        });
	                        if (!_equal(properties, currentProperties)) {
	                            remoteService.registration.setProperties(properties);
	                        }
	                        return;
	                    }
	                    remoteService.registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	                _registerService(service);
	            });
	            Object.keys(_remoteServices).forEach(function(serviceId) {
	                if (serviceIds.indexOf(serviceId) === -1) {
	                    _remoteServices[serviceId].registration.unregister();
	                    delete _remoteServices[serviceId];
	                }
	            });
           }
           
           if (_connectCallback) {
               _connectCallback();
               _connectCallback = null;
           }
        }

        this.updateHeaders = function(headers) {
            if (_connected) {
                throw new Error("Cannot update headers. Plugin Provider is connected");
            }
            _headers = headers;
        };

        this.registerService = function(names, implementation, properties) {
            if (_connected) {
                throw new Error("Cannot register service. Plugin Provider is connected");
            }

            if (typeof names === "string") {
                names = [names];
            } else if (!Array.isArray(names)) {
                names = [];
            }

            var method = null;
            var methods = [];
            for (method in implementation) {
                if (typeof implementation[method] === 'function') {
                    methods.push(method);
                }
            }
            _serviceReferences[_currentServiceId++] = {
                names: names,
                methods: methods,
                implementation: implementation,
                properties: properties || {},
                listeners: {}
            };
            heartbeat();
        };
        this.registerServiceProvider = this.registerService;

        this.connect = function(callback, errback) {
            if (_connected) {
                if (callback) {
                    callback();
                }
                return;
            }
            removeEventListener("error", _errHandler);
            var message = {
                method: "plugin",
                params: [_getPluginData()]
            };
            if (!_shared) {
                if (!_target) {
                    if (errback) {
                        errback("No valid plugin target");
                    }
                    return;
                }           
                addEventListener("message", _handleMessage, false);
                _publish(message);
            }
            if (typeof(window) !== "undefined") {
            	var head = document.getElementsByTagName("head")[0] || document.documentElement;
            	var title = head.getElementsByTagName("title")[0];
            	if (!title) {
	            	title = document.createElement("title");
	            	title.textContent = _headers ? _headers.name : '';
	            	head.appendChild(title);
	        	}
        	}

            _ports.forEach(function(port) {
                _publish(message, port);
            });
            _connected = true;
            if (_registry) {
            	_connectCallback = callback;
            } else {
	            if (callback) {
	                callback();
	            }
            }
        };

        this.disconnect = function() {
            if (_connected) {
                removeEventListener("message", _handleMessage);
                _ports.forEach(function(port) {
                    port.close();
                });
                _ports = null;
                _target = null;
                _connected = false;
            }
            // Note: re-connecting is not currently supported
        };            
    }
    
    return PluginProvider;
}));

/*******************************************************************************
 * @license
 * Copyright (c) 2014 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/lib/syntax", [], function() {
	return {
		id: "orion.lib",
		grammars: [{
			id: "orion.lib",
			repository: {
				brace_open: {
					match: "{",
					name: "punctuation.section.block.begin"
				},
				brace_close: {
					match: "}",
					name: "punctuation.section.block.end"
				},
				bracket_open: {
					match: "\\[",
					name: "punctuation.section.bracket.begin"
				},
				bracket_close: {
					match: "\\]",
					name: "punctuation.section.bracket.end"
				},
				parenthesis_open: {
					match: "\\(",
					name: "punctuation.section.parens.begin"
				},
				parenthesis_close: {
					match: "\\)",
					name: "punctuation.section.parens.end"
				},
				operator: {
					match: "(\\+|-|!|=|>|<|&|(\\|\\|))+",
					name: "punctuation.operator"
				},
				doc_block: {
					begin: "/\\*\\*",
					end: "\\*/",
					name: "comment.block.documentation",
					beginCaptures: {
						0: {name: "comment.block.documentation.start"}
					},
					endCaptures: {
						0: {name: "comment.block.documentation.end"}
					},
					patterns: [
						{
							match: "@(?:(?!\\*/)\\S)*",
							name: "meta.documentation.annotation"
						}, {
							match: "<[^\\s>]*>",
							name: "meta.documentation.tag"
						}, {
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				},
				number_decimal: {
					match: "\\b-?(?:\\.\\d+|\\d+\\.?\\d*)(?:[eE][+-]?\\d+)?\\b",
					name: "constant.numeric.number"
				},
				number_hex: {
					match: "\\b0[xX][0-9A-Fa-f]+\\b",
					name: "constant.numeric.hex"
				},
				string_doubleQuote: {
					match: '"(?:\\\\.|[^"])*"?',
					name: "string.quoted.double"
				},
				string_singleQuote: {
					match: "'(?:\\\\.|[^'])*'?",
					name: "string.quoted.single"
				},
				todo_comment_singleLine: {
					match: "(\\b)(TODO)(\\b)(.*)",
					name: "meta.annotation.task.todo",
					captures: {
						2: {name: "keyword.other.documentation.task"},
						4: {name: "comment.line"}
					}
				}
			}
		}, {
			id: "orion.c-like",
			repository: {
				comment_singleLine: {
					match: {match: "(//).*", literal: "//"},
					name: "comment.line.double-slash",
					captures: {
						1: {name: "comment.line.double-slash.start"}
					},
					patterns: [
						{
							include: "orion.lib#todo_comment_singleLine"
						}
					]
				},
				comment_block: {
					begin: {match: "/\\*", literal: "/*"},
					end: {match: "\\*/", literal: "*/"}, 
					name: "comment.block",
					beginCaptures: {
						0: {name: "comment.block.start"}
					},
					endCaptures: {
						0: {name: "comment.block.end"}
					},
					patterns: [
						{
							match: "(\\b)(TODO)(\\b)(((?!\\*/).)*)",
							name: "meta.annotation.task.todo",
							captures: {
								2: {name: "keyword.other.documentation.task"},
								4: {name: "comment.block"}
							}
						}
					]
				}
			}
		}],
		keywords: []
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/

/*eslint-env browser, amd*/
define("orion/editor/stylers/text_x-dockerfile/syntax", ["orion/editor/stylers/lib/syntax"], function(mLib) {
	var keywords = [
		"ADD", "ARG",
		"CMD", "COPY",
		"ENTRYPOINT", "ENV", "EXPOSE",
		"FROM",
		"HEALTHCHECK",
		"LABEL",
		"MAINTAINER",
		"ONBUILD",
		"RUN",
		"SHELL",
		"STOPSIGNAL",
		"USER",
		"VOLUME",
		"WORKDIR"
	];

	var id = "orion.dockerfile";
	var grammars = [];
	grammars.push.apply(grammars, mLib.grammars);
	grammars.push({
		id: id,
		contentTypes: ["text/x-dockerfile"],
		patterns: [
			{include: "orion.lib#string_doubleQuote"},
			{include: "#numberSignComment"},
			{
				match: "\\b-?[0-9]+(\\.[0-9]+)?\\b",
				name: "constant.numeric.dockerfile"
			}, {
				match: "(?i)^\\s*(?:" + keywords.join("|") + ")\\b",
				name: "keyword.operator.dockerfile"
			}
		],
		repository: {
			numberSignComment: {
				begin: {match: "^\\s*#", literal: "#"},
				end: {match: "$", literal: ""},
				name: "comment.line.number-sign.dockerfile",
				patterns: [
					{include: "orion.lib#todo_comment_singleLine"}
				]
			}
		}
	});

	return {
		id: id,
		grammars: grammars,
		keywords: keywords
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
/*eslint-env browser, amd*/
define('orion/objects',[], function() {
	function mixin(target/*, source..*/) {
		var hasOwnProperty = Object.prototype.hasOwnProperty;
		for (var j = 1, len = arguments.length; j < len; j++) {
			var source = arguments[j];
			for (var key in source) {
				if (hasOwnProperty.call(source, key)) {
					target[key] = source[key];
				}
			}
		}
		return target;
	}

	/**
	 * @name orion.objects
	 * @class Object-oriented helpers.
	 */
	return {
		/**
		 * Creates a shallow clone of the given <code>object</code>.
		 * @name orion.objects.clone
		 * @function
		 * @static
		 * @param {Object|Array} object The object to clone. Must be a "normal" Object or Array. Other built-ins,
		 * host objects, primitives, etc, will not work.
		 * @returns {Object|Array} A clone of <code>object</code>.
		 */
		clone: function(object) {
			if (Array.isArray(object)) {
				return Array.prototype.slice.call(object);
			}
			var clone = Object.create(Object.getPrototypeOf(object));
			mixin(clone, object);
			return clone;
		},
		/**
		 * Mixes all <code>source</code>'s own enumerable properties into <code>target</code>. Multiple source objects
		 * can be passed as varargs.
		 * @name orion.objects.mixin
		 * @function
		 * @static
		 * @param {Object} target
		 * @param {Object} source
		 */
		mixin: mixin,
		/**
		 * Wraps an object into an Array if necessary.
		 * @name orion.objects.toArray
		 * @function
		 * @static
		 * @param {Object} obj An object.
		 * @returns {Array} Returns <code>obj</code> unchanged, if <code>obj</code> is an Array. Otherwise returns a 1-element Array
		 * whose sole element is <code>obj</code>.
		 */
		toArray: function(o) {
			return Array.isArray(o) ? o : [o];
		}
	};
});
/**
 * @license RequireJS i18n 2.0.6 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like 'i18n!nls/colors'.
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is 'en-us',
 * then the plugin will ask for the 'en-us', 'en' and 'root' bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    'use strict';

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec('foo/bar/baz/nls/en-ca/foo') gives:
    //['foo/bar/baz/nls/en-ca/foo', 'foo/bar/baz/nls/', '/', '/', 'en-ca', 'foo']
    //nlsRegExp.exec('foo/bar/baz/nls/foo') gives:
    //['foo/bar/baz/nls/foo', 'foo/bar/baz/nls/', '/', '/', 'foo', '']
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};
        masterConfig = masterConfig || {};

        return {
            version: '2.0.6',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split('-'),
                    toLoad = [],
                    value = {},
                    i, part, current = '';

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === 'undefined' ? 'root' :
                            ((navigator.languages && navigator.languages[0]) ||
                             navigator.language ||
                             navigator.userLanguage || 'root').toLowerCase();
                    }
                    parts = locale.split('-');
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, 'root', toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? '-' : '') + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }

                    if(config.locales) {
                    	var j, k; 
                    	for (j = 0; j < config.locales.length; j++) {
                    		locale = config.locales[j];
                    		parts = locale.split("-");
                    		current = "";
	                    	for (k = 0; k < parts.length; k++) {
		                        part = parts[k];
		                        current += (current ? "-" : "") + part;
		                        addIfExists(req, current, toLoad, prefix, suffix);
	                    	}
                    	}
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart('root', master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? '-' : '') + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

/*******************************************************************************
 * @license
 * Copyright (c) 2017 Remy Suen and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors:
 *     Remy Suen - initial API and implementation
 ******************************************************************************/
/* eslint-env amd */
define('plugins/languages/docker/nls/messages',{
	root: true
});

/*******************************************************************************
 * @license
 * Copyright (c) 2017 Remy Suen and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     Remy Suen - initial API and implementation
 ******************************************************************************/
/* eslint-env amd */
define('plugins/languages/docker/nls/root/messages',{
	"pluginName": "Orion Dockerfile Editor",
	"pluginDescription": "This plug-in provides Dockerfile editing support for Orion, like keyword completion and syntax highlighting.",
	"dockerContentAssist": "Dockerfile Content Assist",
	"dockerContentHover": "Docker Content Hover",

	"hoverAdd": "Copy files, folders, or remote URLs from `source` to the `dest` path in the image's filesystem.\n\n",
	"hoverArg": "Define a variable with an optional default value that users can override at build-time when using `docker build`.\n\n",
	"hoverCmd": "Provide defaults for an executing container. If an executable is not specified, then `ENTRYPOINT` must be specified as well. There can only be one `CMD` instruction in a `Dockerfile`.\n\n",
	"hoverCopy": "Copy files or folders from `source` to the `dest` path in the image's filesystem.\n\n",
	"hoverEntrypoint": "Configures the container to be run as an executable.\n\n",
	"hoverEnv": "Set the environment variable `key` to the value `value`.\n\n",
	"hoverExpose": "Define the network `port`s that this container will listen on at runtime.\n\n",
	"hoverFrom": "Set the `baseImage` to use for subsequent instructions. `FROM` must be the first instruction in a `Dockerfile`.\n\n",
	"hoverHealthcheck": "Define how Docker should test the container to check that it is still working. Alternatively, disable the base image's `HEALTHCHECK` instruction. There can only be one `HEALTHCHECK` instruction in a `Dockerfile`.\n\nSince Docker 1.12\n\n",
	"hoverLabel": "Adds metadata to an image.\n\nSince Docker 1.6\n\n",
	"hoverMaintainer": "Set the _Author_ field of the generated images. This instruction has been deprecated in favor of `LABEL`.\n\n",
	"hoverOnbuild": "Add a _trigger_ instruction to the image that will be executed when the image is used as a base image for another build.\n\n",
	"hoverRun": "Execute any commands on top of the current image as a new layer and commit the results.\n\n",
	"hoverShell": "Override the default shell used for the _shell_ form of commands.\n\nSince Docker 1.12\n\n",
	"hoverStopsignal": "Set the system call signal to use to send to the container to exit. Signals can be valid unsigned numbers or a signal name in the `SIGNAME` format such as `SIGKILL`.\n\nSince Docker 1.12\n\n",
	"hoverUser": "Set the user name or UID to use when running the image in addition to any subsequent `CMD`, `ENTRYPOINT`, or `RUN` instructions that follow it in the `Dockerfile`.\n\n",
	"hoverVolume": "Create a mount point with the specifid name and mark it as holding externally mounted volumes from the native host or from other containers.\n\n",
	"hoverWorkdir": "Set the working directory for any subsequent `ADD`, `COPY`, `CMD`, `ENTRYPOINT`, or `RUN` instructions that follow it in the `Dockerfile`.\n\n",
	"hoverOnlineDocumentationFooter": "\n\n[Online documentation](${0})",

	"hoverEscape": "Sets the character to use to escape characters and newlines in this Dockerfile. If unspecified, the default escape character is `\\`.\n\n",

 	"proposalArgNameOnly": "Define a variable that users can set at build-time when using `docker build`.\n\n",
	"proposalArgDefaultValue": "Define a variable with the given default value that users can override at build-time when using `docker build`.\n\n",
	"proposalHealthcheckExec": "Define how Docker should test the container to check that it is still working. There can only be one `HEALTHCHECK` instruction in a `Dockerfile`.\n\nSince Docker 1.12\n\n",
	"proposalHealthcheckNone": "Disable the `HEALTHCHECK` instruction inherited from the base image if one exists. There can only be one `HEALTHCHECK` instruction in a `Dockerfile`.\n\nSince Docker 1.12"
});


/*******************************************************************************
 * @license
 * Copyright (c) 2012, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution 
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html). 
 *
 * Contributors: IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env browser, amd*/
/*global requirejs*/
define('orion/i18nUtil',[], function() {
	/**
	 * Performs string substitution. Can be invoked in 2 ways:
	 *
	 * i) vargs giving numbered substition values:
	 *   formatMessage("${0} is ${1}", "foo", "bar")  // "foo is bar"
	 *
	 * ii) a map giving the substitutions:
	 *   formatMessage("${thing} is ${1}", {1: "bar", thing: "foo"})  // "foo is bar"
	 */
	function formatMessage(msg) {
		var pattern = /\$\{([^\}]+)\}/g, args = arguments;
		if (args.length === 2 && args[1] && typeof args[1] === "object") {
			return msg.replace(pattern, function(str, key) {
				return args[1][key];
			});
		}
		return msg.replace(pattern, function(str, index) {
			return args[(index << 0) + 1];
		});
	}
	return {
		formatMessage: formatMessage
	};
});
/*******************************************************************************
 * @license
 * Copyright (c) 2017 Remy Suen and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     Remy Suen - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('plugins/languages/docker/dockerAssist',[
	'orion/objects',
	'i18n!plugins/languages/docker/nls/messages',
	'orion/i18nUtil'
], function(Objects, dockerMessages, i18nUtil) {

	/**
	 * @description Creates a new DockerContentAssist object
	 * @constructor
	 * @public
	 * @param {Object} keywords an array of Dockerfile keywords
	 * @param {Object} markdowns a dictionary of markdown documentation hovers
	 */
	function DockerContentAssist(keywords, markdowns) {
		this.keywords = keywords;
		this.markdowns = markdowns;
	}

	Objects.mixin(DockerContentAssist.prototype, {
		computeProposals: function (buffer, offset, context) {
			var firstCommentIdx = -1;
			var escapeCharacter = "\\";
			directiveCheck: for (var i = 0; i < buffer.length; i++) {
				switch (buffer.charAt(i)) {
					case '#':
						firstCommentIdx = i;
						// in the first comment of the file, look for directives
						var directive = "";
						var capture = false;
						escapeCheck: for (var j = i + 1; j < buffer.length; j++) {
							var char = buffer.charAt(j);
							switch (char) {
								case ' ':
								case '\t':
									// ignore whitespace if directive is well-formed or hasn't been found yet
									if (directive !== "escape" && directive !== "") {
										break escapeCheck;
									}
									continue;
								case '=':
									if (directive === "escape") {
										// '=' found and the directive that has been declared is the escape directive,
										// record its value so we know what the escape character of this Dockerfile is
										capture = true;
									} else {
										// unknown directive found, stop searching
										break escapeCheck;
									}
									break;
								default:
									if (capture) {
										// the escape directive should be a single character and followed by whitespace,
										// it should also either be a backslash or a backtick
										if ((j + 1 === buffer.length || isWhitespace(buffer.charAt(j + 1)))
												&& (char === '\\' || char === '`')) {
											escapeCharacter = char;
										}
										break escapeCheck;
									}
									directive = directive + char.toLowerCase();
									break;
							}
						}
						break directiveCheck;
					case ' ':
					case '\t':
						// ignore whitespace
						continue;
					case '\r':
					case '\n':
						break directiveCheck;
					default:
						// not a comment then not a directive
						break directiveCheck;
				}
			}

			// start from the offset and walk back
			commentCheck: for (i = offset - 1; i >= 0; i--) {
				switch (buffer.charAt(i)) {
					case '#':
						if (i === firstCommentIdx) {
							// we're in the first comment, might need to suggest
							// the escape directive as a proposal
							var directivePrefix = buffer.substring(i + 1, offset).trimLeft().toLowerCase();
							if ("escape".indexOf(directivePrefix) === 0) {
								return [ createEscape(context.prefix, offset - context.prefix.length, this.markdowns["escape"]) ];
							}
						}
						// in a comment, no proposals to suggest
						return [];
					case ' ':
					case '\t':
						// ignore whitespace
						continue;
					case '\r':
					case '\n':
						// walked back to the beginning of this line, not in a comment
						break commentCheck;
				}
			}

			// get every line in the file
			var split = buffer.trim().split("\n");
			var fromOnly = split.some(function(line) {
				var trimmed = line.trim();
				// check if it's a comment or an empty line
				return trimmed.length !== 0 && trimmed.charAt(0) !== '#';
			});
			if (!fromOnly) {
				// if we only have empty lines and comments, only suggest FROM
				return [ createFROM(context.prefix, offset, this.markdowns["FROM"]) ];
			}
			
			if (context.prefix === "") {
				context.prefix = calculateTruePrefix(buffer, offset, escapeCharacter);
			}

			var previousWord = "";
			var whitespace = false;
			var lineStart = 0;
			lineCheck: for (i = offset - 1; i >= 0; i--) {
				char = buffer.charAt(i);
				switch (char) {
					case '\n':
						if (buffer.charAt(i - 1) === escapeCharacter) {
							i--;
							continue;
						} else if (buffer.charAt(i - 1) === '\r' && buffer.charAt(i - 2) === escapeCharacter) {
							i = i - 2;
							continue;
						}

						if (previousWord !== "" && previousWord !== "ONBUILD") {
							// keyword content assist only allowed after an ONBUILD
							return [];
						}
						lineStart = i + 1;
						break lineCheck;
					case ' ':
					case '\t':
						if (whitespace) {
							if (previousWord !== "" && previousWord !== "ONBUILD") {
								// keyword content assist only allowed after an ONBUILD
								return [];
							}
						} else {
							whitespace = true;
						}
						break;
					default:
						if (whitespace) {
							previousWord = char.toUpperCase() + previousWord;
						}
						break;
				}
			}

			if (previousWord !== "" && previousWord !== "ONBUILD") {
				// only suggest proposals if at the front or after an ONBUILD
				return [];
			}

			var proposals = [];
			if (context.prefix === "") {
				createProposals(proposals, this.keywords, previousWord, "", offset, this.markdowns);
				return proposals;
			}

			var suggestions = [];
			var uppercasePrefix = context.prefix.toUpperCase();
			for (i = 0; i < this.keywords.length; i++) {
				if (this.keywords[i] === uppercasePrefix) {
					// prefix is a keyword already, nothing to suggest
					return [];
				} else if (this.keywords[i].indexOf(uppercasePrefix) === 0) {
					suggestions.push(this.keywords[i]);
				}
			}

			if (suggestions.length === 0) {
				// prefix doesn't match any keywords, nothing to suggest
				return [];
			}

			if (lineStart + context.line.indexOf(context.prefix) + context.prefix.length === offset) {
				createProposals(proposals, suggestions, previousWord, context.prefix, offset, this.markdowns);
				return proposals;
			}
			return [];
		}
	});

	function createProposals(proposals, keywords, previousWord, prefix, offset, markdowns) {
		for (var i = 0; i < keywords.length; i++) {
			switch (keywords[i]) {
				case "ARG":
					proposals.push(createARG_NameOnly(prefix, offset - prefix.length));
					proposals.push(createARG_DefaultValue(prefix, offset - prefix.length));
					break;
				case "HEALTHCHECK":
					proposals.push(createHEALTHCHECK_CMD(prefix, offset - prefix.length));
					proposals.push(createHEALTHCHECK_NONE(prefix, offset - prefix.length));
					break;
				default:
					proposals.push(createSingleProposals(keywords[i], prefix, offset, markdowns));
					break;
			}
		}

		if (previousWord === "ONBUILD") {
			// can't have FROM, MAINTAINER, or ONBUILD follow an ONBUILD
			for (i = 0; i < proposals.length; i++) {
				switch (proposals[i].name) {
					case "FROM":
					case "MAINTAINER":
					case "ONBUILD":
						proposals.splice(i, 1);
						i--;
						break;
				}
			}
		}
	}

	function isWhitespace(char) {
		return char === ' ' || char === '\t' || char === '\r' || char === '\n';
	}

	/**
	 * Walks back in the text buffer to calculate the true prefix of the
	 * current text caret offset. Orion's provided prefix does not include
	 * symbols but we do want to consider that a prefix in Dockerfiles.
	 * 
	 * @param buffer the content of the opened file
	 * @param offset the current text caret's offset
	 * @param escapeCharacter the escape character defined in this Dockerfile
	 */
	function calculateTruePrefix(buffer, offset, escapeCharacter) {
		var char = buffer.charAt(offset - 1);
		switch (char) {
			case '\n':
				var escapedPrefix = "";
				for (var i = offset - 1; i >= 0; i--) {
					if (buffer.charAt(i) === '\n') {
						if (buffer.charAt(i - 1) === escapeCharacter) {
							i--;
						} else if (buffer.charAt(i - 1) === '\r' && buffer.charAt(i - 2) === escapeCharacter) {
							i = i -2;
						} else {
							break;
						}
					} else if (buffer.charAt(i) === ' ' || buffer.charAt(i) === '\t') {
						break;
					} else {
						escapedPrefix = buffer.charAt(i).toUpperCase() + escapedPrefix;
					}
				}
				
				if (escapedPrefix !== "") {
					return escapedPrefix;
				}
				break;
			case '\r':
			case ' ':
			case '\t':
				break;
			default:
				var truePrefix = char;
				prefixCheck: for (i = offset - 2; i >= 0; i--) {
					char = buffer.charAt(i);
					switch (char) {
						case '\r':
						case '\n':
						case ' ':
						case '\t':
							break prefixCheck;
						default:
							for (i = offset - 2; i >= 0; i--) {
								truePrefix = char + truePrefix;
							}
							break;
					}
				}
				return truePrefix;
		}
		return "";
	}

	function createSingleProposals(keyword, prefix, offset, markdowns) {
		switch (keyword) {
			case "ADD":
				return createADD(prefix, offset - prefix.length, markdowns[keyword]);
			case "CMD":
				return createCMD(prefix, offset - prefix.length, markdowns[keyword]);
			case "COPY":
				return createCOPY(prefix, offset - prefix.length, markdowns[keyword]);
			case "ENTRYPOINT":
				return createENTRYPOINT(prefix, offset - prefix.length, markdowns[keyword]);
			case "ENV":
				return createENV(prefix, offset - prefix.length, markdowns[keyword]);
			case "EXPOSE":
				return createEXPOSE(prefix, offset - prefix.length, markdowns[keyword]);
			case "FROM":
				return createFROM(prefix, offset - prefix.length, markdowns[keyword]);
			case "LABEL":
				return createLABEL(prefix, offset - prefix.length, markdowns[keyword]);
			case "MAINTAINER":
				return createMAINTAINER(prefix, offset - prefix.length, markdowns[keyword]);
			case "ONBUILD":
				return createONBUILD(prefix, offset - prefix.length, markdowns[keyword]);
			case "RUN":
				return createRUN(prefix, offset - prefix.length, markdowns[keyword]);
			case "SHELL":
				return createSHELL(prefix, offset - prefix.length, markdowns[keyword]);
			case "STOPSIGNAL":
				return createSTOPSIGNAL(prefix, offset - prefix.length, markdowns[keyword]);
			case "WORKDIR":
				return createWORKDIR(prefix, offset - prefix.length, markdowns[keyword]);
			case "VOLUME":
				return createVOLUME(prefix, offset - prefix.length, markdowns[keyword]);
			case "USER":
				return createUSER(prefix, offset - prefix.length, markdowns[keyword]);
		}
	}

	function createADD(prefix, offset, markdown) {
		return {
			name: "ADD",
			description: " source dest",
			proposal: "ADD source dest",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'source'
				{
					offset: offset + 4,
					length: 6
				},
				// linked mode for 'dest'
				{
					offset: offset + 11,
					length: 4
				}
			],
			escapePosition: offset + 15,
			hover: markdown
		};
	}

	function createARG_NameOnly(prefix, offset) {
		return {
			name: "ARG",
			description: " name",
			proposal: "ARG name",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'name'
				{
					offset: offset + 4,
					length: 4
				}
			],
			escapePosition: offset + 8,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalArgNameOnly"] +
					"```\n" +
					"ARG userName\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#arg")
			}
		};
	}

	function createARG_DefaultValue(prefix, offset) {
		return {
			name: "ARG",
			description: " name=defaultValue",
			proposal: "ARG name=defaultValue",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'name'
				{
					offset: offset + 4,
					length: 4
				},
				// linked mode for 'defaultValue'
				{
					offset: offset + 9,
					length: 12
				}
			],
			escapePosition: offset + 21,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalArgDefaultValue"] +
					"```\n" +
					"ARG testOutputDir=test\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#arg")
			}
		};
	}

	function createCMD(prefix, offset, markdown) {
		return {
			name: "CMD",
			description: " [ \"executable\" ]",
			proposal: "CMD [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'executable'
				{
					offset: offset + 7,
					length: 10
				}
			],
			escapePosition: offset + 20,
			hover: markdown
		};
	}

	function createCOPY(prefix, offset, markdown) {
		return {
			name: "COPY",
			description: " source dest",
			proposal: "COPY source dest",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'source'
				{
					offset: offset + 5,
					length: 6
				},
				// linked mode for 'dest'
				{
					offset: offset + 12,
					length: 4
				}
			],
			escapePosition: offset + 16,
			hover: markdown
		};
	}

	function createENTRYPOINT(prefix, offset, markdown) {
		return {
			name: "ENTRYPOINT",
			description: " [ \"executable\" ]",
			proposal: "ENTRYPOINT [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'executable'
				{
					offset: offset + 14,
					length: 10
				}
			],
			escapePosition: offset + 27,
			hover: markdown
		};
	}

	function createENV(prefix, offset, markdown) {
		return {
			name: "ENV",
			description: " key=value",
			proposal: "ENV key=value",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'key'
				{
					offset: offset + 4,
					length: 3
				},
				// linked mode for 'value'
				{
					offset: offset + 8,
					length: 5
				}
			],
			escapePosition: offset + 13,
			hover: markdown
		};
	}

	function createEXPOSE(prefix, offset, markdown) {
		return {
			name: "EXPOSE",
			description: " port",
			proposal: "EXPOSE port",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'port'
				{
					offset: offset + 7,
					length: 4
				}
			],
			escapePosition: offset + 11,
			hover: markdown
		};
	}

	function createFROM(prefix, offset, markdown) {
		return {
			name: "FROM",
			description: " baseImage",
			proposal: "FROM baseImage",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'baseImage'
				{
					offset: offset + 5,
					length: 9
				}
			],
			escapePosition: offset + 14,
			hover: markdown
		};
	}

	function createHEALTHCHECK_CMD(prefix, offset) {
		return {
			name: "HEALTHCHECK",
			description: " --interval=30s --timeout=30s --retries=3 CMD [ \"executable\" ]",
			proposal: "HEALTHCHECK --interval=30s --timeout=30s --retries=3 CMD [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'interval'
				{
					offset: offset + 23,
					length: 3
				},
				// linked mode for 'timeout'
				{
					offset: offset + 37,
					length: 3
				},
				// linked mode for 'retries'
				{
					offset: offset + 51,
					length: 1
				},
				// linked mode for 'executable'
				{
					offset: offset + 60,
					length: 10
				}
			],
			escapePosition: offset + 73,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalHealthcheckExec"] +
				"```\n" +
				"HEALTHCHECK --interval=10m --timeout=5s \\\n" +
				"    CMD curl -f http://localhost/ || exit 1\n" +
				"```" + 
				i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#healthcheck")
			}
		};
	}

	function createHEALTHCHECK_NONE(prefix, offset) {
		return {
			name: "HEALTHCHECK",
			description: " NONE",
			proposal: "HEALTHCHECK NONE",
			prefix: prefix,
			positions: [],
			overwrite: true,
			escapePosition: offset + 16,
			hover: {
				type: "markdown",
				content: dockerMessages["proposalHealthcheckNone"] +
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#healthcheck")
			}
		};
	}

	function createLABEL(prefix, offset, markdown) {
		return {
			name: "LABEL",
			description: " key=\"value\"",
			proposal: "LABEL key=\"value\"",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'key'
				{
					offset: offset + 6,
					length: 3
				},
				// linked mode for 'value'
				{
					offset: offset + 11,
					length: 5
				}
			],
			escapePosition: offset + 17,
			hover: markdown
		};
	}

	function createMAINTAINER(prefix, offset, markdown) {
		return {
			name: "MAINTAINER",
			description: " name",
			proposal: "MAINTAINER name",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'name'
				{
					offset: offset + 11,
					length: 4
				}
			],
			escapePosition: offset + 15,
			hover: markdown
		};
	}

	function createONBUILD(prefix, offset, markdown) {
		return {
			name: "ONBUILD",
			description: " INSTRUCTION",
			proposal: "ONBUILD INSTRUCTION",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'INSTRUCTION'
				{
					offset: offset + 8,
					length: 11
				}
			],
			escapePosition: offset + 19,
			hover: markdown
		};
	}

	function createRUN(prefix, offset, markdown) {
		return {
			name: "RUN",
			description: " command",
			proposal: "RUN command",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'command'
				{
					offset: offset + 4,
					length: 7
				}
			],
			escapePosition: offset + 11,
			hover: markdown
		};
	}

	function createSHELL(prefix, offset, markdown) {
		return {
			name: "SHELL",
			description: " [ \"executable\" ]",
			proposal: "SHELL [ \"executable\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'executable'
				{
					offset: offset + 9,
					length: 10
				}
			],
			escapePosition: offset + 22,
			hover: markdown
		};
	}

	function createSTOPSIGNAL(prefix, offset, markdown) {
		return {
			name: "STOPSIGNAL",
			description: " signal",
			proposal: "STOPSIGNAL signal",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'signal'
				{
					offset: offset + 11,
					length: 6
				}
			],
			escapePosition: offset + 17,
			hover: markdown
		};
	}

	function createUSER(prefix, offset, markdown) {
		return {
			name: "USER",
			description: " daemon",
			proposal: "USER daemon",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'daemon'
				{
					offset: offset + 5,
					length: 6
				}
			],
			escapePosition: offset + 11,
			hover: markdown
		};
	}

	function createVOLUME(prefix, offset, markdown) {
		return {
			name: "VOLUME",
			description: " [ \"/data\" ]",
			proposal: "VOLUME [ \"/data\" ]",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for '/data'
				{
					offset: offset + 10,
					length: 5
				}
			],
			escapePosition: offset + 18,
			hover: markdown
		};
	}

	function createWORKDIR(prefix, offset, markdown) {
		return {
			name: "WORKDIR",
			description: " /path",
			proposal: "WORKDIR /path",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for 'path'
				{
					offset: offset + 8,
					length: 5
				}
			],
			escapePosition: offset + 13,
			hover: markdown
		};
	}

	function createEscape(prefix, offset, markdown) {
		return {
			name: "escape",
			description: "=`",
			proposal: "escape=`",
			prefix: prefix,
			overwrite: true,
			positions: [
				// linked mode for '`'
				{
					offset: offset + 7,
					length: 1
				}
			],
			escapePosition: offset + 8,
			hover: markdown
		};
	}

	return {
		DockerContentAssist : DockerContentAssist
	};
});

/*******************************************************************************
 * @license
 * Copyright (c) 2014, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0
 * (http://www.eclipse.org/legal/epl-v10.html), and the Eclipse Distribution
 * License v1.0 (http://www.eclipse.org/org/documents/edl-v10.html).
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
/*eslint-env amd*/
define('plugins/languages/docker/dockerPlugin',[
	'orion/plugin',
	'orion/editor/stylers/text_x-dockerfile/syntax',
	'plugins/languages/docker/dockerAssist',
	'i18n!plugins/languages/docker/nls/messages',
	'orion/i18nUtil'
], function(PluginProvider, mDockerfile, DockerAssist, dockerMessages, i18nUtil) {

	function connect() {
		var headers = {
			name: dockerMessages['pluginName'],
			version: "1.0",
			description: dockerMessages['pluginDescription'],
		};
		var pluginProvider = new PluginProvider(headers);
		registerServiceProviders(pluginProvider);
		pluginProvider.connect();
	}

	function registerServiceProviders(pluginProvider) {
		var markdowns = {
			ADD: {
				type: "markdown",
				content: dockerMessages["hoverAdd"] +
					"```\n" +
					"ADD hello.txt /absolute/path\n" +
					"ADD hello.txt relative/to/workdir\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#add")
			},

			ARG: {
				type: "markdown",
				content: dockerMessages["hoverArg"] +
					"```\n" +
					"ARG userName\n" +
					"ARG testOutputDir=test\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#arg")
			},

			CMD: {
				type: "markdown",
				content:  dockerMessages["hoverCmd"] +
					"`CMD [ \"/bin/ls\", \"-l\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#cmd")
			},

			COPY: {
				type: "markdown",
				content:  dockerMessages["hoverCopy"] +
					"```\n" +
					"COPY hello.txt /absolute/path\n" +
					"COPY hello.txt relative/to/workdir\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#copy")
			},

			ENTRYPOINT: {
				type: "markdown",
				content:  dockerMessages["hoverEntrypoint"] +
					"`ENTRYPOINT [ \"/opt/app/run.sh\", \"--port\", \"8080\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#entrypoint")
			},

			ENV: {
				type: "markdown",
				content:  dockerMessages["hoverEnv"] +
					"`ENV buildTag=1.0`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#env")
			},

			EXPOSE: {
				type: "markdown",
				content:  dockerMessages["hoverExpose"] +
					"```\n" +
					"EXPOSE 8080\n" +
					"EXPOSE 80 443 22\n" +
					"EXPOSE 7000-8000\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#expose")
			},

			FROM: {
				type: "markdown",
				content:  dockerMessages["hoverFrom"] +
					"```\n" +
					"FROM baseImage\n" +
					"FROM baseImage:tag\n" +
					"FROM baseImage@digest\n" + 
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#from")
			},

			HEALTHCHECK: {
				type: "markdown",
				content:  dockerMessages["hoverHealthcheck"] +
				"```\n" +
				"HEALTHCHECK --interval=10m --timeout=5s \\\n" +
				"    CMD curl -f http://localhost/ || exit 1\n" +
				"HEALTHCHECK NONE\n" +
				"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#healthcheck")
			},

			LABEL: {
				type: "markdown",
				content:  dockerMessages["hoverLabel"] +
					"`LABEL version=\"1.0\"`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#label")
			},

			MAINTAINER: {
				type: "markdown",
				content: dockerMessages["hoverMaintainer"] +
					"`MAINTAINER name`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#maintainer")
			},

			ONBUILD: {
				type: "markdown",
				content: dockerMessages["hoverOnbuild"] +
					"```\n" +
					"ONBUILD ADD . /opt/app/src/extensions\n" +
					"ONBUILD RUN /usr/local/bin/build.sh /opt/app" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#cmd")
			},

			RUN: {
				type: "markdown",
				content: dockerMessages["hoverRun"] +
					"`RUN apt-get update && apt-get install -y curl`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#run")
			},

			SHELL: {
				type: "markdown",
				content: dockerMessages["hoverShell"] +
					"`SHELL [ \"powershell\", \"-command\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#shell")
			},

			STOPSIGNAL: {
				type: "markdown",
				content: dockerMessages["hoverStopsignal"] +
					"`STOPSIGNAL 9`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#stopsignal")
			},

			USER: {
				type: "markdown",
				content: dockerMessages["hoverUser"] +
					"`USER daemon`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#user")
			},

			VOLUME: {
				type: "markdown",
				content: dockerMessages["hoverVolume"] +
					"`VOLUME [ \"/var/db\" ]`" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#volume")
			},

			WORKDIR: {
				type: "markdown",
				content: dockerMessages["hoverWorkdir"] +
					"```\n" +
					"WORKDIR /path/to/workdir\n" +
					"WORKDIR relative/path\n" +
					"```" + 
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#workdir")
			},

			escape: {
				type: "markdown",
				content: dockerMessages["hoverEscape"] +
					"```\n" + 
					"# escape=`\n" +
					"```" +
					i18nUtil.formatMessage.call(null, dockerMessages["hoverOnlineDocumentationFooter"], "https://docs.docker.com/engine/reference/builder/#escape")
			}
		};

		pluginProvider.registerServiceProvider("orion.core.contenttype", {}, {
			contentTypes: [
				{	id: "text/x-dockerfile",
					"extends": "text/plain",
					name: "dockerfile",
					extension: ["dockerfile"],
					filenamePattern: "^dockerfile"
				}
			] 
		});
		
		pluginProvider.registerServiceProvider("orion.edit.contentAssist", new DockerAssist.DockerContentAssist(mDockerfile.keywords, markdowns), {
			name: dockerMessages['dockerContentAssist'],
			contentType:  ["text/x-dockerfile" ]
		});

		pluginProvider.registerServiceProvider("orion.edit.hover", {
			computeHoverInfo: function (editorContext, context) {
				if (context.proposal && context.proposal.hover) {
					return context.proposal.hover;
				}

				var textLength;
				var content;
				return editorContext.getText()
				.then(function(text) {
					content = text;
					textLength = text.length;
					return editorContext.getLineAtOffset(context.offset);
				})
				.then(function(line) {
					return editorContext.getLineStart(line);
				}).then(function(lineStart) {
					for (var i = lineStart; i < textLength; i++) {
						if (content.charAt(i) === '#') {
							// might be hovering over a directive
							var directive = "";
							var directiveOffset = -1;
							var stop = false;
							for (var j = i + 1; j < textLength; j++) {
								if (content.charAt(j) === '=') {
									if (directiveOffset === -1) {
										// record the end offset for the directive if not already recorded
										directiveOffset = j;
									}
									break;
								} else if (content.charAt(j) === ' ' || content.charAt(j)  === '\t'
										|| content.charAt(j) === '\r' || content.charAt(j)  === '\n') {
									if (directive !== "" && !stop) {
										// directive has been captured, stop and record the ending offset
										directiveOffset = j;
										stop = true;
									}
									continue;
								}

								if (stop) {
									// a whitespace was encountered and we should stop capturing but
									// another character was found, so this is not a directive
									return null;
								} else {
									// capture the directive
									directive = directive + content.charAt(j);
								}
							}
							// check to make sure the user is hovering over the directive itself
							if (i <= context.offset && context.offset <= j) {
								return markdowns[directive.toLowerCase()];
							}
							return null;
						}
						// skip initial whitespace at the beginning of the line
						if (content.charAt(i) !== ' ' && content.charAt(i) !== '\t') {
							for (var j = i + 1; j < textLength; j++) {
								// find the end of the word
								if (content.charAt(j) === ' ' || content.charAt(j)  === '\t'
										|| content.charAt(j) === '\r' || content.charAt(j)  === '\n') {
									if (i <= context.offset && context.offset <= j) {
										return markdowns[content.substring(i, j).toUpperCase()];
									}
									return null;
								}
							}
							return markdowns[content.substring(i, j).toUpperCase()];
						}
					}
					return null;
				});
			}
		}, {
			name: dockerMessages['dockerContentHover'],
			contentType: [ "text/x-dockerfile" ]
		});

		mDockerfile.grammars.forEach(function(current) {
			pluginProvider.registerServiceProvider("orion.edit.highlighter", {}, current);
		});
	}

	return {
		connect: connect,
		registerServiceProviders: registerServiceProviders
	};
});

