/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2020, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * If you like this project or if you find it useful, you can support us at:
 * 
 * http://plantuml.com/patreon (only 1$ per month!)
 * http://plantuml.com/paypal
 * 
 * This file is part of PlantUML.
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). [Eclipse Public License - v 1.0]
 * 
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 * 
 * You may obtain a copy of the License at
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.project3;

import java.util.Arrays;
import java.util.Collection;

import net.sourceforge.plantuml.command.regex.IRegex;
import net.sourceforge.plantuml.command.regex.RegexConcat;
import net.sourceforge.plantuml.command.regex.RegexLeaf;
import net.sourceforge.plantuml.command.regex.RegexOr;
import net.sourceforge.plantuml.command.regex.RegexResult;

public class SubjectDaysAsDates implements SubjectPattern {

	public Collection<VerbPattern> getVerbs() {
		return Arrays.<VerbPattern> asList(new VerbIsOrAre(), new VerbIsOrAreNamed());
	}

	public IRegex toRegex() {
		return new RegexOr(regexTo(), regexAnd(), regexThen());

	}

	private IRegex regexTo() {
		return new RegexConcat( //
				new RegexLeaf("YEAR1", "([\\d]{4})"), //
				new RegexLeaf("\\D"), //
				new RegexLeaf("MONTH1", "([\\d]{1,2})"), //
				new RegexLeaf("\\D"), //
				new RegexLeaf("DAY1", "([\\d]{1,2})"), //
				new RegexLeaf("[%s]+to[%s]+"), //
				new RegexLeaf("YEAR2", "([\\d]{4})"), //
				new RegexLeaf("\\D"), //
				new RegexLeaf("MONTH2", "([\\d]{1,2})"), //
				new RegexLeaf("\\D"), //
				new RegexLeaf("DAY2", "([\\d]{1,2})") //
		);
	}

	private IRegex regexAnd() {
		return new RegexConcat( //
				new RegexLeaf("YEAR3", "([\\d]{4})"), //
				new RegexLeaf("\\D"), //
				new RegexLeaf("MONTH3", "([\\d]{1,2})"), //
				new RegexLeaf("\\D"), //
				new RegexLeaf("DAY3", "([\\d]{1,2})"), //
				new RegexLeaf("[%s]+and[%s]+"), //
				new RegexLeaf("COUNT_AND", "([\\d]+)"), //
				new RegexLeaf("[%s]+days?") //

		);
	}

	private IRegex regexThen() {
		return new RegexConcat( //
				new RegexLeaf("then[%s]+"), //
				new RegexLeaf("COUNT_THEN", "([\\d]+)"), //
				new RegexLeaf("[%s]+days?") //

		);
	}

	public Subject getSubject(GanttDiagram project, RegexResult arg) {
		final String countAnd = arg.get("COUNT_AND", 0);
		if (countAnd != null) {
			final DayAsDate date3 = getDate(arg, "3");
			final int nb = Integer.parseInt(countAnd);
			return new DaysAsDates(project, date3, nb);
		}
		final String countThen = arg.get("COUNT_THEN", 0);
		if (countThen != null) {
			final DayAsDate date3 = project.getThenDate();
			final int nb = Integer.parseInt(countThen);
			return new DaysAsDates(project, date3, nb);			
		}
		final DayAsDate date1 = getDate(arg, "1");
		final DayAsDate date2 = getDate(arg, "2");
		return new DaysAsDates(date1, date2);
	}

	private DayAsDate getDate(RegexResult arg, String suffix) {
		final int day = Integer.parseInt(arg.get("DAY" + suffix, 0));
		final int month = Integer.parseInt(arg.get("MONTH" + suffix, 0));
		final int year = Integer.parseInt(arg.get("YEAR" + suffix, 0));
		return DayAsDate.create(year, month, day);
	}

}
