/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2020, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * If you like this project or if you find it useful, you can support us at:
 * 
 * http://plantuml.com/patreon (only 1$ per month!)
 * http://plantuml.com/paypal
 * 
 * This file is part of PlantUML.
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). [Eclipse Public License - v 1.0]
 * 
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 * 
 * You may obtain a copy of the License at
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.command;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import net.sourceforge.plantuml.StringUtils;
import net.sourceforge.plantuml.WithSprite;
import net.sourceforge.plantuml.command.note.SingleMultiFactoryCommand;
import net.sourceforge.plantuml.command.regex.RegexConcat;
import net.sourceforge.plantuml.command.regex.RegexLeaf;
import net.sourceforge.plantuml.command.regex.RegexResult;
import net.sourceforge.plantuml.ugraphic.sprite.Sprite;
import net.sourceforge.plantuml.ugraphic.sprite.SpriteColorBuilder4096;
import net.sourceforge.plantuml.ugraphic.sprite.SpriteGrayLevel;

public final class FactorySpriteCommand implements SingleMultiFactoryCommand<WithSprite> {

	private RegexConcat getRegexConcatMultiLine() {
		return new RegexConcat(new RegexLeaf("^"), //
				new RegexLeaf("sprite[%s]+\\$?"), //
				new RegexLeaf("NAME", "([\\p{L}0-9_]+)[%s]*"), //
				new RegexLeaf("DIM", "(?:\\[(\\d+)x(\\d+)/(?:(\\d+)(z)?|(color))\\])?"), //
				new RegexLeaf("[%s]*\\{"), //
				new RegexLeaf("$"));
	}

	private RegexConcat getRegexConcatSingleLine() {
		return new RegexConcat(new RegexLeaf("^"), //
				new RegexLeaf("sprite[%s]+\\$?"), //
				new RegexLeaf("NAME", "([\\p{L}0-9_]+)[%s]*"), //
				new RegexLeaf("DIM", "(?:\\[(\\d+)x(\\d+)/(?:(\\d+)(z)|(color))\\])?"), //
				new RegexLeaf("[%s]+"), //
				new RegexLeaf("DATA", "([-_A-Za-z0-9]+)"), //
				new RegexLeaf("$"));
	}

	public Command<WithSprite> createSingleLine() {
		return new SingleLineCommand2<WithSprite>(getRegexConcatSingleLine()) {

			@Override
			protected CommandExecutionResult executeArg(final WithSprite system, RegexResult arg) {
				return executeInternal(system, arg, Arrays.asList((CharSequence) arg.get("DATA", 0)));
			}

		};
	}

	public Command<WithSprite> createMultiLine(boolean withBracket) {
		return new CommandMultilines2<WithSprite>(getRegexConcatMultiLine(), MultilinesStrategy.REMOVE_STARTING_QUOTE) {

			@Override
			public String getPatternEnd() {
				return "(?i)^end[%s]?sprite|\\}$";
			}

			protected CommandExecutionResult executeNow(final WithSprite system, BlocLines lines) {
				lines = lines.trim(true);
				final RegexResult line0 = getStartingPattern().matcher(StringUtils.trin(lines.getFirst499()));

				lines = lines.subExtract(1, 1);
				lines = lines.removeEmptyColumns();
				if (lines.size() == 0) {
					return CommandExecutionResult.error("No sprite defined.");
				}
				return executeInternal(system, line0, lines.getLines());
			}

		};
	}

	private CommandExecutionResult executeInternal(WithSprite system, RegexResult line0,
			final List<CharSequence> strings) {
		try {
			final Sprite sprite;
			if (line0.get("DIM", 0) == null) {
				sprite = SpriteGrayLevel.GRAY_16.buildSprite(-1, -1, strings);
			} else {
				final int width = Integer.parseInt(line0.get("DIM", 0));
				final int height = Integer.parseInt(line0.get("DIM", 1));
				if (line0.get("DIM", 4) == null) {
					final int nbLevel = Integer.parseInt(line0.get("DIM", 2));
					if (nbLevel != 4 && nbLevel != 8 && nbLevel != 16) {
						return CommandExecutionResult.error("Only 4, 8 or 16 graylevel are allowed.");
					}
					final SpriteGrayLevel level = SpriteGrayLevel.get(nbLevel);
					if (line0.get("DIM", 3) == null) {
						sprite = level.buildSprite(width, height, strings);
					} else {
						sprite = level.buildSpriteZ(width, height, concat(strings));
					}
				} else {
					sprite = SpriteColorBuilder4096.buildSprite(strings);
				}
			}
			system.addSprite(line0.get("NAME", 0), sprite);
			return CommandExecutionResult.ok();
		} catch (IOException e) {
			return CommandExecutionResult.error("Cannot decode sprite.");
		}
	}

	private String concat(final List<? extends CharSequence> strings) {
		final StringBuilder sb = new StringBuilder();
		for (CharSequence s : strings) {
			sb.append(StringUtils.trin(s));
		}
		return sb.toString();
	}

}
