/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2020, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * If you like this project or if you find it useful, you can support us at:
 * 
 * http://plantuml.com/patreon (only 1$ per month!)
 * http://plantuml.com/paypal
 * 
 * This file is part of PlantUML.
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). [Eclipse Public License - v 1.0]
 * 
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 * 
 * You may obtain a copy of the License at
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml;

public class CharSequence2Impl implements CharSequence2 {

	private final CharSequence s;
	private final LineLocation location;
	private String preprocessorError;

	public CharSequence2Impl(CharSequence s, LineLocation location) {
		this(s, location, null);
	}

	public CharSequence2Impl(CharSequence s, LineLocation location, String preprocessorError) {
		if (s == null) {
			throw new IllegalArgumentException();
		}
		this.s = s;
		this.location = location;
		this.preprocessorError = preprocessorError;
	}

	// public static CharSequence2 errorPreprocessor(CharSequence s, String preprocessorError) {
	// return new CharSequence2Impl("FOO4242", null, preprocessorError);
	// }

	public CharSequence2 withErrorPreprocessor(String preprocessorError) {
		return new CharSequence2Impl(s, location, preprocessorError);
	}

	public int length() {
		return s.length();
	}

	public char charAt(int index) {
		return s.charAt(index);
	}

	public CharSequence2 subSequence(int start, int end) {
		return new CharSequence2Impl(s.subSequence(start, end), location, preprocessorError);
	}

	public CharSequence toCharSequence() {
		return s;
	}

	@Override
	public String toString() {
		return s.toString();
	}

	public String toString2() {
		return s.toString();
	}

	public LineLocation getLocation() {
		return location;
	}

	public CharSequence2 trin() {
		return new CharSequence2Impl(StringUtils.trin(s.toString()), location, preprocessorError);
	}

	public boolean startsWith(String start) {
		return s.toString().startsWith(start);
	}

	public String getPreprocessorError() {
		return preprocessorError;
	}

	public CharSequence2 removeInnerComment() {
		final String string = s.toString();
		final String trim = string.replace('\t', ' ').trim();
		if (trim.startsWith("/'")) {
			final int idx = string.indexOf("'/");
			if (idx != -1) {
				return new CharSequence2Impl(removeSpecialInnerComment(s.subSequence(idx + 2, s.length())), location,
						preprocessorError);
			}
		}
		if (trim.endsWith("'/")) {
			final int idx = string.lastIndexOf("/'");
			if (idx != -1) {
				return new CharSequence2Impl(removeSpecialInnerComment(s.subSequence(0, idx)), location,
						preprocessorError);
			}
		}
		if (trim.contains("/'''") && trim.contains("'''/")) {
			return new CharSequence2Impl(removeSpecialInnerComment(s), location, preprocessorError);
		}
		return this;
	}

	private CharSequence removeSpecialInnerComment(CharSequence cs) {
		final String s = cs.toString();
		if (s.contains("/'''") && s.contains("'''/")) {
			return s.replaceAll("/'''[-\\w]*'''/", "");

		}
		return cs;
	}
}
