/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 *
 * $Id: RDFSReasoner.java,v 1.1 2007/03/18 08:39:02 lzhang Exp $
 */
package org.eclipse.eodm.rdf.reasoner;

import java.util.List;

import org.eclipse.eodm.rdf.rdfbase.RDFGraph;
import org.eclipse.eodm.rdf.rdfbase.RDFProperty;
import org.eclipse.eodm.rdf.rdfs.RDFSClass;

/**
 * 
 * RDFSReasoner : The basic interface of RDF Reasoner, It provide interfaces to
 * <ol>
 * <li>Initialize a reasoner from given rdf model</li>
 * <li>Validate a given ontology model</li>
 * <li>Get all entailed sub/super classes of a RDFSClass</li>
 * <li>Get all entailed sub/super properties of a RDFProperty</li>
 * <li>Get all entailed instances that are rdf:type of a RDFSClass</li>
 * <li>Get all entailed triples of a RDFProperty</li>
 * <li>Test whether two RDFSClass have subClassOf/superClassOf relationship
 * </li>
 * <li>Test whether two RDFProperty have subPropertyOf/superPropertyOf
 * relationship</li>
 * </ol>
 */

public interface RDFSReasoner {

    /**
     * Initialize Reasoner with a given rdf/rdfs ontology
     * 
     * @param ontology
     *            the given RDFS ontology
     * @throws InconsistentOntologyException
     *             if there exist some facts that will cause contradition in the
     *             context of RDF/RDFS Semantic
     */
    public void initialize(RDFGraph graph) throws RDFSReasonerException;
            
    /**
     * Get all entailed sub classes of a RDFSClass
     * 
     * @param c
     *            the RDFSClass to be queried
     * @return all entailed sub classes of the given RDFSClass
     */
    List getDescendantClasses(RDFSClass c);

    /**
     * Get all entailed super classes of a RDFSClass
     * 
     * @param c
     *            the RDFSClass to be queried
     * @return all entailed super classes of the given RDFSClass c
     */
    List getAncestorClasses(RDFSClass c);

    /**
     * Get all entailed sub properties of a RDFProperty
     * 
     * @param property
     *            the RDFProperty to be queried
     * @return all entailed sub properties of the given property
     */
    List getDescendantProperties(RDFProperty property);

    /**
     * Get all entailed super properties of a RDFProperty
     * 
     * @param property
     *            the RDFProperty to be queried
     * @return all entailed sub properties of the given property
     */
    List getAncestorProperties(RDFProperty property);

    /**
     * Test whether two RDFSClass have subClassOf relationship
     * 
     * @param c1
     *            RDFSClass
     * @param c2
     *            RDFSClass
     * @return true if c1 is subclass of c2
     */
    boolean isDescendantClassOf(RDFSClass c1, RDFSClass c2);

    /**
     * Test whether two RDFSClass have superClassOf relationship
     * 
     * @param c1
     *            RDFSClass
     * @param c2
     *            RDFSClass
     * @return true if c1 is super class of c2
     */
    boolean isAncestorClassOf(RDFSClass c1, RDFSClass c2);

    /**
     * Test whether two RDFProperty have subPropertyOf relationship
     * 
     * @param p1
     *            RDFProperty
     * @param p2
     *            RDFProperty
     * @return true if p1 is subPropertyOf of p2
     */
    boolean isDescendantPropertyOf(RDFProperty p1, RDFProperty p2);

    /**
     * Test whether two RDFProperty have superPropertyOf relationship
     * 
     * @param p1
     *            RDFProperty
     * @param p2
     *            RDFProperty
     * @return true if p1 is superPropertyOf of p2
     */
    boolean isAncestorPropertyOf(RDFProperty p1, RDFProperty p2);

    /**
     * Get all entailed instances of a RDFSClass
     * 
     * @param c
     *            the RDFSClass to be queried
     * @return all entailed individuals that are rdf:type of the given class c
     */
    List getInstances(RDFSClass c);

    /**
     * Get all entailed triples whose predicate is the given property
     * 
     * @param property
     *            the RDFProperty to be queried
     * @return all entailed triples whose predicate is the given property
     */
    List getTriples(RDFProperty property);

}
