/**
 * Copyright (c) 2014, 2015, 2017, 2020 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Benoit Maggi       - Bug 474408: order by identifier the generated file
 * Ansgar Radermacher - Bug 526155: set element type name from profile
 * Camille Letavernier - Bug 569354: remove StereotypeAdvice; use StereotypeMatcherAdvice instead
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeSetConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypesConfigurationsFactory;
import org.eclipse.papyrus.infra.types.SpecializationTypeConfiguration;
import org.eclipse.papyrus.uml.profile.types.generator.DeltaStrategy;
import org.eclipse.papyrus.uml.profile.types.generator.strategy.ElementTypeConfigHelper;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * <p>
 * Transformation rule for generating an {@link ElementTypeSetConfiguration} from a UML {@link Profile}.
 * </p>
 * <p>
 * Supports incremental (re)generation, if an Optional {@link ElementTypeSetConfiguration} and {@link DeltaStrategy.Diff}
 * are provided.
 * </p>
 */
@Singleton
@SuppressWarnings("all")
public class ConfigurationSetRule {
  @Extension
  private static ElementTypesConfigurationsFactory elementtypesconfigurationsFactory = ElementTypesConfigurationsFactory.eINSTANCE;

  @Inject
  @Extension
  private Identifiers _identifiers;

  @Inject
  @Extension
  private UML _uML;

  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;

  @Inject
  @Extension
  private ElementTypeRule _elementTypeRule;

  @Inject
  @Extension
  private ElementTypeConfigHelper _elementTypeConfigHelper;

  @Inject
  private Optional<DeltaStrategy.Diff> diff;

  private static List<ElementTypeConfiguration> elementTypeConfigurationList;

  public static ElementTypeConfiguration addElementType(final ElementTypeConfiguration elementtype) {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration el) -> {
      return Boolean.valueOf(el.getIdentifier().equals(elementtype.getIdentifier()));
    };
    ElementTypeConfiguration found = IterableExtensions.<ElementTypeConfiguration>findFirst(ConfigurationSetRule.elementTypeConfigurationList, _function);
    boolean _equals = Objects.equal(found, null);
    if (_equals) {
      ConfigurationSetRule.elementTypeConfigurationList.add(elementtype);
      return elementtype;
    } else {
      return found;
    }
  }

  /**
   * <p>
   * Create or (incrementally) regenerate an ElementTypeSetConfiguration, and populate
   * it with ElementTypeConfigurations, from the given umlProfile.
   * </p>
   * 
   * @param umlProfile
   * 		The source profile used to generate the ElementTypeSetConfiguration
   * @param originalOutput
   * 		The contents EList from the existing ElementTypeSetConfiguration, or null if we are generating a new Config.
   */
  public ElementTypeSetConfiguration toConfigurationSet(final Profile umlProfile, final EList<? super EObject> originalOutput) {
    if (((!Objects.equal(originalOutput, null)) && (!originalOutput.isEmpty()))) {
      boolean _isEmpty = this.diff.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final Function1<Object, Boolean> _function = (Object it) -> {
          return Boolean.valueOf((it instanceof ElementTypeSetConfiguration));
        };
        Object _findFirst = IterableExtensions.findFirst(originalOutput, _function);
        final ElementTypeSetConfiguration typeSet = ((ElementTypeSetConfiguration) _findFirst);
        this.updateElementTypeSet(umlProfile, typeSet, this.diff.get());
        return typeSet;
      }
      return null;
    } else {
      return this.newConfigurationSet(umlProfile);
    }
  }

  /**
   * <p>
   * Create a new ElementTypeSetConfiguration and populate it with generated ElementTypeConfigurations,
   * from the given umlProfile.
   * </p>
   */
  public ElementTypeSetConfiguration newConfigurationSet(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final ElementTypeSetConfiguration _result;
    synchronized (_createCache_newConfigurationSet) {
      if (_createCache_newConfigurationSet.containsKey(_cacheKey)) {
        return _createCache_newConfigurationSet.get(_cacheKey);
      }
      ElementTypeSetConfiguration _createElementTypeSetConfiguration = ConfigurationSetRule.elementtypesconfigurationsFactory.createElementTypeSetConfiguration();
      _result = _createElementTypeSetConfiguration;
      _createCache_newConfigurationSet.put(_cacheKey, _result);
    }
    _init_newConfigurationSet(_result, umlProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, ElementTypeSetConfiguration> _createCache_newConfigurationSet = CollectionLiterals.newHashMap();

  private void _init_newConfigurationSet(final ElementTypeSetConfiguration it, final Profile umlProfile) {
    final DeltaStrategy.DiffImpl newDiff = new DeltaStrategy.DiffImpl();
    Iterables.<Stereotype>addAll(newDiff.addedStereotypes, this._uML.getAllStereotypes(umlProfile));
    this._identifiers.setIdentifierBase(umlProfile);
    it.setIdentifier(this._identifiers.getQualified("elementTypes"));
    this.updateElementTypeSet(umlProfile, it, newDiff);
  }

  public ElementTypeSetConfiguration updateElementTypeSet(final Profile umlProfile, final ElementTypeSetConfiguration typeSet, final DeltaStrategy.Diff diff) {
    ElementTypeSetConfiguration _xblockexpression = null;
    {
      final ElementTypeConfigHelper helper = new ElementTypeConfigHelper();
      ConfigurationSetRule.elementTypeConfigurationList = CollectionLiterals.<ElementTypeConfiguration>newArrayList();
      this._identifiers.setIdentifierBase(umlProfile);
      String _elvis = null;
      ElementTypeSetConfiguration _baseUMLElementTypeSet = this._uMLElementTypes.getBaseUMLElementTypeSet();
      String _metamodelNsURI = null;
      if (_baseUMLElementTypeSet!=null) {
        _metamodelNsURI=_baseUMLElementTypeSet.getMetamodelNsURI();
      }
      if (_metamodelNsURI != null) {
        _elvis = _metamodelNsURI;
      } else {
        _elvis = UMLPackage.eNS_URI;
      }
      typeSet.setMetamodelNsURI(_elvis);
      List<Stereotype> _addedStereotypes = diff.getAddedStereotypes();
      final List<Stereotype> addedStereotypes = ((List<Stereotype>) _addedStereotypes);
      for (final Stereotype addedStereotype : addedStereotypes) {
        Iterable<ImpliedExtension> _impliedExtensions = this._uML.impliedExtensions(addedStereotype);
        for (final ImpliedExtension ext : _impliedExtensions) {
          Iterable<? extends ElementTypeConfiguration> _diagramSpecificElementTypes = this._uMLElementTypes.getDiagramSpecificElementTypes(ext.getMetaclass());
          for (final ElementTypeConfiguration element : _diagramSpecificElementTypes) {
            {
              final SpecializationTypeConfiguration elementtype = this._elementTypeRule.toElementType(ext, element);
              ConfigurationSetRule.elementTypeConfigurationList.add(elementtype);
            }
          }
        }
      }
      List<String> _removedStereotypes = diff.getRemovedStereotypes();
      final List<String> removedStereotypes = ((List<String>) _removedStereotypes);
      final Consumer<String> _function = (String it) -> {
        if ((Objects.equal(it, null) || it.isEmpty())) {
          return;
        }
        EList<ElementTypeConfiguration> _elementTypeConfigurations = typeSet.getElementTypeConfigurations();
        final Consumer<ElementTypeConfiguration> _function_1 = (ElementTypeConfiguration typeConfig) -> {
          String _stereotypeName = this._elementTypeConfigHelper.getStereotypeName(typeConfig);
          boolean _equals = Objects.equal(_stereotypeName, it);
          if (_equals) {
            typeSet.getElementTypeConfigurations().remove(typeConfig);
          }
        };
        new ArrayList<ElementTypeConfiguration>(_elementTypeConfigurations).forEach(_function_1);
      };
      removedStereotypes.forEach(_function);
      Map<String, Stereotype> _renamedStereotypes = diff.getRenamedStereotypes();
      final Map<String, Stereotype> renamedStereotypes = ((Map<String, Stereotype>) _renamedStereotypes);
      Set<Map.Entry<String, Stereotype>> _entrySet = renamedStereotypes.entrySet();
      for (final Map.Entry<String, Stereotype> entry : _entrySet) {
        {
          final String oldName = entry.getKey();
          final Stereotype stereotype = entry.getValue();
          final Function1<ElementTypeConfiguration, Boolean> _function_1 = (ElementTypeConfiguration type) -> {
            String _stereotypeName = helper.getStereotypeName(type);
            return Boolean.valueOf(Objects.equal(_stereotypeName, oldName));
          };
          final Consumer<ElementTypeConfiguration> _function_2 = (ElementTypeConfiguration type) -> {
            this._elementTypeRule.setStereotypeName(type, stereotype);
          };
          IterableExtensions.<ElementTypeConfiguration>filter(typeSet.getElementTypeConfigurations(), _function_1).forEach(_function_2);
        }
      }
      List<ImpliedExtension> _addedExtensions = diff.getAddedExtensions();
      final List<ImpliedExtension> addedExtensions = ((List<ImpliedExtension>) _addedExtensions);
      for (final ImpliedExtension ext_1 : addedExtensions) {
        Iterable<? extends ElementTypeConfiguration> _diagramSpecificElementTypes_1 = this._uMLElementTypes.getDiagramSpecificElementTypes(ext_1.getMetaclass());
        for (final ElementTypeConfiguration element_1 : _diagramSpecificElementTypes_1) {
          {
            final SpecializationTypeConfiguration elementtype = this._elementTypeRule.toElementType(ext_1, element_1);
            ConfigurationSetRule.elementTypeConfigurationList.add(elementtype);
          }
        }
      }
      final Predicate<ElementTypeConfiguration> _function_1 = (ElementTypeConfiguration type) -> {
        return diff.getRemovedExtensions().contains(this._elementTypeConfigHelper.getExtension(type, umlProfile));
      };
      typeSet.getElementTypeConfigurations().removeIf(_function_1);
      final Procedure1<ElementTypeSetConfiguration> _function_2 = (ElementTypeSetConfiguration it) -> {
        boolean _useDiPostfix = this._identifiers.useDiPostfix();
        if (_useDiPostfix) {
          String _name = umlProfile.getName();
          String _plus = (_name + " DI");
          it.setName(_plus);
        } else {
          it.setName(umlProfile.getName());
        }
        final Function1<ElementTypeConfiguration, String> _function_3 = (ElementTypeConfiguration it_1) -> {
          return it_1.getIdentifier();
        };
        it.getElementTypeConfigurations().addAll(IterableExtensions.<ElementTypeConfiguration, String>sortBy(ConfigurationSetRule.elementTypeConfigurationList, _function_3));
      };
      _xblockexpression = ObjectExtensions.<ElementTypeSetConfiguration>operator_doubleArrow(typeSet, _function_2);
    }
    return _xblockexpression;
  }
}
